#!/usr/local/bin/perl



=pod

=head1 FontMonkey

version 1.0
by P.D. Magnus
pmagnus@fecundity.com
original script 4/14/99
this version 8/20/99
Copyright (c) 1999. All rights reserved.
This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

FontMonkey is designed for use in cgi's that generate output in graphical fonts.
It uses the GD module, a port of Thomas Boutell's gd graphics library.
For more about GD, see http://www.genome.wi.mit.edu/pub/software/WWW/GD.html

=head1 FontMonkey::Image

A sub-class of GD::Image which introduces two new methods.

=cut



package FontMonkey::Image;
$VERSION = 1.0;
@ISA = qw(GD::Image);

use GD;
use strict;

my $path = 'FontMonkey';

my $pi = 3.14159;
srand;



# returns the least whole number greater than or equal to n
sub upint ($) {
	return (($_[0] == int($_[0])) ? $_[0] : int($_[0] + 1));
}



=pod

=head1 FontMonkey::Image::render

I<object method>
Prints the appropriate header and outputs the image as a gif.
Makes for all the required output from a cgi.

=cut

sub render {

	my $self = shift;

	# print the image to stdout
	print "Content-type: image/gif\n\n";
	binmode STDOUT;
	print $self->gif;
	
	return 1;
}



=pod

=head1 FontMonkey::Image::newmsg(message, layout, font, transparent)

I<class method>
Alternate constructor returns an Image containing the formatted message.
Formatting is controlled by four inputs.

=head2 message

I<string>
The message can contain any combination of letters, numbers, underscores, and dashes.
No distinction is made between upper and lower case.
For most fonts, underscores are rendered as bullets.
For most layouts, dashes are left as blank spaces.

=head2 layout

I<string>
The following layouts are supported:

=over 4

=item horiz

Outputs the characters of the message left to right on one row.
The default layout.

=item vert

Outputs the characters of the message top to bottom in one column.

=item circle

Outputs the characters of the message along a circle, clockwise from 3 o'clock.
If you want to start at another point, you can reformat the message.
eg: the message 'hnigeb---ere' with the layout circle returns 'begin here' counterclockwise from 9 to 12 o'clock.

=item spiral

Outputs the characters of the message in a spiral, spiralling clockwise out from 9 o'clock.
If you want the characters to spiral in, you can reformat the message.
A dash won't make for blank spaces in spiral.

=item spirlcc

Like spiral, but counterclockwise out from 3 o'clock.

=item sine

Outputs the characters of the message left to right along a sine wave.

=item frame

Outputs the characters of the message along the edge of a square, beginning in the upper left and moving clockwise.
If the length of the message isn't mod 4, additional blank space is left at the end.

=item box

Outputs the characters of the message in rows so as to make a square, left to right in each row.
If the length of the message isn't a perfect square, additional blank space is left at the end.

=back

=head2 font

I<string>
Fonts are a collection of gif's, stored together in a directory, eg: the letter 'a' for the font 'basic' is basicfont/a.gif.
Each gif is the same size, so the fonts are monospaced.
All of the font directories are assumed to be at the location indicated in $path at the top of the module. You should change this as necessary.
Not every font has every character, but if you request a character that isn't part of the font it'll treat it as an underscore.
Ther are five standard fonts:

=over 4

=item basic

Hand-written characters.
The default font.
Underscore is a dot.

=item digital

Characters that could be produced on 8 LED digital displays.
There are no half lines, angled lines, or anything like that, so some of the letters are a bit contrived.
Underscore is a colon.

=item typewriter

Uppercase characters meant to look like type from an old typewriter.
Underscore is an asterisk.

=item counter

Odometer-style numbers as often used in hit counters.
Underscore is a blank.
There are no alpha characters in this font.

=item semaphore

The standard semaphore flag signalling system.
Note that there are no separate number signs. To signal numbers, there is a numerical sign.
After the numerical sign, A-I represent the digits one through nine and K respresents zero.
To switch back to alpha characters, J is alphabetic sign.
In this font, '1' is the numerical sign and '0' is the error signal.
Underscore is the annul sign.

=item creating new fonts

Although it might take a little time, creating new fonts isn't rocket science.
At present there is no automated way to handle new fonts, but you can add your new font to the FONTASSIGN block manually.
You only need to make characters that you think you'll need, but make sure there's a _.gif since that's the default.

=back

=head2 transparent

I<flag>
If true, newmsg will make the background of the Image transparent.
Note that it assumes the pixel in the upperleft is the background colour, so that if you want some other colour transparent you should set the flag false and do it yourself.

=cut

sub newmsg {

my $class = shift;
$class = ref($class) || $class;

my $msg = lc(shift);
my $layout = lc(shift);
my $fontdir = lc(shift).'font';
my $trfl = shift;

my $imgdat = {};
my $im = undef;

# assign character size based on font

FONTASSIGN: for ($fontdir) {
	/basic/ ? do {
	$imgdat->{WIDTH} = 15;
	$imgdat->{HEIGHT} = 20;
	} :
	/digital/ ? do {
	$imgdat->{WIDTH} = 16;
	$imgdat->{HEIGHT} = 21;
	} :
	/typewriter/ ? do {
	$imgdat->{WIDTH} = 14;
	$imgdat->{HEIGHT} = 15;
	} :
	/counter/ ? do {
	$imgdat->{WIDTH} = 15;
	$imgdat->{HEIGHT} = 20;
	} :
	/semaphore/ ? do {
	# _ is the repeat code
	# 0 is the error code
	# 1 is the numeric switch (the alpha switch is j)
	$imgdat->{WIDTH} = 35;
	$imgdat->{HEIGHT} = 32;
	} : do {
	# changes to default value
	# and comes around again
	$fontdir = 'basicfont';
	redo FONTASSIGN;
	};

}

# start the gif building
# working variables are assigned as needed
# image size depends on layout

SWITCH: for ($layout) {
	/horiz/ ? do {
	# outputs the message in a straight horizontal line
	$im = new GD::Image($imgdat->{WIDTH} * length($msg), $imgdat->{HEIGHT});
	}:
	/vert/ ? do {
	# outputs the message in a straight vertical line
	$im = new GD::Image($imgdat->{WIDTH}, $imgdat->{HEIGHT} * length($msg));
	} :
	/circle/ ? do {
	# outputs the message in a circle, clockwise from 3 o'clock
	$imgdat->{EDGE} = ( $imgdat->{WIDTH} > $imgdat->{HEIGHT} ) ? $imgdat->{WIDTH} : $imgdat->{HEIGHT};
	$imgdat->{RADIUS} = $imgdat->{EDGE} * length($msg) / 4;
	$imgdat->{CHANGLE} = 2 * 3.14159 / length($msg);
	$im = new GD::Image(2 * $imgdat->{RADIUS} + $imgdat->{WIDTH}, 2 * $imgdat->{RADIUS} + $imgdat->{HEIGHT});
	} :
	/spiral/ ? do{
	# outputs the message in a spiral, clockwise out from 9 o'clock
	$imgdat->{EDGE} = ( $imgdat->{WIDTH} > $imgdat->{HEIGHT} ) ? $imgdat->{WIDTH} : $imgdat->{HEIGHT};
	$imgdat->{RADIUS} = $imgdat->{EDGE};
	$imgdat->{CENTER} = (length($msg) < 11) ? ($imgdat->{EDGE} * 2) : (length($msg) < 27) ? ($imgdat->{EDGE} * 3) : ($imgdat->{EDGE} * 4);
	$imgdat->{ANGLE} = $pi;
	$im = new GD::Image(2 * $imgdat->{CENTER} + $imgdat->{WIDTH}, 2 * $imgdat->{CENTER} + $imgdat->{HEIGHT});
	} :
	/spirlcc/ ? do{
	# outputs the message in a spiral, counter-clockwise out from 3 o'clock
	$imgdat->{EDGE} = ( $imgdat->{WIDTH} > $imgdat->{HEIGHT} ) ? $imgdat->{WIDTH} : $imgdat->{HEIGHT};
	$imgdat->{RADIUS} = $imgdat->{EDGE};
	$imgdat->{CENTER} = (length($msg) < 11) ? ($imgdat->{EDGE} * 2) : (length($msg) < 27) ? ($imgdat->{EDGE} * 3) : ($imgdat->{EDGE} * 4);
	$imgdat->{ANGLE} = 0;
	$im = new GD::Image(2 * $imgdat->{CENTER} + $imgdat->{WIDTH}, 2 * $imgdat->{CENTER} + $imgdat->{HEIGHT});
	} :
	/sine/ ? do{
	# outputs the message along a sign wave
	$imgdat->{ANGINC} = .5;
	$imgdat->{PHASE} = rand ($pi);
	$im = new GD::Image($imgdat->{WIDTH} * length($msg), $imgdat->{HEIGHT} * 5);
	} :
	/frame/ ? do{
	# outputs the message along the edge of a square, starting the in the upper left
	$imgdat->{EACHLINE} = int((length($msg) - 1) / 4) + 1;
	$im = new GD::Image(($imgdat->{EACHLINE} + 1) * $imgdat->{WIDTH}, ($imgdat->{EACHLINE} + 1) * $imgdat->{HEIGHT});
	} :
	/box/ ? do{
	# outputs the message in a square, line by line
	$imgdat->{EACHLINE} = upint(sqrt(length($msg)));
	$im = new GD::Image($imgdat->{WIDTH} * $imgdat->{EACHLINE}, $imgdat->{HEIGHT} * upint(length($msg) / $imgdat->{EACHLINE}));
	} : do{
	# changes to default value
	# and comes around again
	$layout = "horiz";
	redo SWITCH;
	};
}


# pre-load all the characters that will be needed
my $font = {};
my $ch = undef;
my $i = undef;

for ($i=0; ($i < length($msg)); $i++) {
	$ch = substr $msg, $i, 1;
	
	if (!(exists $font->{$ch}) && ($ch ne '-')) {
	    open ('GIF',"$path/$fontdir/$ch.gif") || open (GIF,"$path/$fontdir/_.gif") || die;
	    $font->{$ch} = newFromGif GD::Image('GIF') || die;
	    close 'GIF';
	}
}


# cycle through the letters of the message

MAINLOOP: for ($i=0; ($i < length($msg)); $i++) {
	
	$ch = substr $msg, $i, 1;
	
	# any non word character gets skipped over--
	# leaving a blank space	for most layouts,
	# notable exception being the spirals
	if ($ch =~ /\W/) {next MAINLOOP;}
	    
    # location of each character depends on layout
	SWITCH: for ($layout) {
		/horiz/ ? do {
		$im->copy($font->{$ch}, ($imgdat->{WIDTH} * $i), 0, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		}:
		/vert/ ? do {
		$im->copy($font->{$ch}, 0, ($imgdat->{HEIGHT} * $i), 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		}:
		/circle/ ? do {
		$im->copy($font->{$ch}, $imgdat->{RADIUS} * cos($imgdat->{CHANGLE} * $i) + $imgdat->{RADIUS}, $imgdat->{RADIUS} * sin($imgdat->{CHANGLE} * $i) + $imgdat->{RADIUS}, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		}:
		/spiral/ ? do {
		$im->copy($font->{$ch}, $imgdat->{RADIUS} * cos($imgdat->{ANGLE}) + $imgdat->{CENTER}, $imgdat->{RADIUS} * sin($imgdat->{ANGLE}) + $imgdat->{CENTER}, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		$imgdat->{ANGLE} = $imgdat->{ANGLE} + $imgdat->{EDGE} / $imgdat->{RADIUS};
		#if ($imgdat->{ANGLE} > $pi) { $imgdat->{ANGLE} = $imgdat->{ANGLE} - $pi }
		# increase radius so that the spiral will grow (edge) larger each full rotation
		$imgdat->{RADIUS} = $imgdat->{RADIUS} + ($imgdat->{EDGE}*$imgdat->{EDGE}) / (2 * $pi * $imgdat->{RADIUS});
		}:
		/spirlcc/ ? do {
		$im->copy($font->{$ch}, $imgdat->{RADIUS} * cos($imgdat->{ANGLE}) + $imgdat->{CENTER}, $imgdat->{RADIUS} * sin($imgdat->{ANGLE}) + $imgdat->{CENTER}, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		$imgdat->{ANGLE} = $imgdat->{ANGLE} - $imgdat->{EDGE} / $imgdat->{RADIUS};
		#if ($imgdat->{ANGLE} > $pi) { $imgdat->{ANGLE} = $imgdat->{ANGLE} - $pi }
		# increase radius so that the spiral will grow (edge) larger each full rotation
		$imgdat->{RADIUS} = $imgdat->{RADIUS} + ($imgdat->{EDGE}*$imgdat->{EDGE}) / (2 * $pi * $imgdat->{RADIUS});
		}:
		/sine/ ? do {
		$im->copy($font->{$ch}, ($imgdat->{WIDTH} * $i), (1 + sin($imgdat->{ANGINC} * $i + $imgdat->{PHASE})) * $imgdat->{HEIGHT} * 2, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		}:
		/frame/ ? do {
		for (int($i / $imgdat->{EACHLINE})) {
		  ($_ == 0) ? do {$im->copy($font->{$ch}, ($imgdat->{WIDTH} * $i), 0, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});} :
		  ($_ == 1) ? do {$im->copy($font->{$ch}, ($imgdat->{EACHLINE} * $imgdat->{WIDTH}), (($i - $imgdat->{EACHLINE}) * $imgdat->{HEIGHT}), 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});} :
		  ($_ == 2) ? do {$im->copy($font->{$ch}, ((3 * $imgdat->{EACHLINE} - $i) * $imgdat->{WIDTH}), ($imgdat->{EACHLINE} * $imgdat->{HEIGHT}), 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});} :
	    	          do {$im->copy($font->{$ch}, 0, ((4 * $imgdat->{EACHLINE} - $i) * $imgdat->{HEIGHT}), 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});};
	    }
		}:
		/box/ ? do {
		$imgdat->{DIV} = int($i / $imgdat->{EACHLINE});
		$im->copy($font->{$ch}, $imgdat->{WIDTH} * ($i - $imgdat->{DIV}*$imgdat->{EACHLINE}), $imgdat->{HEIGHT} * $imgdat->{DIV}, 0, 0, $imgdat->{WIDTH}, $imgdat->{HEIGHT});
		}: exit;
	}
	
}

# act on transparency
if ($trfl) {
	$im->transparent($im->getPixel(0,0));
}

bless ($im, $class);
return $im;

}

1;
