## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----load-data, echo=FALSE----------------------------------------------------
load("data_futures.RData")

## ----setup-load, message=FALSE, warning=FALSE, echo=TRUE----------------------
library(rb3) # B3 data access
library(ggplot2) # data visualization
library(stringr) # string manipulation
library(dplyr) # data transformation
library(bizdays) # business days calculations

## ----fetch-marketdata, eval=FALSE---------------------------------------------
# dates <- getdate("first mon", seq(as.Date("2021-01-01"), as.Date("2022-12-24"), by = 7), "actual") |>
#   following("Brazil/B3")
# fetch_marketdata("b3-futures-settlement-prices", refdate = dates)

## ----query-data, eval=FALSE---------------------------------------------------
# # Retrieve the futures settlement prices data
# df <- futures_get() |>
#   collect()

## ----display-data-------------------------------------------------------------
# Display the first few rows of the dataset
head(df)

## ----create-di1-futures-------------------------------------------------------
di1_futures <- df |>
  filter(commodity == "DI1") |>
  mutate(
    maturity_date = maturitycode2date(maturity_code),
    fixing = following(maturity_date, "Brazil/ANBIMA"),
    business_days = bizdays(refdate, maturity_date, "Brazil/ANBIMA"),
    adjusted_tax = (100000 / price)^(252 / business_days) - 1
  ) |>
  filter(business_days > 0)

## ----di1-futures-plot, fig.width=7, fig.height=5, fig.cap="DI1 Future Rates - Nominal Interest Rates"----
di1_futures |>
  filter(symbol %in% c("DI1F23", "DI1F33")) |>
  ggplot(aes(x = refdate, y = adjusted_tax, color = symbol, group = symbol)) +
  geom_line() +
  labs(
    title = "DI1 Future Rates - Nominal Interest Rates",
    caption = "Source B3 - package rb3",
    x = "Date",
    y = "Interest Rates",
    color = "Symbol"
  ) +
  scale_y_continuous(labels = scales::percent)

## ----create-dap-futures-------------------------------------------------------
dap_futures <- df |>
  filter(commodity == "DAP") |>
  mutate(
    maturity_date = maturitycode2date(maturity_code, "15th day"),
    fixing = following(maturity_date, "Brazil/ANBIMA"),
    business_days = bizdays(refdate, maturity_date, "Brazil/ANBIMA"),
    adjusted_tax = (100000 / price)^(252 / business_days) - 1
  ) |>
  filter(business_days > 0)

## ----dap-futures-plot, fig.width=7, fig.height=5, fig.cap="DAP Future Rates - Real Interest Rates"----
dap_futures |>
  filter(symbol %in% c("DAPF23", "DAPK35")) |>
  ggplot(aes(x = refdate, y = adjusted_tax, group = symbol, color = symbol)) +
  geom_line() +
  geom_point() +
  labs(
    title = "DAP Future Rates - Real Interest Rates",
    caption = "Source B3 - package rb3",
    x = "Date",
    y = "Interest Rates",
    color = "Symbol"
  ) +
  scale_y_continuous(labels = scales::percent)

## ----create-implied-inflation-data, warning=FALSE-----------------------------
infl_futures <- df |>
  filter(symbol %in% c("DI1F23", "DAPF23"))

infl_expec <- infl_futures |>
  select(symbol, price, refdate) |>
  tidyr::pivot_wider(names_from = symbol, values_from = price) |>
  mutate(inflation = DAPF23 / DI1F23 - 1)

## ----implied-inflation-plot, warning=FALSE, fig.width=7, fig.height=5, fig.cap="Implied Inflation from futures"----
infl_expec |>
  ggplot(aes(x = refdate, y = inflation)) +
  geom_line() +
  geom_point() +
  labs(
    x = "Date", y = "Inflation",
    title = "Implied Inflation from futures DI1F23, DAPF23",
    caption = "Source B3 - package rb3"
  )

## ----implied-inflation-F24, warning=FALSE, fig.width=7, fig.height=5, fig.cap="Implied Inflation from futures"----
df |>
  filter(symbol %in% c("DI1F24", "DAPF24")) |>
  select(symbol, price, refdate) |>
  tidyr::pivot_wider(names_from = symbol, values_from = price) |>
  mutate(inflation = DAPF24 / DI1F24 - 1) |>
  na.omit() |>
  ggplot(aes(x = refdate, y = inflation)) +
  geom_line() +
  geom_point() +
  labs(
    x = "Date", y = "Inflation",
    title = "Implied Inflation from futures DI1F24, DAPF24",
    caption = "Source B3 - package rb3"
  )

## ----filter-data--------------------------------------------------------------
df_fut <- df |>
  filter(symbol %in% c("DI1F23", "DI1F33"))

## ----enrich-data, warning=FALSE-----------------------------------------------
df_fut <- df_fut |>
  mutate(
    maturity_date = maturitycode2date(maturity_code) |>
      following("Brazil/ANBIMA"),
    business_days = bizdays(refdate, maturity_date, "Brazil/ANBIMA")
  )

## ----calculate-business-days--------------------------------------------------
df_du <- df_fut |>
  select(refdate, symbol, business_days) |>
  tidyr::pivot_wider(names_from = symbol, values_from = business_days) |>
  mutate(
    du = DI1F33 - DI1F23
  ) |>
  select(refdate, du)

## ----join-dataframes----------------------------------------------------------
df_fut <- df_fut |>
  select(refdate, symbol, price) |>
  tidyr::pivot_wider(names_from = symbol, values_from = price) |>
  inner_join(df_du, by = "refdate")

## ----calculate-forward-rates--------------------------------------------------
df_fwd <- df_fut |>
  mutate(
    fwd = (DI1F23 / DI1F33)^(252 / du) - 1
  ) |>
  select(refdate, fwd) |>
  na.omit()

## ----plot-forward-rates, warning=FALSE, fig.width=7, fig.height=5, fig.cap="Forward Rates"----
df_fwd |>
  ggplot(aes(x = refdate, y = fwd)) +
  geom_line() +
  labs(
    x = "Date", y = "Forward Rates",
    title = "Historical 10Y Forward Rates - F23:F33",
    caption = "Source B3 - package rb3"
  )

