#define version "patc v1.1c (c) Jeroen Hellingman 01-MAY-1993\n"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "PSTree.h"

#define TRUE    (1==1)
#define FALSE   (1==0)

#define NUMPATS 10  /* number of pattern trees */
#define PATLEN  50  /* maximum length of pattern */
#define BUFSIZE 512 /* pushback buffer size (BUFSIZE >= PATLEN) */

/* datatypen */

typedef struct patterntree
{   PSTree *t;      /* pattern tree for this node */
    int     r;      /* restrictive or not? */
} patterntree;

/* globals */

FILE *infile;
FILE *outfile;
patterntree pat[NUMPATS];
char *progname = "patc";
char *patfilename = "patc.pat";
char *infilename;
char *outfilename;
static int quiet = TRUE;            /* be quiet */
static int linenumber = 1;          /* current line in infile */

/* prototypes */

void parsetables(FILE *patfile);
void patc(void);
void processflags(int argc, char** argv);
void usage(void);
void copytexcommand(void);
void skiptexcommand(void);
void skiptillmatchingbrace(void);
void skiptillchar(char c1);
void copycomment(void);
void skipcomment(void);
int readchar(void);
void unreadchar(int);
int what_escape(const char *s, char *result);

/***********************************************************************/

void PUSHBACK(char *c)
/* push the characters in string c back into the inputstream, works
 * with the pair of functions readchar() and unreadchar()
 */
{   int i = (int)strlen(c)-1;
    for( ;i >= 0; i--) unreadchar((int)c[i]);
}

void main(int argc, char** argv)
/* check arguments, intialize tables, open files
 */
{   FILE *patfile;

    processflags(argc, argv);
    if(!quiet) fputs(version, stderr);
    
    patfile = fopen(patfilename, "r");
    if(patfile==NULL)
    {   fprintf(stderr, "%s: can't open %s\n", progname, patfilename);
        exit(2);
    }
    parsetables(patfile);
    fclose(patfile);
    
    infile = fopen(infilename, "r");
    if(infile==NULL)
    {   fprintf(stderr, "%s: can't open %s\n", progname, infilename);
        exit(2);
    }
    outfile = fopen(outfilename, "w");
    if(outfile==NULL)
    {   fprintf(stderr, "%s: can't create %s\n", progname, outfilename);
        exit(2);
    }    
    patc();
    fclose(infile);
    fclose(outfile);
    exit(0);
}

void processflags(int argc, char** argv)
{   int nextoption = FALSE;
    int i = 1;
    
    if(argc < i+2) usage();
    if(argv[i][0] == '-') nextoption = TRUE;
    
    while(nextoption)
    {   switch(argv[i][1])
        {   case 'v':   quiet = FALSE;      break;
            case 'p':   patfilename = argv[i+1]; i++; break;
            default:    usage();
        }
        i++;
        if(argc < i+2) usage();
        if(argv[i][0] != '-') nextoption = FALSE;
    }
    
    infilename  = argv[i];
    outfilename = argv[i+1];
}

void usage()
{   fprintf(stderr, "usage: %s [-v] [-p patfile] infile outfile\n", progname);
    exit(1);
}

/***********************************************************************/

int readline(char *l, FILE* infile)
{   int i = 0;
    int c = fgetc(infile);
    while(isspace(c) && c != '\n' && c != EOF) c = fgetc(infile); /* skip whitespace */
    if(c == '%') while(c != '\n' && c != EOF) c = fgetc(infile); /* skip comments */
    while(c != '\n' && c != EOF && i < BUFSIZE)
    {   l[i] = (c == EOF) ? '\0' : (int) c;
        i++;
        c = getc(infile);
    }
    l[i] = '\0';    /* NULL-terminate */
    return (c!=EOF);
}

int getword(char *s, char *d)
{   int i = 0, j = 0;
    while(isspace(s[i]) && s[i] != '\0') i++;
    while(isalnum(s[i])) { d[j] = s[i]; j++; i++; }
    d[j] = '\0';
    return i;
}

int getquotedstring(const char *s, char *d)
{   int i = 0;  /* no of chars read in source */
    int j = 0;  /* no of chars inserted in destination */
    while(isspace(s[i]) && s[i] != '\0') i++;

    if(s[i] == '"')
    {   i++;
        while(s[i] != '"')
        {   if(s[i] == '\\') /* escape char */
                i += what_escape(&s[i], &d[j]);
            else
            {   d[j] = s[i];
            }
            j++; i++;
        }
    }
    d[j] = '\0';    /* NULL-terminate */
    i++;            /* skip final " */
    return i;
}

/* Find out what escape sequence is used. If non can be found, we just 
   forget about the backslash. Interprete numbers up to 255/177/FF
*/

#define UNSIGNED(t) (char)(((t) < 0) ? (t) + 256 : (t))

int what_escape(const char *s, char *result)
{   int i = 1;      /* length of escape sequence read */
    int ok = TRUE;
    int t = 0;      /* temporary result */

    switch(s[1])
    {   case '"':   *result = '"';  break;
        case '\\':  *result = '\\'; break;
        case 't':   *result = '\t'; break;
        case 'n':   *result = '\n'; break;
        case 'b':   *result = '\b'; break;
        case 'h':   /* hexadecimal */
            while(i < 3 && ok)
            {   i++;
                if(s[i]>='0' && s[i]<='9') t = t * 16 + (s[i] - '0');
                else if(s[i]>='A' && s[i]<='F') t = t * 16 + (s[i] - 'A') + 10;
                else if(s[i]>='a' && s[i]<='f') t = t * 16 + (s[i] - 'a') + 10;
                else
                {   if(i==2) /* no number after \h */
                        *result = 'h';
                    else /* short number after \h */
                        *result = UNSIGNED(t);
                    i--;
                    ok = FALSE; 
                }
            }
            if(ok) *result = UNSIGNED(t);
            break;
        case 'd':   /* decimal */
            while(i < 4 && ok)
            {   i++;
                if(s[i]>='0' && s[i]<='9') t = t * 10 + (s[i] - '0');
                else
                {   if(i==2) /* no number after \d */
                        *result = 'd';
                    else /* short number after \d */
                        *result = UNSIGNED(t);
                    i--;
                    ok = FALSE; 
                }
            }
            if(ok) *result = UNSIGNED(t);
            break;
        default:    /* try octal interpretation */
            i--;
            while(i < 3 && ok)
            {   i++;
                if(s[i]>='0' && s[i]<='7') t = t * 8 + (s[i] - '0');
                else
                {   if(i==1) /* no number after \ */
                        *result = s[i];
                    else /* short number after \ */
                    {   *result = UNSIGNED(t);
                        i--;
                    }
                    ok = FALSE; 
                }
            }
            if(ok) *result = UNSIGNED(t);
    }
    return i;
}

void parsetables(FILE *patfile)
{   char line[BUFSIZE];
    char command[BUFSIZE];
    char pattern[BUFSIZE];
    char action[BUFSIZE];
    char *tmp;
    int pos = 1;
    char notEOF = TRUE;
    int currentpat = 0; /* current patterntree under construction */

    while(notEOF)
    {   notEOF = readline(line, patfile);
        pos = 0;
        switch(line[0])
        {   case '\0':  /* empty line */    break;
            case '@':   /* command */
                        pos++;
                        pos += tolower(getword(&line[1], command));
                        if(strcmp(command, "patterns") == 0)
                        {   pos += getword(&line[pos], command);
                            currentpat = atoi(command);
                            pat[currentpat].r = FALSE;
                        }
                        else if(strcmp(command, "rpatterns") == 0)
                        {   pos += getword(&line[pos], command);
                            currentpat = atoi(command);
                            pat[currentpat].r = TRUE;
                        }
                        else if(strcmp(command, "end") == 0) return;
                        else
                        {   fprintf(stderr, "Error: unknown command %s\n", command);
                            exit(1);
                        }
                        break;
            case '"':   /* action */
                        pos += getquotedstring(line, pattern);
                        pos += getword(&line[pos], command);
                        pos += getquotedstring(&line[pos], &action[1]);
                        action[0] = command[0];
                        tmp = malloc(strlen(action) + 1);
                        if(tmp == NULL)
                        {   fprintf(stderr, "Error: cannot allocate\n");
                            exit(3);
                        }
                        strcpy(tmp, action);
                        PSTinsert(&pat[currentpat].t, pattern, tmp);
                        break;
            default:    fprintf(stderr, "Error: illegal line '%s' in %s\n",
                                    line, patfilename);
                        fprintf(stderr, "I will forget it\n");
        }
    }
}

/***********************************************************************/

void patc()
{   char ps[PATLEN+1];      /* pattern to be search for */
    char *action;           /* action with pattern */
    int len = PATLEN;       /* length of found pattern; part of ps to be read */
    int current = 0;        /* current active patterntree */
    int i, j;               /* counters */

    while(TRUE)
    {
        /* fill pattern */
        for(i = 0, j = len; j < PATLEN; i++, j++) ps[i] = ps[j];
        for(i = PATLEN - len; i < PATLEN; i++)
        {   int c = readchar();
            ps[i] = (c == EOF) ? '\0' : c;
        }
        ps[PATLEN] = '\0'; /* NULL-terminate */
        if(ps[0] == '\0') break;
        
        /* find action */
    
        action = PSTmatch(pat[current].t, ps, &len);
        
        if(len == 0)
        {   if(pat[current].r) /* complain */
                fprintf(stderr, "Error: illegal character '%c' near line %d\n", ps[0], linenumber);
            else /* copy silently */
                fputc(ps[0], outfile);
            len = 1;
        }
        else /* do action */
        {   switch(action[0])
            {   case 'p':   fputs(&action[1], outfile);         break;
                case 'c':   PUSHBACK(ps); len = PATLEN; copycomment();      break;
                case 't':   PUSHBACK(ps); len = PATLEN; copytexcommand();       break;
                case 'T':   PUSHBACK(ps); len = PATLEN; skiptexcommand();       break;
                case 's':   PUSHBACK(ps); len = PATLEN; skipcomment();      break;
                case 'f':   /* forget */                        break;
                case 'e':   fprintf(stderr, "Error: %s near line %d\n", &action[1], linenumber);
                            break;
				case 'S':   PUSHBACK(ps); len = PATLEN; skiptillchar(action[1]);	break;
			    case 'B':   PUSHBACK(ps); len = PATLEN; skiptillmatchingbrace();	break;
                case '0':   current = 0; fputs(&action[1], outfile); break;
                case '1':   current = 1; fputs(&action[1], outfile); break;
                case '2':   current = 2; fputs(&action[1], outfile); break;
                case '3':   current = 3; fputs(&action[1], outfile); break;
                case '4':   current = 4; fputs(&action[1], outfile); break;
                case '5':   current = 5; fputs(&action[1], outfile); break;
                case '6':   current = 6; fputs(&action[1], outfile); break;
                case '7':   current = 7; fputs(&action[1], outfile); break;
                case '8':   current = 8; fputs(&action[1], outfile); break;
                case '9':   current = 9; fputs(&action[1], outfile); break;
                default:    fprintf(stderr, "Internal error: unknown action\n");
                            exit(10);
            } /* switch */
        } /* else */
    } /* while */
    if(current != 0)
        fprintf(stderr, "Warning: mode = %d at end of file\n", current);
} /* patc() */

/***********************************************************************/

void skiptillchar(char c1)
{	char c2;
	do
	{	c2 = readchar();
		if(c2 == EOF) return;
	} while(c2 != c1);
}

void skiptillmatchingbrace()
/* skip a TeX group enclosed in braces.
 * next brace on input opens the group to skip
 */
{	int i = 1;
	int c;	
	do
	{	c = readchar();
		if(c == EOF) return;
	} while(c != '{');
	while(i>0)
	{	c = readchar();
		if(c == '{') i++;
		if(c == '}') i--;
		if(c == EOF) return;
	}
}

void copytexcommand()
/* copy TeX commmand, including preceding \
 * this will work in plain TeX and LaTeX
 */
{   int c = readchar();
    if(c=='\\')
    {   fputc(c, outfile);
        c = readchar();
        if(isalpha(c))
        {   while(isalpha(c))
            {   fputc(c, outfile);
                c = readchar();
            }
            unreadchar(c);
        }
        else
            fputc(c, outfile);
    }
    else
        unreadchar(c);
}

void skiptexcommand()
/* skip TeX commmand, including preceding \
 * this will work in plain TeX and LaTeX
 */
{   int c = readchar();
    if(c=='\\')
    {   c = readchar();
        if(isalpha(c))
        {   while(isalpha(c))
            {   c = readchar();
            }
            unreadchar(c);
        }
    }
    else
        unreadchar(c);
}


void copycomment()
{   int c = readchar();
    if(c=='%')
    {   while(c != '\n' && c != EOF)
        {   fputc(c, outfile);
            c = readchar();
        }
        fputc('\n', outfile);
    }
    else
        unreadchar(c);
}

void skipcomment()
{   int c = readchar();
    if(c=='%')
    {   while(c != '\n' && c != EOF)
            c = readchar();
    }
    else
        unreadchar(c);
}

/***********************************************************************/
/* file access with buffer */

static int fbuffer[BUFSIZE];        /* buffer for file operations */
static int last = 0;                /* last + 1 used in fbuffer */

int readchar()
{   int result;

    if(last==0) /* niets in buffer */
        result = fgetc(infile);
    else /* pak first uit buffer */
    {   last--;
        result = fbuffer[last];
    }
    if(result == '\n') linenumber++;
    return result;
}

void unreadchar(int c)
{  
    if(last == BUFSIZE)
    {   fprintf(stderr, "%s: push-back file buffer full\n", progname);
        exit(1);
    }
    else /* voeg na last in buffer */
    {   fbuffer[last] = c;
        last++;
        if(c == '\n') linenumber--;
    }
}

/***********************************************************************/
