/* fnamedb.c - File Name Database
   Time-stamp: "97/08/29 13:13:10 cschenk"

   Copyright (C) 1996, 97 Christian Schenk <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <miktex.h>
#include <miktex.rc>

#define ENABLE_ITERATIONS

#define stack_strdup(s) strcpy ((char *) _alloca (strlen (s) + 1), (s))
#define isslash(ch) ((ch) == '/' || (ch) == '\\')

static int
lastch (const char *s)

{
  size_t len;

  xassert (s != 0);

  len= strlen (s);
  return (len == 0 ? 0 : s[len - 1]);
}

/* directory/file properties */
typedef struct

{
  char *	name;		/* file name */
} properties;

/* directory node */
typedef struct dirnode_struct

{
  properties			props; /* properties of this directory */
  size_t			nsubdirs; /* number of sub-directories */
  struct dirnode_struct *	parent; /* pointer to parent directory */
  struct dirnode_struct *	subdirs; /* sorted array of sub-directories */
  size_t			nfiles;	/* number of files */
  properties *			files; /* sorted array of files */
} dirnode;

/* file name data base */
typedef struct

{
  int		reserved_flag;	/* 0, if this fndb is not in use. */
  char *	root;		/* path name of root directory */
  dirnode	dir;		/* the top node */
  int		root_dir_handle;
} fndb;

/* _________________________________________________________________________

   PATH NAME OPERATIONS.
   _________________________________________________________________________ */


/* Return the filename part of PATH. Zero-terminate the directory part. */
static char *
chop_path (char *path)

{
  char *cp =  path + strlen (path) - 1;
  while (! isslash (*cp) && cp != path)
    cp--;
  if (isslash (*cp))
    {
      *cp = 0;
      return (cp + 1);
    }
  else
    return (path);
}

#define is_relative_path(path) (! is_absolute_path (path))

/* Test if PATH is absolute. */
static int
is_absolute_path (const char *path)

{
  if (isslash (path[0]))
    return (1);
  else if (isalpha (path[0])
	   && path[1] == ':'
	   && isslash (path[2]))
    return (1);
  else
    return (0);
}

/* Return the part of PATH that is relative to ROOT. */
static const char *
relative_path (const char *path,
	       const char *root)

{
  const char *ret;
  if (is_relative_path (path))
    ret = path;
  else
    {
      size_t rootlen = strlen (root);
      if (_strnicmp (path, root, rootlen) == 0)
	{
	  ret = path + rootlen;
	  if (! isslash (lastch (root)))
	    ret += 1;
	}
      else
	ret = 0;
    }
  return (ret);
}

/* _________________________________________________________________________

   THE STRING POOL.
   _________________________________________________________________________ */


/* Return a private copy of STR. */
static char *
save_string (const char *str)

{
#if 1
  return (strdup (str));	/* fixme: never freed */
#else /* NEVER */
  static size_t max_string_pool_size = 0;
  static size_t current_string_pool_size = 0;
  static char *string_pool = 0;
  size_t l = strlen (str);
  char *ret;
  if (string_pool == 0)
    {
#if defined (DEBUG)
      max_string_pool_size = 1024 * 8;
#else
      max_string_pool_size = 1024 * 1024;
#endif
      string_pool = (char *) malloc (max_string_pool_size);
    }
  if (current_string_pool_size + l + 1 > max_string_pool_size)
    {
      while (current_string_pool_size + l + 1 > max_string_pool_size)
	max_string_pool_size *= 2;
      string_pool = (char *) realloc (string_pool, max_string_pool_size);
    }
  if (string_pool == 0)
    return (0);
  ret = string_pool + current_string_pool_size;
  strcpy (ret, str);
  current_string_pool_size += l + 1;
  return (ret);
#endif /* NEVER */
}

/* _________________________________________________________________________

   THE FNDB TABLE.
   _________________________________________________________________________ */


#define max_fndb_handle 10
static fndb fndb_table[max_fndb_handle];

/* Get a free fndb handle. */
static int
get_fndb_handle ()

{
  int handle;
  for (handle = 0; handle < max_fndb_handle; handle++)
    {
      if (fndb_table[handle].reserved_flag == 0)
	{
	  fndb_table[handle].reserved_flag = 1;
	  return (handle);
	}
    }
  return (-1);
}

/* Unget a reserved fndb handle. */
static void
unget_fndb_handle (int handle)

{
  xassert (handle >= 0 && handle < max_fndb_handle);
  xassert (fndb_table[handle].reserved_flag != 0);
  fndb_table[handle].reserved_flag = 0;
}

/* Get address of fndb for a given fndb handle. */
static fndb *
get_fndb (int handle)

{
  xassert (handle >= 0 && handle < max_fndb_handle);
  xassert (fndb_table[handle].reserved_flag != 0);
  return (&fndb_table[handle]);
}

/* _________________________________________________________________________

   FNDB CONSTRUCTION.
   _________________________________________________________________________ */


static read_dir (FILE *, dirnode *, const char *);

/* Create a filename database. Open the root directory. */
int
fndb_new (const char *	dbfilename,
	  const char *	root)

{
  int handle;
  FILE *fp;
  int rc;
  fndb *fndb;

  xassert (dbfilename != 0);
  xassert (root != 0);

  handle = get_fndb_handle ();
  if (handle < 0)
    return (-1);

  fp = fopen (dbfilename, "r");
  if (fp == 0)
    {
      unget_fndb_handle (handle);
      return (-1);
    }

  fndb = get_fndb (handle);
  fndb->root = save_string (root);
  rc = read_dir (fp, & fndb->dir, root);
  fclose (fp);

  if (rc >= 0)
    fndb->root_dir_handle = new_search_context (&fndb->dir);
  else
    unget_fndb_handle (handle);

  return (rc < 0 ? -1 : handle);
}

/* Read a directory listing from DBSTREAM. Store infos in DIR. */
static int
read_dir (FILE *	dbstream,
	  dirnode *	dir,
	  const char *	dirname)

{
  int i;
  char linebuf[50];
  long tmp;
  char *cp, *cp2;
  size_t l;

  /* Read header line. */
  if (fgets (linebuf, sizeof (linebuf), dbstream) == 0)
    return (-1);
  l = strlen (linebuf);
  if (l == 0 || linebuf[l - 1] != '\n')
    return (-1);
  linebuf[l - 1] = 0;
  tmp = strtol (linebuf, &cp, 10);
  dir->nsubdirs = tmp;
  tmp = strtol (cp, &cp2, 10);
  dir->nfiles = tmp;
  while (*cp2 == ' ')
    *cp2++;
  dir->props.name = save_string (dirname ? dirname : cp2);
  dir->files = (properties *) malloc (sizeof (properties) * dir->nfiles);
  if (dir->files == 0)
    return (-1);

  /* Read file names. */
  for (i = 0; i < dir->nfiles; i++)
    {
      if (fgets (linebuf, sizeof (linebuf), dbstream) == 0)
	return (-1);
      l = strlen (linebuf);
      if (l == 0 || linebuf[l - 1] != '\n')
	return (-1);
      linebuf[l - 1] = 0;
      if ((dir->files[i].name = save_string (linebuf)) == 0)
	return (-1);
    }

  /* Read sub-directories. */
  dir->subdirs = (dirnode *) malloc (sizeof (dirnode) * dir->nsubdirs);
  if (dir->subdirs == 0)
    return (-1);
  for (i = 0; i < dir->nsubdirs; i++)
    {
      dir->subdirs[i].parent = dir;
      if (read_dir (dbstream, &dir->subdirs[i], 0) < 0)
	return (-1);
    }

  return (0);
}

/* _________________________________________________________________________

   SEARCH CONTEXTS.
   _________________________________________________________________________ */


/* search context */
typedef struct

{
  const dirnode *	dir;	/* directory to be searched */
#if defined (ENABLE_ITERATIONS)
  size_t		subidx;	/* index into dir->subdirs */
  size_t		fileidx; /* index into dir->files */
#endif
} search_context;

#define max_directory_handle ((int) 100)
static search_context search_context_table[max_directory_handle];

/* Return a free directory handle (index into search_context_table). */
static int
new_search_context (const dirnode *dir)

{
  int h;
  for (h = 0; h < max_directory_handle; h++)
    {
      if (search_context_table[h].dir == 0)
	{
	  search_context_table[h].dir = dir;
#if defined (ENABLE_ITERATIONS)
	  search_context_table[h].subidx = 0;
	  search_context_table[h].fileidx = 0;
#endif
	  return (h);
	}
    }
  return (-1);
}

/* Get address of search context for a given handle. */
static search_context *
get_search_context (int handle)

{
  xassert (handle >= 0 && handle < max_directory_handle);
  return (&search_context_table[handle]);
}

/* Unreserve a directory handle. */
static void
delete_search_context (int h)

{
  xassert (h >= 0 && h < max_directory_handle);
  search_context_table[h].dir = 0;
}

/* _________________________________________________________________________

   SEARCH OPERATIONS.
   _________________________________________________________________________ */


static const dirnode *find_dir (const dirnode *, const char *);

/* Open a directory node. Return a handle for this node. */
int
fndb_open_directory (int		fndb_handle,
		     int		directory_handle,
		     const char *	path)

{
  fndb *		fndb;
  search_context *	c;
  const dirnode *	dir;

  fndb= get_fndb (fndb_handle);
  xassert (fndb != 0);

  if (directory_handle < 0)
    return (fndb->root_dir_handle);

  c = get_search_context (directory_handle);
  xassert (c != 0);

  if (is_relative_path (path))
    dir = find_dir (c->dir, path);
  else
    {
      const char *rel_path = relative_path (stack_strdup (path), fndb->root);
      if (rel_path)
	dir = find_dir (c->dir, rel_path);
      else
	dir = 0;
    }
  if (dir)
    return (new_search_context (dir));
  else
    return (-1);
}

int
fndb_open_next_sub_directory (int directory_handle)

{
#if defined (ENABLE_ITERATIONS)
  search_context *c = get_search_context (directory_handle);
  if (c->dir == 0 || c->subidx >= c->dir->nsubdirs)
    return (-1);
  else
    return (new_search_context (&c->dir->subdirs[c->subidx++]));
#else /* ! ENABLE_ITERATIONS */
  return (-1);
#endif /* ! ENABLE_ITERATIONS */
}

void
fndb_close_directory (int directory_handle)

{
  delete_search_context (directory_handle);
}

/* Return the directory node for a given relative path and a root node. */
static const dirnode *
find_dir (const dirnode *	root,
	  const char *		rel_dirname_path)

{
  const dirnode *dir = root;
  char *dirname = strtok (stack_strdup (rel_dirname_path), "\\/");
  while (dirname)
    {
      /* usually the directory list isn't very long, so we don't use
         bsearch() here */
      size_t i;
      int cmp = -1;
      for (i = 0;
	   (i < dir->nsubdirs
	    && ((cmp = _stricmp (dir->subdirs[i].props.name, dirname)) < 0));
	   i++)
	;
      if (cmp != 0)
	return (0);
      dir = &dir->subdirs[i];
      dirname = strtok (0, "\\/");
    }
  return (dir);
}

/* A helper function for bsearch(). */
static int
propcmp (const properties *ele1,
	 const properties *ele2)

{
  return (_stricmp (ele1->name, ele2->name));
}

static properties *
find_file_in_dir (const dirnode *	dir,
		  char *		path)

{
  char *directory_part = path;
  char *filename_part = chop_path (path);
  properties key;
  if (filename_part != directory_part)
    {
      dir = find_dir (dir, directory_part);
      if (dir == 0)
	return (0);
    }
  key.name = filename_part;
  return ((properties *) bsearch (&key, dir->files, dir->nfiles,
				  sizeof (dir->files[0]), propcmp));
}

static void
makepath (const dirnode *	dir,
	  char *		buffer,
	  size_t		buffer_size)

{
  if (dir->parent)
    {
      makepath (dir->parent, buffer, buffer_size);
      if (! isslash (lastch (buffer)))
	strcat (buffer, "\\");
    }
  strcat (buffer, dir->props.name);
}

/* Try to find a file. */
int
fndb_find_file (int		fndb_handle,
		int		directory_handle,
		const char *	filename_path,
		char *		buffer,
		size_t		buffer_size)

{
  fndb *		fndb = get_fndb (fndb_handle);
  search_context *	c = get_search_context (directory_handle);

  xassert (fndb != 0);
  xassert (c != 0);
  xassert (filename_path != 0);

  if (! is_relative_path (filename_path))
    {
      filename_path = relative_path (stack_strdup (filename_path), fndb->root);
      if (filename_path == 0)
	return (0);
    }

  if (c->dir
      && find_file_in_dir (c->dir, stack_strdup (filename_path)))
    {
      if (buffer)
	{
	  *buffer = 0;
	  makepath (c->dir, buffer, buffer_size);
	  if (! isslash (lastch (buffer)))
	    strcat (buffer, "\\");
	  strcat (buffer, filename_path);
	}
      return (1);
    }
  else
    return (0);
}

/* _________________________________________________________________________

   TEST DRIVER.
   _________________________________________________________________________ */


#if defined (TEST)
static int
_find_file (int			fndb_handle,
	    int			dir_handle,
	    const char *	subdir,
	    const char *	searchspec,
	    char *		result)

{
  int subdir_handle;
  if (subdir)
    {
      subdir_handle = fndb_open_directory (fndb_handle, dir_handle, subdir);
      if (subdir_handle >= 0)
	{
	  int found;
	  found = _find_file (fndb_handle, subdir_handle, 0,
			      searchspec, result);
	  fndb_close_directory (subdir_handle);
	  if (found)
	    return (1);
	}
      while ((subdir_handle = fndb_open_next_sub_directory (dir_handle)) >= 0)
	{
	  int found = _find_file (subdir_handle, subdir, searchspec, result);
	  fndb_close_directory (subdir_handle);
	  if (found)
	    return (1);
	}
    }
  else
    {
      char *slsl = strstr (searchspec, "//");
      if (slsl)
	{
	  size_t len = slsl - searchspec;
	  char *subdir = _alloca (len + 1);
	  strncpy (subdir, searchspec, len);
	  subdir[len] = 0;
	  return (_find_file (dir_handle, subdir, slsl + 2, result));
	}
      else
	{
	  if (fndb_find_file (fndb_handle, dir_handle, searchspec, result, 0))
	    return (1);
	  else
	    {
	      while ((subdir_handle
		      = fndb_open_next_sub_directory (dir_handle)) >= 0)
		{
		  int found = _find_file (subdir_handle, 0,
					  searchspec, result);
		  fndb_close_directory (subdir_handle);
		  if (found)
		    return (1);
		}
	    }
	}
    }
  return (0);
}

static int
find_file (int		fndb_handle,
	   const char *	filename,
	   const char *	path_list,
	   char *	result)

{
  size_t filename_len = strlen (filename);
  char *my_path_list = strdup (path_list);
  char *path = strtok (my_path_list, ";");
  int found = 0;
  while (path && ! found)
    {
      size_t path_len = strlen (path);
      char *search_spec = _alloca (path_len + filename_len + 3);
      strcpy (search_spec, path);
      if (search_spec[path_len - 1] != '/'
	  && search_spec[path_len - 1] != '\\')
	strcat (search_spec, "/");
      strcat (search_spec, filename);
      if (_find_file (fndb_handle, 0, 0, search_spec, result))
	found = 1;
      path = strtok (0, ";");
    }
  free (my_path_list);
  return (found);
}

int
main (int		argc,
      const char **	argv)

{
  char result[300];
  int fndb_handle;
  fndb_handle = fndb_new ("c:/usr/local/texmf/miktex/config/texmf.fndb",
			  "c:/usr/local/texmf");
  if (find_file (fndb_handle, argv[1], argv[2], result))
    printf ("%s\n", result);
  return (0);
}

#endif /* TEST */

/* fnamedb.c ends here */
