% File: latex_cmds.sl                            -*- mode: SLang -*-
%
% Copyright (c)
%       2006--2007      Jörg Sommer <joerg@alea.gnuu.de>
%       $Id: latex_cmds.sl 200 2007-08-23 23:29:21Z joerg $
%
%       -*- This file is part of Jörg's LaTeX Mode (JLM) -*-
%
% Description:   Fixme
%
% License: This program is free software; you can redistribute it and/or
%	   modify it under the terms of the GNU General Public License as
%	   published by the Free Software Foundation; either version 2 of
%	   the License, or (at your option) any later version.
%
%          This program is distributed in the hope that it will be
%	   useful, but WITHOUT ANY WARRANTY; without even the implied
%	   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
%	   PURPOSE.  See the GNU General Public License for more details.

% Sort the file contents:
%   sed '/^\(% *\)\?cmd_register/!d; /^% /{s/^% //; s/$/%/;}' latex_cmds.sl |sort | sed '/%$/{s/%$//; s/^/% /;}'
% Find duplicates:
%   sed '/^\(% *\)\?cmd_register/!d; s/^[^"]*"//; s/".*//;' latex_cmds.sl | uniq -d
% Extract commands from latex files:
%   grep -Eo '\\[a-zA-Z]+(\{[^}]*\})*' skript.latex | sed '/\\begin{/d; s/{[^}]*}/{}/g'
% Align command:
%   while (down(1)) if (ffind_char(',')) { ()=right(1); trim(); whitespace(37-what_column()); };

static define cmd_hook_null(name, mark)
{
    % This is a simple function to remove the arguments from the
    % stack put by cmd_insert() there
}

static define cmd_hook_goto_mark(name, mark)
{
    % This simple function is used for \LaTeX, \TeX and other
    % commands they need {} after it to not eat up the space
    goto_user_mark(mark);
}

static define cmd_hook_file_select(name, mark)
{
    try
    {
        variable file = read_with_completion("Which file?", "", "", 'f');
        if ( path_is_absolute(file) )
        {
            variable bufdir;
            (,bufdir,,) = getbuf_info();
            variable bufdir_tok = strtok(bufdir, "/");
            variable file_tok = strtok(file, "/"), x;

            for (x=0; x < length(file_tok) and x < length(bufdir_tok); ++x)
              if (strcmp(file_tok[x], bufdir_tok[x]) != 0)
                break;

            % We convert the absolute path into a relative path if the
            % number of directories we must step downward in bufdir
            % (=length(bufdir_tok)-x) is less than the number of elements
            % we must step upward from the root in file (=x)
            if ( length(bufdir_tok) <= 2*x )
            {
                --x;
                file = "";
                loop ( length(bufdir_tok)-x-1 )
                  file += "../";

                file += strjoin(file_tok[[x+1:]], "/");
            }
        }
        () = right(1);
        insert(file);
        goto_user_mark(mark);
    }
    catch UserBreakError: {}
}

static define cmd_hook_math(name, mark)
{
    variable my_mark = create_user_mark();
    if (mark == my_mark)
      % we don't need to watch because the jump would go to this position
      return;

    goto_user_mark(mark);         % move behind the $
    update(0);
    variable ch = getkey();

    if (is_substr("=-+<>", char(ch)))
      goto_user_mark(my_mark);

    ungetkey(ch);
}

private variable not_table = Assoc_Type[String_Type];

not_table["cong"] = "ncong";
% not_table["equiv"] = "notequiv";
not_table["exists"] = "nexists";
% not_table["gg"] = "ngg";
not_table["in"] = "notin";
not_table["Leftarrow"] = "nLeftarrow";
not_table["leftarrow"] = "nleftarrow";
not_table["Leftrightarrow"] = "nLeftrightarrow";
not_table["leftrightarrow"] = "nleftrightarrow";
% not_table["ll"] = "nll";
not_table["mid"] = "nmid";
not_table["models"] = "nvDash";
not_table["ni"] = "notni";
not_table["parallel"] = "nparallel";
% not_table["perp"] = "notperp";
not_table["Rightarrow"] = "nRightarrow";
not_table["rightarrow"] = "nrightarrow";
not_table["sim"] = "nsim";
% not_table["subset"] = "nsubset";
not_table["subseteq"] = "nsubseteq";
% not_table["supset"] = "nsupset";
not_table["supseteq"] = "nsupseteq";
not_table["vDash"] = "nvDash";
not_table["vdash"] = "nvdash";

static define cmd_hook_math_not(name, mark)
{
    variable len = strlen(name) + 5;      % \not\$name
    () = left(len);
    if ( looking_at("\\not") )
    {
        if ( assoc_key_exists(not_table, name) )
        {
            try
            {
                cmd_insert(not_table[name], 0);
                deln(len);
            }
            catch ApplicationError;
        }
    }

    goto_user_mark(mark);
}

private variable ref_filt = "";
private variable ref_mark, ref_line_mark, ref_omark;
private variable REF_MODE = MODE+"-label";

static define ref_next(dir)
{
    if (dir)
    {
        % forward
        !if ( orelse {not right(1)} {fsearch("\\label{" + ref_filt)} )
        {
            () = left(1);
            throw UsageError, "Nothing found";
        }
    }
    else
    {
        % search backward
        !if ( bsearch("\\label{" + ref_filt) )
          throw UsageError, "Nothing found";
    }
    ref_line_mark = create_line_mark(color_number("region"));
    message("UP/DOWN: label matchs \"" + ref_filt +
            "\"  Q quit  ENTER insert  SPACE set filter");
}

static define ref_quit()
{
    use_keymap(MODE);
    set_readonly(0);

    goto_user_mark(ref_mark);
    __uninitialize(&ref_line_mark);	% remove the highlighting
}

static define ref_insert()
{
    !if ( looking_at("\\label{") )
      throw UsageError, "Did not find \"\\label{\". Do you moved the cursor?";

    () = right(7);
    push_mark();
    !if ( ffind_char('}') )
      eol();

    bufsubstr();

    ref_quit();

    insert( () );
    goto_user_mark(ref_omark);
}

static define ref_filter()
{
    try
      ref_filt = read_mini("Label filter:", "", ref_filt);
    catch UserBreakError;
    message("UP/DOWN: label matchs \"" + ref_filt +
            "\"  Q quit  ENTER insert  SPACE set filter");
}

static define cmd_hook_ref(name, mark)
{
    ref_omark = mark;
    !if ( keymap_p(REF_MODE) )
    {
        make_keymap(REF_MODE);

        definekey("latex->ref_next(1)", "n", REF_MODE);
        definekey("latex->ref_next(1)", Key_Down, REF_MODE);
        definekey("latex->ref_next(0)", "p", REF_MODE);
        definekey("latex->ref_next(0)", Key_Up, REF_MODE);

        definekey("latex->ref_insert()", Key_Enter, REF_MODE);
        definekey("latex->ref_insert()", Key_Return, REF_MODE);
        definekey("latex->ref_quit()", "q", REF_MODE);
        definekey("latex->ref_quit()", "\007", REF_MODE);  % Ctrl-G
        definekey("latex->ref_filter()", Key_Space, REF_MODE);
    }
    use_keymap(REF_MODE);
    set_readonly(1);
    () = right(1);
    ref_mark = create_user_mark();
    ref_filt = "";
    message("UP/DOWN: label matchs \"\"  Q quit  ENTER insert  SPACE set filter");
}

static define cmd_hook_label(name, mark)
{
    variable start_mark = create_user_mark();
    variable label = "";

    % defaults as proposed in info latex Node label
    if ( is_math() )
      label = "eq:";
    else
    {
        % figure -> fig: table -> tab: and so on
        label = env_name();
        if (andelse {label == "document"} {re_bsearch("\\[sc][euh][cba]"R)}) % Fixme: PCRE
        {
            if ( orelse {looking_at("\\section{")} {looking_at("\\subsection{")}
                 {looking_at("\\subsubsection{")} )
              label = "sec";
            else if ( looking_at("\\chapter{") )
              label = "cha";
        }

        if (strlen(label) > 4)
          label = substr(label, 1, 3);

        label += ":";
    }

    if (label != "")
    {
        variable l = "\\$name{$label"$;
        if ( bsearch(l) )
        {
            () = right( strlen(l) );
            push_mark();
            () = ffind_char('}');
            try
              label += string(integer( bufsubstr() )+1);
            catch ParseError;
        }
        else
          label += "1";
    }

    goto_user_mark(start_mark);
    try
    {
        do
        {
            label = read_mini("Label:", "", label);
            bob();
        }
        while ( andelse {fsearch("\\$name{$label}"$)}
                  {get_y_or_n("Label already defined. Use it") == 0});

        goto_user_mark(start_mark);
        () = right(1);
        insert(label);
        goto_user_mark(mark);
    }
    catch UserBreakError:
    {
        goto_user_mark(mark);
        push_mark();
        () = bfind("\\" + name + "{");
        del_region();
    }
}

%%%%%%%%%%
%
% Stuff for the cmd_hook_bib_cite
%

private variable BIB_KEYMAP = "latex_bib";
private variable bib_mark, bib_old_status_line;

!if ( keymap_p(BIB_KEYMAP) )
{
    make_keymap (BIB_KEYMAP);

    definekey_reserved("latex->bib_quit", "q", BIB_KEYMAP);
    definekey_reserved("latex->bib_quit", "^q", BIB_KEYMAP);
    definekey_reserved("latex->bib_select", "s", BIB_KEYMAP);
    definekey_reserved("latex->bib_select", "^s", BIB_KEYMAP);
    definekey("latex->newline_with_completion", Key_Shift_Return, BIB_KEYMAP);
    definekey_reserved("latex->newline_with_completion", "\n", BIB_KEYMAP);
}

static define cmd_hook_bib_cite(name, mark)
{
    bib_mark = create_user_mark();

    bob();
    if ( fsearch("\\begin{thebibliography}") )
    {
        push_mark();
        !if ( fsearch("\\end{thebibliography}") )
        {
            pop_mark(0);
            throw ApplicationError, "\\begin{thebibliography} found but no \\end";
        }
    }
    else
    {
        !if ( fsearch("\\end{document}") )
          throw ApplicationError, "No \\end{document} found";

        insert("\\clearpage\n\n");
        () = up(1);
        push_spot();
        env_insert("thebibliography", "{99}");
        () = fsearch("\\end{thebibliography}");
        push_mark();
        pop_spot();
    }

    narrow();
    bob();
    use_keymap(BIB_KEYMAP);
    bib_old_status_line = Status_Line_String;
    set_status_line(" "+expand_keystring(_Reserved_Key_Prefix)+
                    "s: select bibitem    "+
                    expand_keystring(_Reserved_Key_Prefix)+"q: quit  ", 0);
}

static define bib_quit()
{
    widen();
    goto_user_mark(bib_mark);
    use_keymap(MODE);
    set_status_line(bib_old_status_line, 0);
}

static define bib_select()
{
    bol();
    !if ( ffind("\\bibitem{") )
      throw UsageError, "No \\bibitem in this line";

    () = right(9);
    push_mark();
    !if ( ffind_char('}') )
    {
        pop_mark(0);
        throw ApplicationError, "Malformed \\bibitem: No } found";
    }

    variable item = bufsubstr();
    bib_quit();
    if ( looking_at_char('{') )
    {
        () = right(1);
        insert(item);
    }
    else
    {
        push_spot();
        ffind_char('{');
        () = right(1);
        insert(item);
        pop_spot();
    }
    () = right(1);
}

%%%%%%%%%%
%
% command definitions
%

% Fixme: many symbols are in groff_char(7)

cmd_register("acute",               0, 1, "", "", NULL);
cmd_register("add",                 2, 1, "adds row/column n to row/column k in a gmatrix", "gauss", NULL);
cmd_register("addtocounter",        2, 0, "Increments the counter (1st arg) by 2nd arg", "", NULL);
cmd_register("addtolength",         2, 0, "Adds #2 to the length register #1", "", NULL);
#if (_slang_utf8_ok)
cmd_register("alpha",               0, 1, "Greek character α", "", &cmd_hook_math());
#else
cmd_register("alpha",               0, 1, "Greek character", "", &cmd_hook_math());
#endif
cmd_register("AMS",                 1, 0, "The AMS logo", "dtklogos", &cmd_hook_goto_mark());
cmd_register("AMSLaTeX",            1, 0, "The AMS-LaTeX logo", "dtklogos", &cmd_hook_goto_mark());
cmd_register("AMSTeX",              1, 0, "The AMS-TeX logo", "dtklogos", &cmd_hook_goto_mark());
cmd_register("analg",               0, 1, "binary operator", "", NULL);
cmd_register("appendix",            0, 0, "Start the appendix", "", NULL);
#if (_slang_utf8_ok)
cmd_register("approx",              0, 1, "Mathematical sign for approx ≈", "", NULL);
#else
cmd_register("approx",              0, 1, "Mathematical sign for approx", "", NULL);
#endif
cmd_register("arccos",              0, 1, "mathematical function", "", NULL);
cmd_register("arcsin",              0, 1, "mathematical function", "", NULL);
cmd_register("arctan",              0, 1, "mathematical function", "", NULL);
cmd_register("arg",                 0, 1, "mathematical function", "", NULL);
cmd_register("arrowvert",           0, 1, "mathematical delimiter", "", NULL);
cmd_register("Arrowvert",           0, 1, "mathematical delimiter", "", NULL);
cmd_register("ast",                 0, 1, "Asterisk *", "", NULL);
cmd_register("asymp",               0, 1, "binary relations", "", NULL);
% cmd_register("atop",                2, 0, "", "", NULL);
cmd_register("author",              1, 0, "Sets the author of the document", "", NULL);
cmd_register("autoref",             1, 0, "", "hyperref", &cmd_hook_ref());
% cmd_register("backslash",           0, 0, "", "", NULL);
cmd_register("backslashbox",        2, 0, "A big slash in the upper left corner of a tabular", "slashbox", NULL);
cmd_register("bar",                 1, 1, "Draw a line over a math symbol", "", NULL);
cmd_register("beta",                0, 1, "Greek character", "", &cmd_hook_math());
cmd_register("bfseries",            0, 0, "Changes the font series to bold face", "", NULL);
cmd_register("bibitem",             1, 0, "Specify a bibliography item", "", NULL);
cmd_register("BibTeX",              1, 0, "The BibTeX logo", "dtklogos", &cmd_hook_goto_mark());
% cmd_register("big",                 0, 1, "", "", NULL);
% cmd_register("Big",                 0, 1, "", "", NULL);
cmd_register("bigcap",              0, 1, "Big intersection sign", "", NULL);
cmd_register("bigcirc",             0, 1, "binary operator", "", NULL);
cmd_register("bigcup",              0, 1, "Big union sign", "", NULL);
% cmd_register("bigg",                0, 0, "", "", NULL);
% cmd_register("Bigg",                0, 0, "", "", NULL);
% cmd_register("biggl",               0, 0, "", "", NULL);
% cmd_register("Biggl",               0, 0, "", "", NULL);
% cmd_register("biggr",               0, 0, "", "", NULL);
% cmd_register("Biggr",               0, 0, "", "", NULL);
% cmd_register("bigl",                0, 0, "", "", NULL);
% cmd_register("Bigl",                0, 0, "", "", NULL);
% cmd_register("bigr",                0, 0, "", "", NULL);
% cmd_register("Bigr",                0, 0, "", "", NULL);
cmd_register("bigtriangledown",     0, 1, "binary operator", "", NULL);
cmd_register("bigtriangleup",       0, 1, "binary operator", "", NULL);
cmd_register("bigvee",              0, 1, "", "", NULL);
cmd_register("bigwedge",            0, 1, "", "", NULL);
cmd_register("binom",               2, 1, "binomial expression; \\frac without horizontal line", "amsmath", NULL);
cmd_register("blacksquare",         0, 1, "A black filled square", "amssymb", NULL);
cmd_register("blacktriangle",       0, 1, "A black filled triangle", "amssymb", NULL);
cmd_register("bmod",                1, 1, "The mod operator for binary relation", "amsmath", NULL);
cmd_register("boolean",             1, 0, "Queries a boolean register defined the \\newboolean", "ifthen", NULL);
cmd_register("bottomrule",          0, 0, "A horizontal line with better spacing at bottom of a tabular"R, "booktabs", NULL);
cmd_register("bowtie",              0, 1, "binary relations", "", NULL);
cmd_register("Box",                 0, 1, "an empty box", "amssymb", NULL);
% cmd_register("boxed",               0, 0, "", "", NULL);
cmd_register("bracevert",           0, 1, "mathematical delimiter", "", NULL);
cmd_register("breve",               0, 1, "", "", NULL);
cmd_register("bullet",              0, 1, "binary operator", "", NULL);
cmd_register("cap",                 0, 1, "Sign of intersection of sets", "", NULL);
cmd_register("caption",             1, 0, "Caption of a figure or a table", "", NULL);
cmd_register("carriagereturn",      1, 0, "Symbol for carriage return"R, "dingbat", &cmd_hook_goto_mark());
cmd_register("cdot",                0, 1, "Centered dot ⋅", "", NULL);
cmd_register("cdots",               0, 1, "Three centered dots", "", NULL);
cmd_register("centering",           0, 0, "Switches the text alignment to centering", "", NULL);
cmd_register("chapter",             1, 0, "Starts a new chapter in a book or report", "", NULL);
cmd_register("check",               1, 1, "Reverse \\hat", "", NULL);
cmd_register("chi",                 0, 1, "Greek character", "", &cmd_hook_math());
cmd_register("circ",                0, 1, "binary operator", "", NULL);
cmd_register("cite",                1, 0, "Refer to a bibliography item", "", &cmd_hook_bib_cite());
cmd_register("cleardoublepage",     0, 0, "", "", NULL);
cmd_register("clearpage",           0, 0, "", "", NULL);
cmd_register("cline",               1, 0, "a horizontal line spanning multiple columns, e.g. 2-5", "", NULL);
cmd_register("colon",               0, 1, "A colon in math mode; : is the division operator 1:2", "", NULL);
cmd_register("colops",              0, 1, "switches in a gmatrix into column operation mode", "gauss", NULL);
cmd_register("color",               0, 0, "", "", NULL);
#if (_slang_utf8_ok)
cmd_register("cong",                0, 1, "binary relations ≅", "", &cmd_hook_math_not());
#else
cmd_register("cong",                0, 1, "binary relations", "", &cmd_hook_math_not());
#endif
cmd_register("cos",                 0, 1, "mathematical function", "", NULL);
cmd_register("cosh",                0, 1, "mathematical function", "", NULL);
cmd_register("cot",                 0, 1, "mathematical function", "", NULL);
cmd_register("coth",                0, 1, "mathematical function", "", NULL);
cmd_register("csc",                 0, 1, "mathematical function", "", NULL);
cmd_register("cup",                 0, 1, "Sign of union of sets", "", NULL);
cmd_register("curvearrowright",     0, 1, "", "", NULL);
cmd_register("dagger",              0, 1, "binary operator", "", NULL);
cmd_register("dante",               1, 0, "The dante logo", "dtxlogos", &cmd_hook_goto_mark());
cmd_register("Dante",               1, 0, "The full dante logo", "dtxlogos", &cmd_hook_goto_mark());
cmd_register("dashv",               0, 1, "binary relations", "", NULL);
cmd_register("date",                0, 0, "Current date", "", NULL);
cmd_register("ddagger",             0, 1, "binary operator", "", NULL);
cmd_register("ddot",                0, 1, "", "", NULL);
cmd_register("ddots",               0, 1, "Diagonal dots", "", NULL);
cmd_register("DeclareMathOperator", 2, 0, "declare a string as operator like \log"R, "amsmath", NULL);
cmd_register("deg",                 0, 1, "mathematical function", "", NULL);
cmd_register("delta",               0, 1, "", "", &cmd_hook_math());
cmd_register("Delta",               0, 1, "", "", &cmd_hook_math());
cmd_register("det",                 0, 1, "Mathematical function: determinant", "", &cmd_hook_null());
cmd_register("diamond",             0, 1, "binary operator", "", NULL);
cmd_register("dim",                 0, 1, "Mathematical function: dimension", "", NULL);
cmd_register("displaybreak",        0, 1, "Defined the point where an equantion can be broken; Place it immedeatly before \\"R, "amsmath", NULL);
cmd_register("displaystyle",        0, 1, "Switches the style of a formula to display", "", NULL);
cmd_register("div",                 0, 1, "Mathematical operator: divide", "", NULL);
cmd_register("dot",                 0, 1, "", "", NULL);
cmd_register("doteq",               0, 1, "binary relations", "", NULL);
cmd_register("dots",                1, 0, "Dots in textmode with special space correction", "ellipsis", &cmd_hook_goto_mark());
cmd_register("dotsb",               0, 0, "Dots in mathmode for binary operators/relations", "amsmath", NULL);
cmd_register("dotsc",               0, 0, "Dots in mathmode for commas", "amsmath", NULL);
cmd_register("dotsi",               0, 0, "Dots in mathmode for integrals", "amsmath", NULL);
cmd_register("dotsm",               0, 0, "Dots in mathmode for multiplication", "amsmath", NULL);
cmd_register("dotso",               0, 0, "Dots in mathmode for everything else", "amsmath", NULL);
cmd_register("downarrow",           0, 1, "", "", NULL);
cmd_register("dq",                  1, 0, "A \" character", "babel,ngerman", &cmd_hook_goto_mark());
cmd_register("ell",                 0, 1, "", "", NULL);
cmd_register("emph",                1, 0, "Emphasize", "", NULL);
cmd_register("emptyset",            0, 1, "", "", NULL);
cmd_register("enlargethispage",     1, 0, "", "", NULL);
cmd_register("ensuremath",          1, 0, "", "", NULL);
cmd_register("epsilon",             0, 1, "", "", &cmd_hook_math());
cmd_register("eqref",               1, 0, "", "amsmath", &cmd_hook_ref());
#if (_slang_utf8_ok)
cmd_register("equiv",               0, 1, "Symbol for \"equivalent\" ≡", "", NULL);
cmd_register("eta",                 0, 1, "Grrek character ε", "", &cmd_hook_math());
cmd_register("EUR",                 1, 0, "Typsets a amount with the Euro symbol €; {amount}", "eurosym", NULL);
cmd_register("euro",                1, 0, "The Euro symbol €", "eurosym", &cmd_hook_goto_mark());
cmd_register("exists",              0, 1, "Symbol for \"there exists\" ∃", "", NULL);
#else
cmd_register("equiv",               0, 1, "Symbol for \"equivalent\"", "", NULL);
cmd_register("eta",                 0, 1, "Grrek character", "", &cmd_hook_math());
cmd_register("euro",                1, 0, "The Euro symbol", "eurosym", &cmd_hook_goto_mark());
cmd_register("exists",              0, 1, "Symbol for \"there exists\"", "", &cmd_hook_math_not());
#endif
cmd_register("exp",                 0, 1, "exponential function", "", NULL);
% cmd_register("fbox",                0, 0, "", "", NULL);
cmd_register("fg",                  1, 0, "right french quoting sign >>", "babel,frenchb", &cmd_hook_goto_mark());
cmd_register("footnote",            1, 0, "", "", NULL);
cmd_register("footnotemark",        0, 0, "Inserts the next mark for a footnote", "", NULL);
cmd_register("footnotetext",        1, 0, "Sets a new footnote with the current mark", "", NULL);
#if (_slang_utf8_ok)
cmd_register("forall",              0, 1, "Logical \"for all\" ∀", "", &cmd_hook_null());
#else
cmd_register("forall",              0, 1, "Logical \"for all\"", "", &cmd_hook_null());
#endif
cmd_register("foreignlanguage",     2, 0, "Treats #2 as text in language #1", "babel", NULL);
cmd_register("frac",                2, 1, "Fraction [math]", "", NULL);
cmd_register("framebox",            1, 0, "Draws a border around the text", "", NULL);
cmd_register("frown",               0, 1, "binary relations", "", NULL);
cmd_register("gamma",               0, 1, "", "", &cmd_hook_math());
cmd_register("Gamma",               0, 1, "", "", &cmd_hook_math());
cmd_register("gcd",                 0, 1, "mathematical function", "", NULL);
#if (_slang_utf8_ok)
cmd_register("geq",                 0, 1, "Symbol for \"greater or equal than\" ≥", "", NULL);
#else
cmd_register("geq",                 0, 1, "Symbol for \"greater or equal than\"", "", NULL);
#endif
cmd_register("gg",                  0, 1, "binary relations", "", NULL);
cmd_register("glqq",                1, 0, "german left double quote", "babel,ngerman", &cmd_hook_goto_mark());
cmd_register("glq",                 1, 0, "german left single quote", "babel,ngerman", &cmd_hook_goto_mark());
cmd_register("grave",               0, 1, "", "", NULL);
cmd_register("grqq",                1, 0, "german right double quote", "babel,ngerman", &cmd_hook_goto_mark());
cmd_register("grq",                 1, 0, "german right single quote", "babel,ngerman", &cmd_hook_goto_mark());
cmd_register("hat",                 1, 1, "", "", NULL);
cmd_register("hbox",                1, 0, "", "", NULL);
% cmd_register("hdots",               0, 1, "", "", NULL);
cmd_register("hfill",               0, 0, "", "", NULL);
cmd_register("hline",               0, 0, "", "", NULL);
cmd_register("hom",                 0, 1, "mathematical function", "", NULL);
cmd_register("hookrightarrow",      0, 1, "", "", NULL);
cmd_register("href",                2, 0, "Reference to an external resources", "hyperref", NULL);
cmd_register("hspace",              1, 0, "", "", NULL);
cmd_register("hyperref",            1, 0, "Make the arg. be a reference to the target in the opt. arg.", "", NULL); % &label_ref_hook()
cmd_register("hyphenation",         1, 0, "Defines hyphenation points for a list of words", "", NULL);
% cmd_register("id",                  0, 0, "", "", NULL);
cmd_register("ifthenelse",          3, 0, "", "ifthen", NULL);
cmd_register("iiiint",              0, 1, "", "", NULL);
cmd_register("iiint",               0, 1, "", "", NULL);
cmd_register("iint",                0, 1, "", "", NULL);
% cmd_register("ilimits",             0, 0, "", "", NULL);
cmd_register("Im",                  0, 1, "Imaginary part of a number", "", NULL);
cmd_register("in",                  0, 1, "binary relations", "", &cmd_hook_math_not());
cmd_register("include",             1, 0, "", "", &cmd_hook_file_select());
cmd_register("includegraphics",     1, 0, "Inserts a graphic", "graphicx", &cmd_hook_file_select());
% cmd_register("ind",                 0, 0, "", "", NULL);
cmd_register("index",               1, 0, "Creates an index entry", "makeidx", NULL);
cmd_register("inf",                 0, 1, "mathematical function: infimum", "", NULL);
cmd_register("infty",               0, 1, "", "", NULL);
cmd_register("input",               1, 0, "", "", &cmd_hook_file_select());
cmd_register("int",                 0, 1, "", "", NULL);
cmd_register("intertext",           1, 1, "", "", NULL);
cmd_register("item",                0, 0, "New item in an enumerate, itemize or list", "", NULL);
cmd_register("itshape",             0, 0, "", "", NULL);
cmd_register("Join",                0, 1, "binary relations", "", NULL);
cmd_register("kappa",               0, 1, "", "", &cmd_hook_math());
cmd_register("ker",                 0, 1, "mathematical function", "", NULL);
cmd_register("label",               1, 0, "Create an anchor for a reference (\\ref)", "", &cmd_hook_label());
cmd_register("lambda",              0, 1, "", "", &cmd_hook_math());
cmd_register("Lambda",              0, 1, "", "", &cmd_hook_math());
#if (_slang_utf8_ok)
cmd_register("langle",              0, 1, "mathematical delimiter: 〈", "", NULL);
#else
cmd_register("langle",              0, 1, "mathematical delimiter: <", "", NULL);
#endif
cmd_register("LaTeX",               1, 0, "", "", &cmd_hook_goto_mark());
cmd_register("lceil",               0, 1, "mathematical delimiter", "", NULL);
cmd_register("ldots",               1, 0, "Three dots at the base line", "", &cmd_hook_goto_mark());
cmd_register("le",                  0, 1, "the same as \leq"R, "", NULL);
cmd_register("leadsto",             0, 1, "", "wasysym", NULL);
cmd_register("left",                0, 1, "", "", NULL);
#if (_slang_utf8_ok)
cmd_register("Leftarrow",           0, 1, "⇐", "", &cmd_hook_math_not());
cmd_register("leftarrow",           0, 1, "←", "", &cmd_hook_math_not());
cmd_register("leftrightarrow",      0, 1, "↔", "", &cmd_hook_math_not());
cmd_register("Leftrightarrow",      0, 1, "⇔", "", &cmd_hook_math_not());
cmd_register("leq",                 0, 1, "Symbol for \"lower or equal than\" ≤", "", NULL);
#else
cmd_register("Leftarrow",           0, 1, "<=", "", &cmd_hook_math_not());
cmd_register("leftarrow",           0, 1, "<-", "", &cmd_hook_math_not());
cmd_register("leftrightarrow",      0, 1, "<->", "", &cmd_hook_math_not());
cmd_register("Leftrightarrow",      0, 1, "<=>", "", &cmd_hook_math_not());
cmd_register("leq",                 0, 1, "Symbol for \"lower or equal than\"", "", NULL);
#endif
cmd_register("lfloor",              0, 1, "mathematical delimiter", "", NULL);
cmd_register("lg",                  0, 1, "mathematical function: logarithm base 10", "", NULL);
cmd_register("lgroup",              0, 1, "mathematical delimiter", "", NULL);
cmd_register("Lightning",           0, 0, "", "marvosym", NULL);
cmd_register("lightning",           1, 0, "", "wasysym", &cmd_hook_goto_mark());
cmd_register("lim",                 0, 1, "mathematical function: limit", "", NULL);
cmd_register("liminf",              0, 1, "mathematical function: lower limit", "", NULL);
cmd_register("limits",              0, 1, "forces limits to be place under and above", "", NULL);
cmd_register("limsup",              0, 1, "mathematical function: upper limit", "", NULL);
cmd_register("linebreak",           0, 0, "", "", NULL);
cmd_register("lineskip",            0, 0, "Abstand zwischen Zeilen", "", NULL);
cmd_register("linewidth",           0, 0, "", "", NULL);
cmd_register("listfiles",           0, 0, "Print list of used files after compilation", "", NULL);
cmd_register("listtheorems",        0, 0, "", "ntheorem", NULL);
cmd_register("ll",                  0, 1, "binary relations: <<", "", NULL);
cmd_register("lmoustache",          0, 1, "mathematical delimiter", "", NULL);
cmd_register("ln",                  0, 1, "mathematical function: natural logarithm", "", NULL);
cmd_register("log",                 0, 1, "mathematical function: logarithm", "", NULL);
cmd_register("longmapsto",          0, 1, "", "", NULL);
#if (_slang_utf8_ok)
cmd_register("longleftarrow",       0, 1, "←⎯", "", NULL);
cmd_register("Longleftarrow",       0, 1, "⇐═", "", NULL);
cmd_register("longleftrightarrow",  0, 1, "←⎯→", "", NULL);
cmd_register("Longleftrightarrow",  0, 1, "⇐═⇒", "", NULL);
cmd_register("longrightarrow",      0, 1, "⎯→", "", NULL);
cmd_register("Longrightarrow",      0, 1, "═⇒", "", NULL);
#else
cmd_register("longleftarrow",       0, 1, "<--", "", NULL);
cmd_register("Longleftarrow",       0, 1, "<==", "", NULL);
cmd_register("longleftrightarrow",  0, 1, "<-->", "", NULL);
cmd_register("Longleftrightarrow",  0, 1, "<==>", "", NULL);
cmd_register("longrightarrow",      0, 1, "-->", "", NULL);
cmd_register("Longrightarrow",      0, 1, "==>", "", NULL);
#endif
cmd_register("lseil",               0, 1, "", "", NULL);
cmd_register("lvert",               0, 1, "", "amsmath", NULL);
cmd_register("lVert",               0, 1, "", "amsmath", NULL);
cmd_register("makebox",             1, 0, "", "", NULL);
cmd_register("makeindex",           0, 0, "", "makeidx", NULL);
cmd_register("maketitle",           0, 0, "", "", NULL);
cmd_register("mapsto",              0, 1, "", "", NULL);
cmd_register("marginpar",           1, 0, "", "mparhack", NULL);
#if (_slang_utf8_ok)
cmd_register("mathbb",              1, 1, "Characters with double strokes like ℝ", "amssymb", NULL);
#else
cmd_register("mathbb",              1, 1, "Characters with double strokes like |R", "amssymb", NULL);
#endif
cmd_register("mathbf",              1, 1, "Bold font in math mode", "", NULL);
cmd_register("mathcal",             1, 1, "Calligraphic letter in math mode", "", NULL);
cmd_register("mathfrak",            1, 1, "Fraktur font in math mode", "amssymb", NULL);
cmd_register("mathit",              1, 1, "Italic font in math mode (e.g. variable names, NULL)", "", NULL);
cmd_register("mathnormal",          1, 1, "Normal font in math mode", "", NULL);
cmd_register("mathrm",              1, 1, "Roman font in math mode", "", NULL);
cmd_register("mathscr",             1, 0, "", "mathrsfs", NULL);
cmd_register("mathsf",              1, 1, "Sans serif font in math mode", "", NULL);
cmd_register("mathtt",              1, 1, "Typewriter font in math mode", "", NULL);
cmd_register("max",                 0, 1, "mathematical function: maximum", "", NULL);
cmd_register("mbox",                1, 0, "", "", NULL);
cmd_register("mdseries",            0, 0, "", "", NULL);
cmd_register("mid",                 0, 1, "binary relations", "", &cmd_hook_math_not());
cmd_register("midrule",             0, 0, "A horizontal line with better spacing within a tabular"R, "booktabs", NULL);
cmd_register("min",                 0, 1, "mathematical function: minimum", "", NULL);
cmd_register("minisec",             1, 0, "", "", NULL);
cmd_register("mod",                 0, 1, "", "", NULL);
cmd_register("models",              0, 1, "binary relations", "", &cmd_hook_math_not());
cmd_register("mp",                  0, 1, "binary operator: minus plus", "", NULL);
cmd_register("mspace",              1, 1, "Space (like \\hspace) in math; with mu as math units", "amsmath", NULL);
cmd_register("mu",                  0, 1, "", "", &cmd_hook_math());
cmd_register("mult",                2, 1, "multiply the n-th row/column in a gmatrix by b", "gauss", NULL);
cmd_register("multicolumn",         3, 0, "", "", NULL);
cmd_register("nabla",               0, 1, "", "", NULL);
cmd_register("ncong",               0, 1, "short for \\not\\cong", "amssymb", NULL);
cmd_register("ne",                  0, 1, "short for \\not=", "", NULL);
#if (_slang_utf8_ok)
cmd_register("neg",                 0, 1, "Logical negation ¬", "", &cmd_hook_null());
#else
cmd_register("neg",                 0, 1, "Logical negation", "", &cmd_hook_null());
#endif
cmd_register("neq",                 0, 1, "binary relation: not equal", "", NULL);
cmd_register("newboolean",          1, 0, "Defines a new boolean register", "ifthen", NULL);
cmd_register("newcommand*",         2, 0, "", "", NULL);
cmd_register("newcounter",          1, 0, "", "", NULL);
cmd_register("newenvironment*",     3, 0, "", "", NULL);
cmd_register("newlength",           1, 0, "Create a new length register", "", NULL);
cmd_register("newline",             0, 0, "", "", NULL);
cmd_register("newpage",             0, 0, "", "", NULL);
cmd_register("newtheorem",          2, 0, "", "ntheorem", NULL);
cmd_register("nexists",             0, 1, "short for \\not\\exists", "amsmath", NULL);
cmd_register("ngg",                 0, 1, "short for \\not\\gg", "txfonts", NULL);
cmd_register("ni",                  0, 1, "binary relations: mirrowed \in"R, "", &cmd_hook_math_not());
cmd_register("nicefrac",            2, 1, "typeset fraction with a slash", "nicefrac", NULL);
cmd_register("nLeftarrow",          0, 1, "short for \\not\\Leftarrow", "amsmath", NULL);
cmd_register("nleftarrow",          0, 1, "short for \\not\\leftarrow", "amsmath", NULL);
cmd_register("nLeftrightarrow",     0, 1, "short for \\not\\Leftrightarrow", "amsmath", NULL);
cmd_register("nleftrightarrow",     0, 1, "short for \\not\\leftrightarrow", "amsmath", NULL);
cmd_register("nll",                 0, 1, "short for \\not\\ll", "txfonts", NULL);
cmd_register("nmid",                0, 1, "short for \\not\\mid", "amssymb", NULL);
cmd_register("nocite",              1, 0, "Include an item in the bibliography", "", &cmd_hook_bib_cite());
cmd_register("noindent",            0, 0, "", "", NULL);
cmd_register("nolimits",            0, 1, "", "", NULL);
cmd_register("nolinebreak",         0, 0, "", "", NULL);
cmd_register("nolinkurl",           1, 0, "Definition of \url from url.sty if hyperref is loaded"R, "hyperref", NULL);
cmd_register("nopagebreak",         0, 0, "", "", NULL);
cmd_register("normalfont",          0, 0, "", "", NULL);
cmd_register("not",                 0, 1, "binary relation", "", &cmd_hook_null());
cmd_register("notag",               0, 1, "State that an equation should not get a number"R, "", NULL);
cmd_register("notin",               0, 1, "short for \not\in"R, "", &cmd_hook_null());
cmd_register("notni",               0, 1, "short for \\not\\ni", "txfonts", NULL);
cmd_register("nparallel",           0, 1, "short for \\not\\parallel", "amsmath", NULL);
cmd_register("nRightarrow",         0, 1, "short for \\not\\Rightarrow", "amsmath", NULL);
cmd_register("nrightarrow",         0, 1, "short for \\not\\rightarrow", "amsmath", NULL);
cmd_register("nsim",                0, 1, "short for \\not\\sim", "amsmath", NULL);
cmd_register("nsubseteq",           0, 1, "short for \\not\\subseteq", "amsmath", NULL);
cmd_register("nsupseteq",           0, 1, "short for \\not\\supseteq", "amsmath", NULL);
cmd_register("nu",                  0, 1, "", "", &cmd_hook_math());
cmd_register("nvdash",              0, 1, "short for \\not\\vdash", "amsmath", NULL);
cmd_register("nvDash",              0, 1, "short for \\not\\vDash", "amsmath", NULL);
cmd_register("odot",                0, 1, "binary operator", "", NULL);
cmd_register("og",                  1, 0, "left french quoting sign <<", "babel,frenchb", &cmd_hook_goto_mark());
cmd_register("oint",                0, 1, "", "", NULL);
cmd_register("oldnum",              1, 0, "Old style numbers (Minuskelziffern)", "", NULL);
cmd_register("omega",               0, 1, "", "", &cmd_hook_math());
cmd_register("Omega",               0, 1, "", "", &cmd_hook_math());
cmd_register("ominus",              0, 1, "binary operator", "", NULL);
cmd_register("operatorname",        1, 1, "typesets a string as operator name", "amsmath", NULL);
cmd_register("oplus",               0, 1, "binary operator", "", NULL);
cmd_register("oslash",              0, 1, "binary operator", "", NULL);
cmd_register("otimes",              0, 1, "binary operator", "", NULL);
cmd_register("overbrace",           1, 1, "", "", NULL);
cmd_register("overleftarrow",       0, 1, "", "", NULL);
cmd_register("overline",            1, 1, "", "", NULL);
cmd_register("overrightarrow",      0, 1, "", "", NULL);
cmd_register("overset",             0, 1, "", "", NULL);
cmd_register("pagebreak",           0, 0, "", "", NULL);
cmd_register("pageref",             1, 0, "", "", &cmd_hook_ref());
cmd_register("pagestyle",           1, 0, "", "", NULL);
cmd_register("paragraph",           1, 0, "Starts a new paragraph", "", NULL);
cmd_register("parallel",            0, 1, "binary relations: a parallel b", "", &cmd_hook_math_not());
cmd_register("parbox",              2, 0, "", "", NULL);
cmd_register("parindent",           0, 0, "", "", NULL);
cmd_register("part",                1, 0, "Starts a new part of a book or report", "", NULL);
#if (_slang_utf8_ok)
cmd_register("partial",             0, 1, "Differentiation character ∂", "", NULL);
#else
cmd_register("partial",             0, 1, "Differentiation character", "", NULL);
#endif
cmd_register("pdfbookmark",         2, 0, "", "hyperref", NULL);
cmd_register("pdsetup",             1, 0, "Settings for powerdot presentations", "", NULL);
cmd_register("perp",                0, 1, "binary relations", "", NULL);
cmd_register("phantom",             1, 0, "", "", NULL);
cmd_register("phi",                 0, 1, "", "", &cmd_hook_math());
cmd_register("Phi",                 0, 1, "", "", &cmd_hook_math());
cmd_register("pi",                  0, 1, "", "", &cmd_hook_math());
cmd_register("Pi",                  0, 1, "", "", &cmd_hook_math());
cmd_register("pm",                  0, 1, "binary operator: plus minus", "", NULL);
cmd_register("pmod",                1, 1, "modulo; (mod X)", "", NULL);
cmd_register("Pr",                  0, 1, "mathematical function", "", NULL);
cmd_register("prec",                0, 1, "binary relations", "", NULL);
cmd_register("preceq",              0, 1, "binary relations", "", NULL);
cmd_register("printindex",          0, 0, "", "makeidx", NULL);
cmd_register("prod",                0, 1, "", "", NULL);
cmd_register("propto",              0, 1, "binary relations", "", NULL);
cmd_register("pscircle",            0, 0, "Draws a PSTricks circle", "pstricks", NULL);
cmd_register("psframe",             0, 0, "Draws a PSTricks box", "pstricks", NULL);
cmd_register("psi",                 0, 1, "", "", &cmd_hook_math());
cmd_register("psline",              0, 0, "Draws a PSTricks line", "pstricks", NULL);
cmd_register("rput",                1, 0, "Puts the argument in a PSTricks picture", "pstricks", NULL);
cmd_register("Psi",                 0, 1, "", "", &cmd_hook_math());
cmd_register("publishers",          1, 0, "Lowermost element in title of KOMA script documents", "", NULL);
cmd_register("qquad",               0, 0, "", "", NULL);
cmd_register("quad",                0, 0, "", "", NULL);
cmd_register("raggedleft",          0, 0, "Switches to right aligned text", "", NULL);
cmd_register("raggedright",         0, 0, "Switches to left aligned text", "", NULL);
cmd_register("RaggedRight",         0, 0, "Reimplementation of \raggedright from ragged2e"R, "ragged2e", NULL);
cmd_register("raisebox",            2, 0, "Moves the text upward -- {distance}{text}", "", NULL);
#if (_slang_utf8_ok)
cmd_register("rangle",              0, 1, "mathematical delimiter: 〉", "", NULL);
#else
cmd_register("rangle",              0, 1, "mathematical delimiter: >", "", NULL);
#endif
cmd_register("rceil",               0, 1, "mathematical delimiter", "", NULL);
cmd_register("Re",                  0, 1, "Real part of a number", "", NULL);
cmd_register("ref",                 1, 0, "", "", &cmd_hook_ref());
cmd_register("renewcommand*",       2, 0, "", "", NULL);
cmd_register("rfloor",              0, 1, "mathematical delimiter", "", NULL);
cmd_register("rgroup",              0, 1, "mathematical delimiter", "", NULL);
cmd_register("rho",                 0, 1, "", "", &cmd_hook_math());
cmd_register("right",               0, 1, "", "", NULL);
#if (_slang_utf8_ok)
cmd_register("rightarrow",          0, 1, "→", "", &cmd_hook_math_not());
cmd_register("Rightarrow",          0, 1, "⇒", "", &cmd_hook_math_not());
#else
cmd_register("rightarrow",          0, 1, "->", "", &cmd_hook_math_not());
cmd_register("Rightarrow",          0, 1, "=>", "", &cmd_hook_math_not());
#endif
cmd_register("rmfamily",            0, 0, "Sets a roman font", "", NULL);
cmd_register("rmoustache",          0, 1, "mathematical delimiter", "", NULL);
cmd_register("rowops",              0, 1, "switches in a gmatrix into row operation mode", "gauss", NULL);
cmd_register("rseil",               0, 1, "", "", NULL);
cmd_register("rule",                2, 0, "Draws a black rule; {width}{height}", "", NULL);
cmd_register("rvert",               0, 1, "", "amsmath", NULL);
cmd_register("rVert",               0, 1, "", "amsmath", NULL);
cmd_register("scriptstyle",         0, 1, "", "", NULL);
cmd_register("scriptscriptstyle",   0, 1, "", "", NULL);
cmd_register("scshape",             0, 0, "", "", NULL);
cmd_register("searrow",             0, 1, "south east arrow", "", NULL);
cmd_register("sec",                 0, 1, "mathematical function", "", NULL);
cmd_register("section",             1, 0, "Starts a new section", "", NULL);
cmd_register("see",                 1, 0, "", "", NULL);
cmd_register("setboolean",          2, 0, "Sets the value of a boolean register {register}{true|false}", "iffthen", NULL);
cmd_register("setcounter",          2, 0, "Sets the value of the counter 1st arg to 2nd arg", "", NULL);
cmd_register("setlength",           2, 0, "Sets the value of the length register #1 to #2", "", NULL);
cmd_register("setminus",            0, 1, "binary operator", "", NULL);
cmd_register("settowidth",          2, 0, "", "", NULL);
cmd_register("sffamily",            0, 0, "", "", NULL);
cmd_register("showhyphens",         1, 0, "Prints the hyphenpoints in the argument to the log file"R, "", NULL);
cmd_register("sigma",               0, 1, "", "", &cmd_hook_math());
cmd_register("Sigma",               0, 1, "", "", &cmd_hook_math());
cmd_register("sim",                 0, 1, "binary relations", "", &cmd_hook_math_not());
cmd_register("simeq",               0, 1, "binary relations", "", NULL);
cmd_register("sin",                 0, 1, "mathematical function", "", NULL);
cmd_register("sinh",                0, 1, "mathematical function", "", NULL);
cmd_register("slash",               1, 0, "A / with an hyphenpoint after it", "", &cmd_hook_goto_mark());
cmd_register("slashbox",            2, 0, "A big slash in the upper left corner of a tabular", "slashbox", NULL);
cmd_register("slshape",             0, 0, "", "", NULL);
cmd_register("smile",               0, 1, "binary relations", "", NULL);
cmd_register("sqcap",               0, 1, "binary operator", "", NULL);
cmd_register("sqcup",               0, 1, "binary operator", "", NULL);
cmd_register("sqrt",                1, 1, "", "", NULL);
cmd_register("sqsubset",            0, 1, "binary relations", "", NULL);
cmd_register("sqsubseteq",          0, 1, "binary relations", "", NULL);
cmd_register("sqsupset",            0, 1, "binary relations", "", NULL);
cmd_register("sqsupseteq",          0, 1, "binary relations", "", NULL);
cmd_register("stackrel",            2, 1, "Puts the first arg above the second \\stackrel{!}{=}", "", NULL);
cmd_register("star",                0, 1, "binary operator", "", NULL);
cmd_register("stepcounter",         1, 0, "Increments the counter by one", "", NULL);
cmd_register("subject",             1, 0, "Element in title of KOMA script documents", "", NULL);
cmd_register("subsection",          1, 0, "Starts a new subsection", "", NULL);
cmd_register("subset",              0, 1, "binary relation", "", &cmd_hook_null());
cmd_register("subseteq",            0, 1, "binary relation", "", &cmd_hook_math_not());
cmd_register("subsetneq",           0, 1, "", "", &cmd_hook_null());
cmd_register("substack",            1, 1, "Produce a multiline sub- or superscript", "amsmath", NULL);
cmd_register("subsubsection",       1, 0, "Starts a new subsubsection", "", NULL);
cmd_register("succ",                0, 1, "binary relations", "", NULL);
cmd_register("succeq",              0, 1, "binary relations", "", NULL);
cmd_register("suetterlin",          0, 0, "Switches to a suetterlin font", "suetterlin", NULL);
cmd_register("sum",                 0, 1, "", "", NULL);
cmd_register("sup",                 0, 1, "mathematical function", "", NULL);
cmd_register("supset",              0, 1, "binary relation", "", &cmd_hook_null());
cmd_register("supseteq",            0, 1, "binary relation", "", &cmd_hook_math_not());
cmd_register("supsetneq",           0, 1, "", "", &cmd_hook_null());
cmd_register("SVN",                 0, 0, "", "svn", NULL);
cmd_register("SVNDate",             0, 0, "", "svn", NULL);
cmd_register("SVNLastChangedRevision", 0, 0, "", "svn", NULL);
cmd_register("swap",                2, 1, "swap the row/column n and k in a gmatrix", "gauss", NULL);
cmd_register("tableofcontents",     0, 0, "", "", NULL);
cmd_register("tan",                 0, 1, "mathematical function", "", NULL);
cmd_register("tanh",                0, 1, "mathematical function", "", NULL);
cmd_register("tau",                 0, 1, "", "", &cmd_hook_math());
cmd_register("TeX",                 1, 0, "", "", &cmd_hook_goto_mark());
cmd_register("texorpdfstring",      2, 0, "", "hyperref", NULL);
cmd_register("text",                1, 1, "", "", NULL);
cmd_register("textasciigrave",      1, 0, "A ` character", "textcomp", &cmd_hook_goto_mark());
cmd_register("textasteriskcentered", 1, 0, "A * centered", "", &cmd_hook_goto_mark());
cmd_register("textbackslash",       1, 0, "", "", &cmd_hook_goto_mark());
cmd_register("textbf",              1, 0, "Boldface font", "", NULL);
cmd_register("textcolor",           2, 0, "", "xcolor", NULL);
cmd_register("textdownarrow",       1, 0, "A downarrow for text mode", "textcomp", &cmd_hook_goto_mark());
cmd_register("textit",              1, 0, "Italic font", "", NULL);
cmd_register("textleftarrow",       1, 0, "A leftarrow for text mode", "textcomp", &cmd_hook_goto_mark());
cmd_register("textmd",              1, 0, "Medium weight font (default, NULL)", "", NULL);
#if (_slang_utf8_ok)
cmd_register("textmu",              1, 0, "A µ for textmode", "textcomp", &cmd_hook_goto_mark());
#else
cmd_register("textmu",              1, 0, "A  for textmode", "textcomp", &cmd_hook_goto_mark());
#endif
cmd_register("textnormal",          1, 0, "Main document font", "", NULL);
cmd_register("textquotesingle",     1, 0, "A ' character", "textcomp", &cmd_hook_goto_mark());
cmd_register("textrightarrow",      1, 0, "A rightarrow for text mode", "textcomp", &cmd_hook_goto_mark());
cmd_register("textrm",              1, 0, "Roman font", "", NULL);
cmd_register("textsc",              1, 0, "Small caps", "", NULL);
cmd_register("textsf",              1, 0, "Sans serif", "", NULL);
cmd_register("textsl",              1, 0, "Slanted font", "", NULL);
cmd_register("textsuetterlin",      1, 0, "Suetterlin font", "suetterlin", NULL);
cmd_register("textsuperscript",     1, 0, "Superscript for text (like ^ in math mode)", "", NULL);
cmd_register("texttrademark",       1, 0, "The trademark symbol", "", &cmd_hook_goto_mark());
cmd_register("texttt",              1, 0, "Typewriter", "", NULL);
cmd_register("textup",              1, 0, "Upright font (default, NULL)", "", NULL);
cmd_register("textuparrow",         1, 0, "A uparrow for text mode", "textcomp", &cmd_hook_goto_mark());
cmd_register("textvisiblespace",    1, 0, "Sets `-´ at the baseline", "", &cmd_hook_goto_mark());
% cmd_register("textwidth",           0, 0, "", "", NULL);
cmd_register("thanks",              1, 0, "Footnote to authors in the title of document", "", NULL);
% cmd_register("thepage",             0, 0, "", "", NULL);
cmd_register("theta",               0, 1, "", "", &cmd_hook_math());
cmd_register("Theta",               0, 1, "", "", &cmd_hook_math());
cmd_register("tilde",               1, 1, "", "", NULL);
cmd_register("times",               0, 1, "binary operator", "", NULL);
cmd_register("title",               1, 0, "", "", NULL);
cmd_register("titlehead",           1, 0, "Uppermost element in title of KOMA script documents", "", NULL);
cmd_register("toprule",             0, 0, "A horizontal line with better spacing at top of a tabular"R, "booktabs", NULL);
cmd_register("triangleleft",        0, 1, "binary operator", "", NULL);
cmd_register("triangleright",       0, 1, "binary operator", "", NULL);
cmd_register("ttfamily",            0, 0, "", "", NULL);
cmd_register("twoheadleftarrow",    0, 1, "", "", NULL);
cmd_register("twoheadrightarrow",   0, 1, "", "", NULL);
cmd_register("underbrace",          1, 1, "", "", NULL);
cmd_register("underline",           1, 0, "", "", NULL);
cmd_register("unit",                1, 0, "", "units", NULL);
cmd_register("unitfrac",            2, 0, "", "units", NULL);
cmd_register("uparrow",             0, 1, "", "", NULL);
cmd_register("uplus",               0, 1, "binary operator", "", NULL);
cmd_register("upshape",             0, 0, "", "", NULL);
cmd_register("upsilon",             0, 1, "", "", &cmd_hook_math());
cmd_register("Upsilon",             0, 1, "", "", &cmd_hook_math());
cmd_register("url",                 1, 0, "", "hyperref", NULL);
cmd_register("varepsilon",          0, 0, "", "", &cmd_hook_math());
cmd_register("varointctrclockwise", 0, 1, "", "esint", NULL);
cmd_register("varphi",              0, 1, "", "", &cmd_hook_math());
cmd_register("vDash",               0, 1, "binary relations |=", "", &cmd_hook_math_not());
cmd_register("vdash",               0, 1, "binary relations |-", "", &cmd_hook_math_not());
cmd_register("vdots",               0, 1, "Vertical dots", "", NULL);
cmd_register("vec",                 1, 1, "set an arrow above the arguments", "", NULL);
#if (_slang_utf8_ok)
cmd_register("vee",                 0, 1, "binary operator: Logical or ∨", "", NULL);
#else
cmd_register("vee",                 0, 1, "binary operator: Logical or", "", NULL);
#endif
cmd_register("verb",                0, 0, "", "", NULL);
cmd_register("VerbatimInput",       1, 0, "Inserts the contents of a file as verbatim text", "fancyvrb", &cmd_hook_file_select());
cmd_register("vert",                0, 1, "the bar for absolut values: |-1| = 1", "", NULL);
cmd_register("Vert",                0, 1, "the bar for norm: ||x|| = x_1^2 + x_2^2", "", NULL);
cmd_register("vfill",               0, 0, "", "", NULL);
% cmd_register("vol",                 0, 0, "", "", NULL);
cmd_register("vspace",              1, 0, "Insert a vertical space", "", NULL);
#if (_slang_utf8_ok)
cmd_register("wedge",               0, 1, "binary operator: Logical and ∧", "", NULL);
#else
cmd_register("wedge",               0, 1, "binary operator: Logical and", "", NULL);
#endif
cmd_register("widehat",             1, 1, "", "", NULL);
cmd_register("widetilde",           1, 1, "", "", NULL);
cmd_register("wr",                  0, 1, "binary operator", "", NULL);
cmd_register("xi",                  0, 1, "", "", &cmd_hook_math());
cmd_register("Xi",                  0, 1, "", "", &cmd_hook_math());
cmd_register("xrightarrow",         1, 1, "", "amsmath", NULL);
cmd_register("xspace",              0, 0, "", "xspace", NULL);
cmd_register("zeta",                0, 1, "", "", &cmd_hook_math());
