/****************************************************************************
 *
 * Copyright (c) 1992, John Forkosh Associates.  All rights reserved.
 * --------------------------------------------------------------------------
 *
 * Functions:	--- WordPerfect control code processors ---
 *		wpfunc(tpfcb,inbuff,ifunc,membuff) Process single-byte ifunc.
 *		wpfixlen(tpfcb,inbuff,ifixlen,membuff) Process fixed-length
 *			multi-byte function beginning with ifixlen.
 *		wpvarlen(tpfcb,inbuff,ivarlen,membuff) Process variable-length
 *			multi-byte function beginning with ivarlen.
 *		--- Ancillary functions ---
 *		wppfxcpy(tpfcb) Copy WordPerfect prefix to output file.
 *
 * Notes:     o	See individual function headers.
 *
 * Source:	WPTOOLS.C
 *
 * --------------------------------------------------------------------------
 * Revision History:
 * 12/19/92	J.Forkosh	Installation.
 * 03/22/93	J.Forkosh	Revised for WP outmode only.
 *
 ***************************************************************************/

/* --------------------------------------------------------------------------
Headers and Static Data
-------------------------------------------------------------------------- */
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* --- application headers --- */
#include "texperf.h"


/****************************************************************************
 * Function:	wpfunc ( tpfcb, inbuff, ifunc, membuff )
 * Purpose:	Processes single-byte function ifunc (080x-0BFx).
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of TPFCB,
 *				TexPerfect file control block.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to process chars from the WordPerfect file,
 *				or containing an input buffer to be processed.
 *		ifunc (I)	integer containing single-byte function
 *				to be processed.
 *		membuff (O)	address of MEMBUFF in which string
 *				representing ifunc will be returned.
 * Returns:	(int)		new mode (PARMODE, MATHMODE, or DISPMATH)
 *				if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpfunc ( tpfcb, inbuff, ifunc, membuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
int	ifunc;
MEMBUFF	*membuff;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	funcmode = TOKENMODE;		/* init to token's mode (no change) */
unsigned char *buffer = membuff->buffer; /* ptr to beginning of buffer */
int	wpcodetblsz = tpfcb->wpcodetblsz, /* #entries in wpcodetbl[] */
	*wpcodetbl = tpfcb->wpcodetbl;	/* address of wpcodetbl */

/* --------------------------------------------------------------------------
Process single-byte function.
-------------------------------------------------------------------------- */
membuff->buffhead=membuff->bufftail=0;	/* initialize buffer indexes */
*buffer = '\000';			/* and initialize buffer */
switch ( ifunc )
  {
  case (int)HARDSPACE: /* --- hard space --- */
  case (int)HARDSPACE2:
    strcpy(buffer,			/* set hard space char */
      (tpfcb->texmode==PARMODE)?"~":"\\;");
    membuff->bufftail = strlen(buffer);	/* set buffer length */
    break; /* --- end-of-case HARDSPACE --- */
  case (int)HARDENDOFCENTER: /* --- end-of-center --- */
  case (int)HARDCRSOFTPG: /* --- carriage return --- */
  case (int)DORMHARDCR:
    (tpfcb->nhardcr)++;			/* count another hard cr */
    if ( (funcmode=tpfcb->texmode) != DISPMATH ) /*not using display math*/
      funcmode = PARMODE;		/* so reset to paragraph mode */
  default: /* --- copy function --- */
    *buffer=(unsigned char)(ifunc); buffer[1]='\000'; /* set buffer */
    membuff->bufftail = 1;		/* set buffer length */
    break; /* --- end-of-case default --- */
  } /* --- end-of-switch(ifunc) --- */

/* --------------------------------------------------------------------------
Back to caller with status of wpfunc() processing.
-------------------------------------------------------------------------- */
end_of_job:
  return ( funcmode );			/* mode, or 0=error */
} /* --- end-of-function wpfunc() --- */


/****************************************************************************
 * Function:	wpfixlen ( tpfcb, inbuff, ifixlen, membuff )
 * Purpose:	Processes fixed-length, multi-byte function
 *		ifixlen (0C0x-0CFx).
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of TPFCB,
 *				TexPerfect file control block.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to process chars from the WordPerfect file,
 *				or containing an input buffer to be processed.
 *		ifixlen (I)	integer containing fixed-length,
 *				multi-byte function to be processed.
 *				if ifixlen<=0, membuff contains input.
 *		membuff (I/O)	address of MEMBUFF in which string
 *				representing ifunc will be returned.
 * Returns:	(int)		new mode (PARMODE, MATHMODE, or DISPMATH)
 *				if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpfixlen ( tpfcb, inbuff, ifixlen, membuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
int	ifixlen;
MEMBUFF	*membuff;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	funcmode = TOKENMODE;		/* init to token's mode (no change) */
unsigned char *buffer = membuff->buffer; /* ptr to beginning of buffer */
int	funclen = 0;			/* function length */
int	wpcodetblsz = tpfcb->wpcodetblsz, /* #entries in wpcodetbl[] */
	*wpcodetbl = tpfcb->wpcodetbl;	/* address of wpcodetbl */
char	***wpchartbl = tpfcb->wpchartbl; /* extended character set tables */
int	msglevel = tpfcb->msglevel;	/* debugging message level */
int	nread = 0;			/* wpread() returns #items read */
int	isindent = 0;			/* true for indent function */
int	isread = 1;			/* true to read input from file */
/* --- size, in bytes, of multi-byte, fixed-length functions --- */
static	int ifunc0 = 192;		/* 192=0C0x */
static	int funcsize[] =		/* for 0C0x,0C1x,...,0CFx, -1=err */
  {  4,  9, 11,  3,  3,  5,  6,  7,	/* 0C0x,...,0C7x */
    -1, -1, -1, -1, -1, -1, -1, -1 } ;	/* 0C8x,...,0CFx aren't defined */

/* --------------------------------------------------------------------------
Check whether membuff contains func, and validate arguments passed by caller
-------------------------------------------------------------------------- */
if ( ifixlen <= 0 )			/* membuff is input and output */
  { ifixlen = (int)(buffer[0]);		/* function code in first byte */
    isread = 0; }			/* reset read flag */
/* --- check ifixlen argument passed by caller --- */
if ( ifixlen<=0 || ifixlen>=wpcodetblsz ) /* argument out-of-range */
  goto end_of_job;			/* return with error */
if ( wpcodetbl[ifixlen] != FIXLEN )	/* argument not a fixed-length func */
  goto end_of_job;			/* return with error */
funclen = funcsize[ifixlen-ifunc0];	/* total function length in bytes */
if ( funclen <= 0 )			/* undefined fixed-length function */
  goto end_of_job;			/* return with error */

/* --------------------------------------------------------------------------
Read fixed-length multi-byte function
-------------------------------------------------------------------------- */
if ( isread )
  {
  membuff->buffhead=membuff->bufftail=0; /* initialize buffer indexes */
  *buffer = '\000';			/* and initialize buffer */
  /* --- initialize fixed-byte function buffer --- */
  *buffer = (unsigned char)(ifixlen);	/* function code at start of buffer */
  /* --- read remainder of function --- */
  if ( (nread=wpread(tpfcb,inbuff,(void *)(buffer+1),1,funclen-1))
  !=   (funclen-1) )			/*failed to read correct #bytes */
    { printf("wpfixlen> wpread(...1,%d...) = %d\n", funclen-1,nread);
      goto end_of_job; }		/* return with error */
  } /* --- end-of-if(isread) --- */
membuff->bufftail = funclen;		/* buffer has funclen bytes */
membuff->buffchar = (unsigned char)(ifixlen); /* store function character */

/* --------------------------------------------------------------------------
Process multi-byte function.
-------------------------------------------------------------------------- */
switch ( ifixlen )
  {
  default: /* --- function returned unchanged --- */
    if ( msglevel >= 99 )		/* debugging output wanted */
      printf("wpfixlen> unidentified function code %d\n", ifixlen);
    break;
  /* ------------------------------------------------------------------------
  Extended character
  ------------------------------------------------------------------------ */
  case (int)EXTENDCHAR:
    { /* --- extended character --- */
    int	charcode = (int)(buffer[1]),	/* character code */
	charset = (int)(buffer[2]);	/* character set */
    char **wpsettbl = (char **)NULL;	/* table for character set */
    char *wpcharstr = (char *)NULL;	/* init xlation as undefined */
    char extundefined[32] = "\\undefined"; /* undefined extended char */
    char leadchar = '\000';		/* math mode/undefined char check */
    /* --- look up string representing extended character --- */
    if ( (charset>=0 && charset<=12 )	/* check set out-of-range */
    &&   (charcode>=0 && charcode<256) ) /* check code out-of-range */
      {
      wpsettbl = wpchartbl[charset];	/* character table for this set */
      if ( wpsettbl != (char **)NULL )	/* check for NULL table */
       if ( *wpsettbl != (char *)NULL )	/* then for first entry NULL */
	wpcharstr = wpsettbl[charcode]; /* extract requested character */
      } /* --- end-of-if(0<=charset<=12) --- */
    if ( wpcharstr != (char *)NULL )	/* character undefined */
      leadchar = *wpcharstr;		/* first char of xlated string */
    if ( leadchar == '\000' )		/* undefined char */
      {	sprintf(extundefined,"\\char{%d,%d}", /* format undefined WP char */
	  charset,charcode);		/* undefined char parameters */
	wpcharstr = extundefined; }	/* flip to undefined buffer */
    if ( leadchar == '\\' )		/* check for math mode char */
      if ( tpfcb->texmode != DISPMATH )	/* already in display math mode */
        funcmode = MATHMODE;		/*  set math mode if necessary */
    /* --- construct final string in function buffer --- */
    strcpy(buffer,wpcharstr);		/* extended character xlation */
    if ( leadchar == '\\' )		/* we need a terminator */
      strcat(buffer," ");		/* use blank space */
    membuff->bufftail = strlen(buffer);	/* set buffer length */
    if ( msglevel >= 99 )		/* debugging requested */
      printf("wpfixlen> extended char(%d,%d), xlation=%s\n",
      charset,charcode,buffer);
    } break; /* --- end-of-case EXTENDCHAR --- */
  /* ------------------------------------------------------------------------
  Indent and Tab, Center, Align, Left Margin Release
  ------------------------------------------------------------------------ */
  case (int)ENDOFINDENT:
    if ( funcmode != DISPMATH )		/* unless in display math mode */
      funcmode = PARMODE;		/* flip back to paragraph */
    break;
  case (int)INDENT:
  case (int)TABCENTER:
    break; /* --- end-of-case TABCENTER --- */
  /* ------------------------------------------------------------------------
  Attribute On
  ------------------------------------------------------------------------ */
  case (int)ATTRIBUTEON:
    { /* --- turn on attribute --- */
    int	attribute = ATTRIBUTE0 + (int)(buffer[1]); /*attribute code*/
    char *cmdstr = texcommand(tpfcb,attribute,0); /* look up attribute */
    *buffer = '\000';  membuff->bufftail = 0; /* init buffer and length */
    if ( cmdstr != (char *)NULL )	/* check that we xlated attribute */
      {	strcpy(buffer,cmdstr);		/* return attribute string */
	membuff->bufftail = strlen(buffer); } /* and length */
    switch ( attribute )		/* math mode check */
      { default: break;			/* no mode change */
        case SUBSCRIPT:
	case SUPERSCRIPT:
	  if ( funcmode != DISPMATH ) funcmode = MATHMODE;
	  break; }
    } break; /* --- end-of-case ATTRIBUTEON --- */
  /* ------------------------------------------------------------------------
  Attribute Off
  ------------------------------------------------------------------------ */
  case (int)ATTRIBUTEOFF:
    { /* --- turn off attribute --- */
    int	attribute = ATTRIBUTE0 + (int)(buffer[1]); /*attribute code*/
    char *cmdstr = texcommand(tpfcb,ATTRIBUTEOFF,0); /* one string fits all */
    *buffer = '\000';  membuff->bufftail = 0; /* init buffer and length */
    switch ( attribute )		/* special attribute check */
      { default: break;			/* just use normal off string */
        case REDLINE:
	case SMALLCAPS:
	  break; }
    if ( cmdstr != (char *)NULL )	/* check that we xlated attribute */
      {	strcpy(buffer,cmdstr);		/* return attribute string */
	membuff->bufftail = strlen(buffer); } /* and length */
    } break; /* --- end-of-case ATTRIBUTEOFF --- */
  } /* --- end-of-switch(ifixlen) --- */

/* --------------------------------------------------------------------------
Back to caller with status of wpfixlen() processing.
-------------------------------------------------------------------------- */
end_of_job:
  return ( funcmode );			/* mode, or 0=error */
} /* --- end-of-function wpfixlen() --- */


/****************************************************************************
 * Function:	wpvarlen ( tpfcb, inbuff, ivarlen, membuff )
 * Purpose:	Processes variable-length, multi-byte function
 *		ifixlen (0D0x-0FFx).
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of TPFCB,
 *				TexPerfect file control block.
 *		inbuff (I)	address of MEMBUFF normally containing NULL
 *				to process chars from the WordPerfect file,
 *				or containing an input buffer to be processed.
 *		ivarlen (I)	integer containing variable-length,
 *				multi-byte function to be processed.
 *				if ivarlen<0, membuff already contains
 *				function to be processed.
 *		membuff (I/O)	address of MEMBUFF in which string
 *				representing ifunc will be returned.
 *				contains function if ivarlen<0.
 * Returns:	(int)		new mode (PARMODE, MATHMODE, or DISPMATH)
 *				if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wpvarlen ( tpfcb, inbuff, ivarlen, membuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
int	ivarlen;
MEMBUFF	*membuff;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	funcmode = TOKENMODE;		/* init to token's mode (no change) */
unsigned char *buffer = membuff->buffer; /* ptr to beginning of buffer */
int	buffsize = membuff->buffsize;	/* size of buffer */
int	isubfunc = 0;			/* subfunction code */
word	funclen = (word)(0);		/* function length word */
int	wdsize = sizeof(word);		/* size of funclen, probably 2 */
int	msglevel = tpfcb->msglevel;	/* debugging message level */

/* --------------------------------------------------------------------------
Read variable-length multi-byte function
-------------------------------------------------------------------------- */
if ( ivarlen > 0 )
  {
  *buffer = '\000';  membuff->bufftail=0; /* initialize buffer */
  /* --- one-byte subfunction code should immediately follow ivarlen --- */
  if ( (isubfunc = wpgetc(tpfcb,inbuff)) /* subfunction should be next byte */
  ==   EOF )				/* failed to read next byte */
    { funcmode=0; goto end_of_job; }	/* return with error */
  /* --- length word should follow subfunction code --- */
  if ( wpread(tpfcb,inbuff,(void *)(&funclen),wdsize,1) /* read length */
  !=   1 )				/*failed to read correct #items (=1)*/
    { funcmode=0; goto end_of_job; }	/* return with error */
  if ( funclen<=0 || funclen+8>=buffsize ) /* invalid function length */
    { funcmode=0; goto end_of_job; }	/* return with error */
  /* --- initialize multi-byte function buffer --- */
  buffer[0] = (unsigned char)(ivarlen);	/* function code */
  buffer[1] = (unsigned char)(isubfunc); /* subfunction code */
  memcpy((void *)(buffer+2),(void *)(&funclen),wdsize); /* function length */
  /* --- read remainder of function --- */
  if ( wpread(tpfcb,inbuff,(void *)(buffer+2+wdsize),1,funclen) /*read func*/
  !=   funclen )			/*failed to read correct #bytes */
    { funcmode=0; goto end_of_job; }	/* return with error */
  } /* --- end-of-if(ivarlen>0) --- */

/* --------------------------------------------------------------------------
membuff already contains input function to be processed
-------------------------------------------------------------------------- */
if ( ivarlen <= 0 )
  {
  ivarlen = (int)(buffer[0]);		/* function code in first byte */
  isubfunc = (int)(buffer[1]);		/* followed by subfunction */
  memcpy((void *)(&funclen),(void *)(buffer+2),wdsize); /*and funclen*/
  } /* --- end-of-if(ivarlen<=0) --- */
membuff->buffchar = (unsigned char)(isubfunc); /* store subfunc character */
membuff->bufftail = 2+wdsize+funclen;	/* set bufftail past last byte */

/* --------------------------------------------------------------------------
Process variable-length, multi-byte function.
-------------------------------------------------------------------------- */
switch ( ivarlen )
  {
  default: /* --- function returned unchanged --- */
    if ( msglevel >= 99 )		/* debugging output wanted */
      printf("wpvarlen> unidentified function code %d\n", ivarlen);
    break;
  /* ------------------------------------------------------------------------
  Function Code 0D0x
  ------------------------------------------------------------------------ */
  case (int)FUNCD0:
    switch ( isubfunc )
      {
      default: /* --- not a 0D0-subfunction we care about --- */
	if ( msglevel >= 99 )		/* debugging output wanted */
	  printf("wpvarlen> unidentified D0-subfunction code %d\n",
	  isubfunc);
	break;
      /* --------------------------------------------------------------------
      Tab Set Function
      -------------------------------------------------------------------- */
      case (int)TABSET:
	break; /* --- end-of-case TABSET --- */
      } /* --- end-of-switch(isubfunc) --- */
    break; /* --- end-of-case FUNCD0 --- */
  /* ------------------------------------------------------------------------
  Function Code 0D1x
  ------------------------------------------------------------------------ */
  case (int)FUNCD1:
    switch ( isubfunc )
      {
      default: /* --- not a 0D1-subfunction we care about --- */
	if ( msglevel >= 99 )		/* debugging output wanted */
	  printf("wpvarlen> unidentified D1-subfunction code %d\n",
	  isubfunc);
	break;
      } /* --- end-of-switch(isubfunc) --- */
    break; /* --- end-of-case FUNCD1 --- */
  /* ------------------------------------------------------------------------
  Function Code 0D4x
  ------------------------------------------------------------------------ */
  case (int)FUNCD4:
    switch ( isubfunc )
      {
      default: /* --- not a 0D4-subfunction we care about --- */
	if ( msglevel >= 99 )		/* debugging output wanted */
	  printf("wpvarlen> unidentified D4-subfunction code %d\n",
	  isubfunc);
	break;
      /* --------------------------------------------------------------------
      End-of-Page Function
      -------------------------------------------------------------------- */
      case (int)ENDOFPAGE:
	{ /* --- end-of-page --- */
	unsigned char *function = buffer; /* address at start-of-func */
	int conditional = (int)(function[14]); /* conditional end-of-page? */
	unsigned char white[16] = " \000"; /* softpg becomes whitespace */
	unsigned char *cmdstr = white;	/* init assuming softpg */
	int wpchar = wpgetc(tpfcb,inbuff); /* check next char for hardpg */
	if ( wpchar == HARDPG )		/* found one, so user page break */
	   { wpungetc(tpfcb,inbuff,(int)(HARDCR));/*force HARDCR after eject*/
	     cmdstr = texcommand(tpfcb,EJECT,0); /* reset command string */
	     if ( funcmode != DISPMATH ) /* not in display math mode */
	       funcmode = PARMODE; }	/* so reset to paragraph mode */
	else wpungetc(tpfcb,inbuff,wpchar);/*unget non-HARDPG for processing*/
	strcpy(buffer,cmdstr);		/* return eject command in buffer */
	membuff->bufftail = strlen(buffer); /* and command length */
	} break; /* --- end-of-case ENDOFPAGE --- */
      } /* --- end-of-switch(isubfunc) --- */
    break; /* --- end-of-case FUNCD4 --- */
  /* ------------------------------------------------------------------------
  Function Code 0DAx
  ------------------------------------------------------------------------ */
  case (int)FUNCDA:
    switch ( isubfunc )
      {
      default: /* --- not a 0DA-subfunction we care about --- */
	if ( msglevel >= 99 )		/* debugging output wanted */
	  printf("wpvarlen> unidentified DA-subfunction code %d\n",
	  isubfunc);
	break;
      /* --------------------------------------------------------------------
      Equation
      -------------------------------------------------------------------- */
      case (int)EQUATION:
	{ /* --- equation --- */
	unsigned char *function = buffer; /* address at start-of-func */
	word caplen = (word)(0);	/* length of caption */
	unsigned char *eqnested = (unsigned char *)NULL; /* eqnested group */
	memcpy((void *)(&caplen),(void *)(function+119),wdsize);
	eqnested = function + (121+caplen); /* eqnested follows caption */
	if ( eqnested[0] == FUNCDF	/* found equation nested group */
	&&   eqnested[1] == EQNESTED )	/* and subfunction */
	  {
	  MEMBUFF *eqnbuff = tpmalloc(-1); /* nested buffer */
	  MEMBUFF *outbuff = tpmalloc(-1); /* output buffer */
	  eqnbuff->bufftail = membuff->bufftail - (int)(eqnested-function);
	  memcpy((void *)(eqnbuff->buffer),(void *)eqnested,
	  eqnbuff->bufftail);
	  funcmode = wpvarlen(tpfcb,inbuff,-1,eqnbuff);
	  wp2tex(tpfcb,eqnbuff,outbuff);
	  (outbuff->buffer)[(outbuff->bufftail)++] = HARDCR;
	  memcpy(membuff->buffer,outbuff->buffer,outbuff->bufftail);
	  membuff->bufftail = outbuff->bufftail;
	  (membuff->buffer)[membuff->bufftail] = '\000';
	  tpfree(eqnbuff); tpfree(outbuff);
	  funcmode = DISPMATH;
	  } /* --- end-of-if(eqnested[0]==FUNCDF) --- */
	} break; /* --- end-of-case EQUATION --- */
      } /* --- end-of-switch(isubfunc) --- */
    break; /* --- end-of-case FUNCDA --- */
  /* ------------------------------------------------------------------------
  Function Code 0DFx
  ------------------------------------------------------------------------ */
  case (int)FUNCDF:
    switch ( isubfunc )
      {
      default: /* --- not a 0DF-subfunction we care about --- */
	if ( msglevel >= 99 )		/* debugging output wanted */
	  printf("wpvarlen> unidentified DF-subfunction code %d\n",
	  isubfunc);
	break;
      /* --------------------------------------------------------------------
      Equation Nested Function
      -------------------------------------------------------------------- */
      case (int)EQNESTED:
	{ /* --- equation nested --- */
	unsigned char *function = buffer; /* address at start-of-func */
	unsigned char *eqntext = function+12; /* start of equation text */
	word flags = (word)(0);		/* alignment */
	word eqnlen = (word)(0);	/* length of equation text */
	memcpy((void *)(&flags),(void *)(function+4),wdsize);
	memcpy((void *)(&eqnlen),(void *)(function+10),wdsize);
	memcpy((void *)buffer,(void *)eqntext,eqnlen);
	buffer[eqnlen] = '\000';
	membuff->bufftail = eqnlen;
	} break; /* --- end-of-case EQNESTED --- */
      } /* --- end-of-switch(isubfunc) --- */
    break; /* --- end-of-case FUNCDF --- */
  } /* --- end-of-switch(ivarlen) --- */

/* --------------------------------------------------------------------------
Back to caller with status of wpvarlen() processing.
-------------------------------------------------------------------------- */
end_of_job:
  return ( funcmode );			/* mode, or 0=error */
} /* --- end-of-function wpvarlen() --- */


/****************************************************************************
 * Function:	wppfxcpy ( tpfcb )
 * Purpose:	Copy WordPerfect prefix to output file,
 *		setting document format flag.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	address of TPFCB,
 *				TexPerfect file control block.
 * Returns:	(int)		1 if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	wppfxcpy ( tpfcb )
TPFCB	*tpfcb;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	status = 0;			/* status returned to caller */
unsigned char *prefix = (unsigned char *)NULL; /* temporary prefix area */
FILE	*wpfp = tpfcb->wpdoc.fp;	/* ptr to WordPerfect input file */
FILE	*tpfp = tpfcb->tpfile.fp;	/* ptr to TeXPerfect output file */
dword docoffset = tpfcb->wpfilepref.docoffset; /* doc offset is prefix size */
int	blknum = 0;			/* #header index blocks read */
dword	prevblk = (dword)(0),		/* previous index block offset */
	nextblk = (dword)(sizeof(WPFILEPREF)); /*1st index after file prefix*/
dword	blksz = (dword)(sizeof(WPINDEXBLK)), /* size of index header */
	pktsz = (dword)(sizeof(WPPKTBLK)); /* size of index packet */
dword	pkt6len=(dword)(0), pkt6pos=(dword)(0); /* packet to be modified */
int	msglevel = tpfcb->msglevel;	/* output message level */

/* --------------------------------------------------------------------------
Malloc and read prefix area
-------------------------------------------------------------------------- */
/* --- malloc temporary buffer for entire document prefix --- */
if ( (prefix = (unsigned char *)(malloc(docoffset))) /*allocate prefix area*/
==   (unsigned char *)NULL )		/* failed to allocate */
  goto end_of_job;			/* so quit */
/* --- place file ptr at beginning of prefix area --- */
if ( fseek(wpfp,0L,SEEK_SET)		/* rewind */
!=   0 ) goto end_of_job;		/* quit if fseek() fails */
/* --- read entire prefix area --- */
if ( fread((void *)prefix,1,docoffset,wpfp) /* issue read */
!=   docoffset ) goto end_of_job;	/* quite if fread() fails */

/* --------------------------------------------------------------------------
Modify Packet Type 6 -- Document-Specific Flags and Information
-------------------------------------------------------------------------- */
if ( msglevel >= 9 )
  printf("wppfxcpy> prefsize=%d, blksz=%d, pktsz=%d\n",
  (int)nextblk,blksz,pktsz);
/* --- find packet 6 --- */
while ( nextblk>prevblk && nextblk<docoffset ) /* still have index blocks */
  {
  WPINDEXBLK *indexblk= (WPINDEXBLK *)(prefix+nextblk); /* ptr to index hdr */
  word nindex = indexblk->nindex;	/* #indexes in block (should be 5) */
  dword nextpkt = nextblk+blksz;	/* 1st index starts after header */
  blknum++;				/* count a header index block */
  if ( indexblk->pkttype != (word)(0xFFFBU) ) /* check packet type */
    { if ( msglevel >= 1 )		/* print error message */
	printf("wppfxcpy> illegal index block#%d, offset=%d, pkttype=%X\n",
	blknum,nextblk,indexblk->pkttype);
      goto end_of_job; }		/* quit due to bad index block */
  if ( msglevel >= 9 )			/* output wanted */
     printf("wppfxcpy> blk#%d/%X, offset/next=%ld,%ld nindex/size=%d,%d\n",
     blknum,indexblk->pkttype, (long)(nextblk),(long)(indexblk->nextblk),
     nindex,indexblk->indexsz);
  while ( --nindex > 0 )		/* don't count header index block */
    {
    WPPKTBLK *pktblk= (WPPKTBLK *)(prefix+nextpkt); /* ptr to index block */
    if ( msglevel >= 9 )		/* output wanted */
     printf("wppfxcpy> pktype=%X, offset/pkt=%ld,%ld pktlen=%ld\n",
     pktblk->pkttype, (long)(nextpkt),(long)(pktblk->pktoffset),
     (long)(pktblk->pktlen));
    nextpkt += pktsz;			/* offset to next index packet */
    if ( pktblk->pkttype != (word)(6) ) continue; /* we want pkt type 6 */
    pkt6len = pktblk->pktlen;		/* length of packet 6 */
    pkt6pos = pktblk->pktoffset;	/* offset to packet 6 */
    break;
    } /* --- end-of-while(--nindex>0) --- */
  nextblk = indexblk->nextblk;		/* walk index block chain */
  if ( pkt6pos > (dword)(0) ) break;	/* found packet 6 */
  } /* --- end-of-while(nextblk>prevblk) --- */
/* --- modify packet 6 --- */
if ( pkt6pos > (dword)(0) )		/* found packet 6 */
  {
  unsigned char *flag = prefix+pkt6pos;	/* first byte of packet 6 */
  setbit(*flag,3);			/* set bit 3 to format document */
  if ( msglevel >= 3 )			/* output wanted */
    printf("wppfxcpy> packet 6 offset=%ld, length=%ld\n",
    (long)(pkt6pos),(long)(pkt6len));
  } /* --- end-of-if(pkt6pos>0) --- */

/* --------------------------------------------------------------------------
Write prefix and return to caller with status
-------------------------------------------------------------------------- */
/* --- write entire prefix area --- */
if ( fwrite((void *)prefix,1,docoffset,tpfp) /* issue write */
==   docoffset ) status=1;		/* set ok status if write succeeds */
/* --- free prefix buffer and return --- */
end_of_job:
  if ( prefix != (unsigned char *)NULL ) free((void *)prefix); /*free prefix*/
  return ( status );			/* 1=success, or 0=error */
} /* --- end-of-function wppfxcpy() --- */
/* ----------------------- END-OF-FILE WPTOOLS.C ------------------------- */
