/*
 * SOURCE:  edit.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: main editing routines
 *
 * UPDATES: 08/15/1991 - major rewrite
 *          11/28/1991 - visible marking added
 *          12/17/1991 - bugs fixed
 *
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

extern struct IntuitionBase *IntuitionBase;
extern struct VideoBase *VideoBase;
struct Editor ed;

        /* if verbose is <= 0, no output is preformed */
int verbose = 1;
        /* if status==0 the status line won't be refreshed */
int status = 1;
        /* current contents of status line */
int contstat = MSG_NULL;

BOOL _redraw = FALSE,
     _remark = FALSE,
     _rewrite = FALSE,
     _restat = FALSE;
static struct Mark ClipMark;

unsigned char currline[EDIT_WIDTH+1];
extern struct MemoryHeader mh;

static char *CODEBASED _msg[] = {
    "No Alternate Buffer",
    "Line %d too long",
};

/* The keycode send in is checked like this:

    if keycode ends with 00
        search in Table 3
    else {
        search in Table 1
        if nothing found
            search in Table 2
    }
 */

static struct KeyTable CODEBASED kt1[] = {
    0x011b, "Esc",       e_cancel,
    0x0e08, "Backsp",    e_backsp,
    0x0e7f, "\02Backsp",     NULL,
    0x0f09, "Tab",          e_tab,
    0x1c0d, "Return",    e_return,
    0x4737, "\01\x37",  e_charins,
    0x4838, "\018",     e_charins,
    0x4939, "\019",     e_charins,
    0x4a2d, "\01-",     NULL,       // cut
    0x4b34, "\01\x34",  e_charins,
    0xff35, "\01\x35",  e_charins,
    0x4d36, "\01\x36",  e_charins,
    0x4e2b, "\01+",     NULL,       // copy
    0x4f31, "\01\x31",  e_charins,
    0x5032, "\01\x32",  e_charins,
    0x5133, "\01\x33",  e_charins,
    0x5230, "\01\x30",  e_charins,
    0x532e, "\01.",     e_charins,
    0x0000, "<soh>",    NULL,
    0x0001, "",        e_charins,
    0x0002, "",        e_charins,
    0x0003, "",        e_charins,
    0x0004, "",        e_charins,
    0x0005, "",        e_charins,
    0x0006, "",        e_charins,
    0x0007, "",        e_charins,
    0x0008, "",        e_charins,
    0x0009, "\x09",     e_tab,
    0x000a, "\x0a",     e_charins,
    0x000b, "",        e_charins,
    0x000c, "",        e_charins,
    0x000d, "\x0d",     e_charins,
    0x000e, "",        e_charins,
    0x000f, "",        e_charins,
    0x0010, "",        e_charins,
    0x0011, "",        e_charins,
    0x0012, "",        e_charins,
    0x0013, "",        e_charins,
    0x0014, "",        e_charins,
    0x0015, "",        e_charins,
    0x0016, "",        e_charins,
    0x0017, "",        e_charins,
    0x0018, "",        e_charins,
    0x0019, "",        e_charins,
    0x001a, "\x1a",     e_charins,
    0x001b, "",        e_charins,
    0x001c, "",        e_charins,
    0x001d, "",        e_charins,
    0x001e, "",        e_charins,
    0x001f, "",        e_charins,
    0x00ff, "\xff",     e_charins,

    0xffff
};


struct KeyTable CODEBASED kt2[] = {
    0xff01, "\02a",             NULL,
    0xff02, "\02b",         et_begin,
    0xff03, "\02c",             NULL,
    0xff04, "\02d",         et_index, // texindex
    0xff05, "\02e",           et_end,
    0xff06, "\02f",         e_format,
    0xff07, "\02g",             NULL, // goto line
    0xff08, "\02h",             NULL, // help
    0xff09, "\02i",      et_lastitem, // lastitem
    0xff0a, "\02j",         e_jpaste,
    0xff0b, "\02k",       e_markline,
    0xff0c, "\02l",          e_learn, // this is the learn key
    0xff0d, "\02m",        e_lineins,
    0xff0e, "\02n",       e_markword,
    0xff0f, "\02o",             NULL, // redo
    0xff10, "\02p",             NULL, // pack
    0xff11, "\02q",             NULL, // query s&r
    0xff12, "\02r",             NULL, // goto mark
    0xff13, "\02s",             NULL, // search
    0xff14, "\02t",       e_prevword,
    0xff15, "\02u",             NULL, // undo
    0xff16, "\02v",             NULL,
    0xff17, "\02w",         e_swapdm,
    0xff18, "\02x",        e_balance,
    0xff19, "\02y",        e_linedel,
    0xff1a, "\02z",       e_nextword,
    0xff1b, "\02[",             NULL,
    0xff1c, "\02|",             NULL,
    0xff1d, "\02]",             NULL,
    0xff1e, "\02^",             NULL,
    0xff1f, "\02_",             NULL,
    0xff20, "Space",      e_charins,
    0xff21, "!",          e_charins,
    0xff22, "\"",         e_charins,
    0xff23, "#",          e_charins,
    0xff24, "$",          e_charins,
    0xff25, "%",          e_charins,
    0xff26, "&",          e_charins,
    0xff27, "'",          e_charins,
    0xff28, "(",          e_charins,
    0xff29, ")",          e_charins,
    0xff2a, "*",          e_charins,
    0xff2b, "+",          e_charins,
    0xff2c, ",",          e_charins,
    0xff2d, "-",          e_charins,
    0xff2e, ".",          e_charins,
    0xff2f, "/",          e_charins,
    0xff30, "0",          e_charins,
    0xff31, "1",          e_charins,
    0xff32, "2",          e_charins,
    0xff33, "3",          e_charins,
    0xff34, "4",          e_charins,
    0xff35, "5",          e_charins,
    0xff36, "6",          e_charins,
    0xff37, "7",          e_charins,
    0xff38, "8",          e_charins,
    0xff39, "9",          e_charins,
    0xff3a, ":",          e_charins,
    0xff3b, ";",          e_charins,
    0xff3c, "<",          e_charins,
    0xff3d, "=",          e_charins,
    0xff3e, ">",          e_charins,
    0xff3f, "?",          e_charins,
    0xff40, "@",          e_charins,
    0xff41, "A",          e_charins,
    0xff42, "B",          e_charins,
    0xff43, "C",          e_charins,
    0xff44, "D",          e_charins,
    0xff45, "E",          e_charins,
    0xff46, "F",          e_charins,
    0xff47, "G",          e_charins,
    0xff48, "H",          e_charins,
    0xff49, "I",          e_charins,
    0xff4a, "J",          e_charins,
    0xff4b, "K",          e_charins,
    0xff4c, "L",          e_charins,
    0xff4d, "M",          e_charins,
    0xff4e, "N",          e_charins,
    0xff4f, "O",          e_charins,
    0xff50, "P",          e_charins,
    0xff51, "Q",          e_charins,
    0xff52, "R",          e_charins,
    0xff53, "S",          e_charins,
    0xff54, "T",          e_charins,
    0xff55, "U",          e_charins,
    0xff56, "V",          e_charins,
    0xff57, "W",          e_charins,
    0xff58, "X",          e_charins,
    0xff59, "Y",          e_charins,
    0xff5a, "Z",          e_charins,
    0xff5b, "[",          e_charins,
    0xff5c, "\\",         e_charins,
    0xff5d, "]",          e_charins,
    0xff5e, "^",          e_charins,
    0xff5f, "_",          e_charins,
    0xff60, "`",          e_charins,
    0xff61, "a",          e_charins,
    0xff62, "b",          e_charins,
    0xff63, "c",          e_charins,
    0xff64, "d",          e_charins,
    0xff65, "e",          e_charins,
    0xff66, "f",          e_charins,
    0xff67, "g",          e_charins,
    0xff68, "h",          e_charins,
    0xff69, "i",          e_charins,
    0xff6a, "j",          e_charins,
    0xff6b, "k",          e_charins,
    0xff6c, "l",          e_charins,
    0xff6d, "m",          e_charins,
    0xff6e, "n",          e_charins,
    0xff6f, "o",          e_charins,
    0xff70, "p",          e_charins,
    0xff71, "q",          e_charins,
    0xff72, "r",          e_charins,
    0xff73, "s",          e_charins,
    0xff74, "t",          e_charins,
    0xff75, "u",          e_charins,
    0xff76, "v",          e_charins,
    0xff77, "w",          e_charins,
    0xff78, "x",          e_charins,
    0xff79, "y",          e_charins,
    0xff7a, "z",          e_charins,
    0xff7b, "{",          e_charins,
    0xff7c, "|",          e_charins,
    0xff7d, "}",          e_charins,
    0xff7e, "~",          e_charins,
    0xff7f, "",          e_charins,
    0xff80, "",          e_charins,
    0xff81, "",          e_charins,
    0xff82, "",          e_charins,
    0xff83, "",          e_charins,
    0xff84, "",          e_charins,
    0xff85, "",          e_charins,
    0xff86, "",          e_charins,
    0xff87, "",          e_charins,
    0xff88, "",          e_charins,
    0xff89, "",          e_charins,
    0xff8a, "",          e_charins,
    0xff8b, "",          e_charins,
    0xff8c, "",          e_charins,
    0xff8d, "",          e_charins,
    0xff8e, "",          e_charins,
    0xff8f, "",          e_charins,
    0xff90, "",          e_charins,
    0xff91, "",          e_charins,
    0xff92, "",          e_charins,
    0xff93, "",          e_charins,
    0xff94, "",          e_charins,
    0xff95, "",          e_charins,
    0xff96, "",          e_charins,
    0xff97, "",          e_charins,
    0xff98, "",          e_charins,
    0xff99, "",          e_charins,
    0xff9a, "",          e_charins,
    0xff9b, "",          e_charins,
    0xff9c, "",          e_charins,
    0xff9d, "",          e_charins,
    0xff9e, "",          e_charins,
    0xff9f, "",          e_charins,
    0xffa0, "",          e_charins,
    0xffa1, "",          e_charins,
    0xffa2, "",          e_charins,
    0xffa3, "",          e_charins,
    0xffa4, "",          e_charins,
    0xffa5, "",          e_charins,
    0xffa6, "",          e_charins,
    0xffa7, "",          e_charins,
    0xffa8, "",          e_charins,
    0xffa9, "",          e_charins,
    0xffaa, "",          e_charins,
    0xffab, "",          e_charins,
    0xffac, "",          e_charins,
    0xffad, "",          e_charins,
    0xffae, "",          e_charins,
    0xffaf, "",          e_charins,
    0xffb0, "",          e_charins,
    0xffb1, "",          e_charins,
    0xffb2, "",          e_charins,
    0xffb3, "",          e_charins,
    0xffb4, "",          e_charins,
    0xffb5, "",          e_charins,
    0xffb6, "",          e_charins,
    0xffb7, "",          e_charins,
    0xffb8, "",          e_charins,
    0xffb9, "",          e_charins,
    0xffba, "",          e_charins,
    0xffbb, "",          e_charins,
    0xffbc, "",          e_charins,
    0xffbd, "",          e_charins,
    0xffbe, "",          e_charins,
    0xffbf, "",          e_charins,
    0xffc0, "",          e_charins,
    0xffc1, "",          e_charins,
    0xffc2, "",          e_charins,
    0xffc3, "",          e_charins,
    0xffc4, "",          e_charins,
    0xffc5, "",          e_charins,
    0xffc6, "",          e_charins,
    0xffc7, "",          e_charins,
    0xffc8, "",          e_charins,
    0xffc9, "",          e_charins,
    0xffca, "",          e_charins,
    0xffcb, "",          e_charins,
    0xffcc, "",          e_charins,
    0xffcd, "",          e_charins,
    0xffce, "",          e_charins,
    0xffcf, "",          e_charins,
    0xffd0, "",          e_charins,
    0xffd1, "",          e_charins,
    0xffd2, "",          e_charins,
    0xffd3, "",          e_charins,
    0xffd4, "",          e_charins,
    0xffd5, "",          e_charins,
    0xffd6, "",          e_charins,
    0xffd7, "",          e_charins,
    0xffd8, "",          e_charins,
    0xffd9, "",          e_charins,
    0xffda, "",          e_charins,
    0xffdb, "",          e_charins,
    0xffdc, "",          e_charins,
    0xffdd, "",          e_charins,
    0xffde, "",          e_charins,
    0xffdf, "",          e_charins,
    0xffe0, "",          e_charins,
    0xffe1, "",          e_charins,
    0xffe2, "",          e_charins,
    0xffe3, "",          e_charins,
    0xffe4, "",          e_charins,
    0xffe5, "",          e_charins,
    0xffe6, "",          e_charins,
    0xffe7, "",          e_charins,
    0xffe8, "",          e_charins,
    0xffe9, "",          e_charins,
    0xffea, "",          e_charins,
    0xffeb, "",          e_charins,
    0xffec, "",          e_charins,
    0xffed, "",          e_charins,
    0xffee, "",          e_charins,
    0xffef, "",          e_charins,
    0xfff0, "",          e_charins,
    0xfff1, "",          e_charins,
    0xfff2, "",          e_charins,
    0xfff3, "",          e_charins,
    0xfff4, "",          e_charins,
    0xfff5, "",          e_charins,
    0xfff6, "",          e_charins,
    0xfff7, "",          e_charins,
    0xfff8, "",          e_charins,
    0xfff9, "",          e_charins,
    0xfffa, "",          e_charins,
    0xfffb, "",          e_charins,
    0xfffc, "",          e_charins,
    0xfffd, "",          e_charins,
    0xfffe, "",          e_charins,

    0xffff
};

struct KeyTable CODEBASED kt3[] = {
    0x0000, "Break",            NULL,
    0x0300, "\02@",             NULL,
    0x0f00, "Back Tab",   e_backtab,
    0x1000, "\04q",             NULL,            // continue
    0x1100, "\04w",             NULL,
    0x1200, "\04e",             NULL,
    0x1300, "\04r",             NULL,
    0x1400, "\04t",             NULL,
    0x1500, "\04y",         e_remark,
    0x1600, "\04u",             NULL,
    0x1700, "\04i",             NULL,
    0x1800, "\04o",             NULL,
    0x1900, "\04p",             NULL,
    0x1e00, "\04a",             NULL,            // set mark
    0x1f00, "\04s",             NULL,
    0x2000, "\04d",             NULL,
    0x2100, "\04f",             NULL,
    0x2200, "\04g",             NULL,
    0x2300, "\04h",             NULL,
    0x2400, "\04j",             NULL,
    0x2500, "\04k",             NULL,
    0x2600, "\04l",             NULL,
    0x2c00, "\04z",             NULL,
    0x2d00, "\04x",             NULL,
    0x2e00, "\04c",             NULL,
    0x2f00, "\04v",             NULL,
    0x3000, "\04b",             NULL,
    0x3100, "\04n",             NULL,
    0x3200, "\04m",             NULL,
    0x3700, "\04Pause",         NULL,
    0x3800, "\04Break",         NULL,
    0x3a00, "Caps Lock",        NULL,
    0x3b00, "f1",               NULL,            // help
    0x3c00, "f2",               NULL,
    0x3d00, "f3",               NULL,
    0x3e00, "f4",               NULL,
    0x3f00, "f5",               e_launch,
    0x4000, "f6",               e_launch,
    0x4100, "f7",               e_launch,
    0x4200, "f8",               e_launch,
    0x4300, "f9",               NULL,
    0x4400, "f10",              NULL,
    0x4500, "\01 Lock",         NULL,
    0x4600, "Scroll Lock",      NULL,
    0x4700, "Home",             e_home,
    0x4800, "Up",               e_up,
    0x4900, "PgUp",             e_pgup,
    0x4b00, "Left",             e_left,
    0x4c00, "Goto",             NULL,
    0x4d00, "Right",            e_right,
    0x4f00, "End",              e_end,
    0x5000, "Down",             e_down,
    0x5100, "PgDown",           e_pgdown,
    0x5200, "\03Ins",           NULL,            // paste, insmode
    0x5300, "Del",              e_chardel,
    0x5400, "\03f1",            NULL,
    0x5500, "\03f2",            NULL,
    0x5600, "\03f3",            NULL,
    0x5700, "\03f4",            NULL,
    0x5800, "\03f5",            NULL,
    0x5900, "\03f6",            NULL,
    0x5a00, "\03f7",            NULL,
    0x5b00, "\03f8",            NULL,
    0x5c00, "\03f9",            NULL,
    0x5d00, "\03f10",           NULL,
    0x5e00, "\02f1",            NULL,
    0x5f00, "\02f2",            NULL,
    0x6000, "\02f3",            NULL,
    0x6100, "\02f4",            NULL,
    0x6200, "\02f5",            NULL,
    0x6300, "\02f6",            NULL,
    0x6400, "\02f7",            NULL,
    0x6500, "\02f8",            NULL,
    0x6600, "\02f9",            NULL,
    0x6700, "\02f10",           NULL,
    0x6800, "\04f1",            NULL,            // LaTeX
    0x6900, "\04f2",            NULL,            // PreView
    0x6a00, "\04f3",            NULL,            // Print
    0x6b00, "\04f4",            NULL,            // View Errors
    0x6c00, "\04f5",            NULL,            // View Error Log
    0x6d00, "\04f6",            NULL,            // Customize Menu
    0x6e00, "\04f7",            NULL,            //   -"-
    0x6f00, "\04f8",            NULL,            //   -"-
    0x7000, "\04f9",            NULL,            //   -"-
    0x7100, "\04f10",           NULL,            //   -"-
    0x7200, "\02PrtSc",         NULL,
    0x7300, "\02Left",      e_delbeg,
    0x7400, "\02Right",     e_delend,
    0x7500, "\02End",       e_endwin,
    0x7600, "\02PgDown",   e_endfile,
    0x7700, "\02Home",     e_homewin,
    0x7800, "\04\x31",          NULL,            // Customize Menu
    0x7900, "\04\x32",          NULL,            //   -"-
    0x7a00, "\04\x33",          NULL,            //   -"-
    0x7b00, "\04\x34",          NULL,            //   -"-
    0x7c00, "\04\x35",          NULL,            //   -"-
    0x7d00, "\04\x36",          NULL,            //   -"-
    0x7e00, "\04\x37",          NULL,            //   -"-
    0x7f00, "\04\x38",          NULL,            //   -"-
    0x8000, "\04\x39",          NULL,            //   -"-
    0x8100, "\04\x30",          NULL,            //   -"-
    0x8200, "\04-",             NULL,
    0x8300, "\04=",             NULL,
    0x8400, "\02PgUp",     e_begfile,
    0x8500, "f11",              NULL,
    0x8600, "f12",              NULL,
    0xffff
};

static void error_readonly( void );
static void line_too_long( unsigned int );
static void _writecurrline( void );
static void _checkmark( void );
static void _setcursorpos( void );
static int _ischar( int );
static int _setfile( unsigned );
static void _getfile( void );

/*
 *    Name: edit
 *  Return: TRUE if successful, FALSE if not
 * Purpose: main edit functions, calls other edit functions
 *
 *
 *
 * (c)M.Schollmeyer
 */
int edit( struct InputEvent *ie )
{
    int cy, pos;
    char buffer[SCREEN_WIDTH];
    int (*func)();

    static void softmark( void );
    static void mouse_scroll( struct InputEvent * );

    if( ie->key )
    {
        func = e_getkey( ie->key );
        if( func ) {
            m_beginundogroup();
            if( (*func)( ie->key, 1 ) )
                _restat = TRUE;
            m_endundogroup();
        } else {
            e_getkeyname( ie->key, buffer );
            PushTopic( HLP_UNASSIGNED );
            if( verbose > 0 ) {
                m_putmsg( MSG_ERROR, "%s is unassigned", buffer );
            }
        }
    }
    else if( ie->buttons & MBUTTON_LEFT )
    {
        if( ie->mousey >= EDITW_TOP && ie->mousey < EDITW_TOP + ed.EditHeight
            && ie->mousex >= EDITW_LEFT && ie->mousex < EDITW_LEFT + EDITW_WIDTH )
        {
            /* clicked inside edit window */
            cy = ie->mousey - EDITW_TOP + ed.WindowY;
            if( (unsigned int)cy != ed.CursorY )
            {
                if( !m_putcurrentline() )
                    return 1;

                ed.CursorY = (unsigned int)_MIN( mh.numlines, cy );
                m_getcurrentline();
            }
            ed.CursorX = ie->mousex - EDITW_LEFT + ed.WindowX;
            if( ed.Flags & MARKED ) {
                CLEARBIT( ed.Flags, MARKED );
                _remark = TRUE;
            }
            if( ie->buttons & DOUBLE_CLICK( MBUTTON_LEFT ) )
                e_markword( ie->key, 1 );
            else
                softmark();

            _restat = TRUE;

        } else if( ie->mousey >= EDITW_TOP + ed.EditHeight ) {

            if( contstat == MSG_TEXERR ) {
                // display error text
                HideMenu();
                viewerrors( VIEW_TEXT );
                ShowMenu();
            } else if( contstat == MSG_SEARCH ) {
                // continue searching
                searchstring( CONTSEARCH );
                return 1;
            } else if( contstat == MSG_NULL ) {

/*             klicked on status line.
 *             The status line appears like this:
 *  |*C:\PATH\FILENAME.TXT cursor=(123,45678) length=12345   mark=(123,45678)|
 *   ---------???---------+----------------19+-----------14+-----------------+
 *
 *      klicked on...         |  action taken
 *      ----------------------+--------------------------
 *      modified char         |  save file
 *      filename              |  next file
 *      cursorpos             |  swap dot & mark
 *      length                |  pack file
 *      markpos               |  goto mark
 *
 */
                if( ie->mousex == 0 ) {
                    HideMenu();
                    if( m_save( SAVE_QUERY ) )
                        e_savekoords();
                    ShowMenu();
                } else {
                    pos = strlen( ed.ShortFileName );
                    if( ie->mousex <= (unsigned int)pos ) {
                        e_savekoords();
                        e_next();
                    } else {
                        pos += 19;
                        if( ie->mousex <= (unsigned int)pos )
                            e_swapdm( ie->key, 1 );
                        else {
                            pos += 14;
                            if( ie->mousex <= (unsigned int)pos )
                                m_pack( 0, 1 );
                            else
                                e_gomark( ie->key, 1 );
                        }
                    }
                }
            }
        } else
            return 0;
    } else if( ie->buttons & MBUTTON_RIGHT ) {

        if( ie->mousey == VideoBase->ScreenHeight - 1 ) {
            if( contstat == MSG_NULL ) {
                pos = strlen( ed.ShortFileName );
                if( ie->mousex <= (unsigned int)pos ) {
                    /* clear the info for right button to prevent
                       from double click */
                    ie->buttons &= MBUTTON_RIGHT;
                    e_savekoords();
                    e_prev();
                    e_refresh();
                    return 1;
                }
            }
        } else {
            if( ! ( ie->buttons & DOUBLE_CLICK( MBUTTON_RIGHT ) ) ) {
                mouse_scroll( ie );
                return 1;
            }
        }
        return 0;
    } else
        return 0;

    e_refresh();
    return 1;
}


static void softmark( void ) {

    struct InputEvent *ie;
    unsigned int mx, my;
    int newy;
    char firstmove = TRUE;
    char wait = FALSE;

    unsigned long clock = 0L;

    ie = IntuitionBase->InputEvent;

    mx = ie->mousex;
    my = ie->mousey;

    if( !m_putcurrentline() )
        return;

    for( ;; ) {
        do {
            ie = GetInput();
            if( !(ie->buttons & MBUTTON_LEFT) )
                return;
        } while( mx == ie->mousex && my == ie->mousey );

        SETBIT( ed.Flags, MARKED|SHIFTMARKED );

        if( firstmove ) {
            ed.MarkX = ed.CursorX;
            ed.MarkY = ed.CursorY;
            firstmove = FALSE;
        }

        if( ie->mousey < EDITW_TOP ) {
            if( ed.CursorY ) {
                if( ed.WindowY ) {
                    newy = ed.WindowY-1;
                    --ed.WindowY;
                    _redraw = TRUE;
                    wait = TRUE;
                } else {
                    newy = 0;
                    my = ie->mousey;
                }
            } else if( mx == ie->mousex ) {
                continue;
            }
            ie = GetInput();
        } else if( ie->mousey >= EDITW_TOP + ed.EditHeight ) {
            if( (ed.WindowY+ed.EditHeight) <= mh.numlines ) {
                newy = ed.WindowY + ed.EditHeight;
                ++ed.WindowY;
                _redraw = TRUE;
                wait = TRUE;
            } else if( mx == ie->mousex )
                    continue;

            ie = GetInput();
        } else {
            if( ed.WindowY + ie->mousey - EDITW_TOP > mh.numlines )
                continue;

            if( wait ) {
                wait = FALSE;
                newy = ed.WindowY + ie->mousey - EDITW_TOP;
            } else
                newy = ed.CursorY + ie->mousey - my;
            my = ie->mousey;
        }

        ed.CursorX = ed.CursorX + ie->mousex - mx;
        mx = ie->mousex;

        newy = _MAX( 0, newy );

        if( newy != ed.CursorY ) {
            ed.CursorY = newy;
            m_getcurrentline();
        }

        _remark = _restat = TRUE;
        e_refresh();

        if( wait ) {
            while( clock == GetClock() );
            clock = GetClock();
        }
    }
}


static void mouse_scroll( struct InputEvent *ie ) {

    unsigned int oldmouse[2];
    unsigned int y;
    int scroll, newy;
    unsigned long clock = 0L;

    oldmouse[0] = ie->mousex;
    oldmouse[1] = ie->mousey;

    y = EDITW_TOP + ed.EditHeight/2;

    SetMouse( EDITW_WIDTH/2, y );

    while( ie->buttons & MBUTTON_RIGHT ) {
        scroll = EDITW_TOP + ed.EditHeight/2 - y;
        scroll = (scroll * scroll * _SGN(scroll) )/3;

        while( scroll ) {
            newy = ed.CursorY - scroll;
            if( (ed.WindowY+ed.EditHeight-2-scroll) < mh.numlines
                && newy >= 0
                && newy <= mh.numlines ) {

                _redraw = _remark = _restat = TRUE;
                e_goto( ed.WindowX, ed.WindowY - scroll, ed.CursorX,
                                            newy );
                e_refresh();

                while( clock >= GetClock()-1L );
                clock = GetClock();
                break;
            }
            /* scroll not possible, try a smaller scroll */
            scroll -= _SGN(scroll);
        }
        ie = GetInput();
        y = ie->mousey;
    }
    SetMouse( oldmouse[0], oldmouse[1] );
}



/*
 *    Name: e_getkey
 *  Return: Pointer to function for key
 * Purpose: Gets the function pointer according to key
 *
 *
 *
 * (c)M.Schollmeyer
 */
int (*e_getkey( unsigned int key ))() {

    struct KeyTable *kt;

    if( (char)key == 0 ) {

        for( kt = kt3; kt->Code != 0xffff; ++kt ) {
            if( kt->Code == key )
                goto done;
        }
        return NULL;

    } else {

        for( kt = kt1; kt->Code != 0xffff; ++kt )
        {
            if( kt->Code == key )
                goto done;
        }

        kt = &kt2[( 0xff & key ) - 1];
    }
done:
    return kt->Macro;
}


/*
 *    Name: e_getkeyname
 *  Return: void
 * Purpose: Gets the name for a key and stores it in name
 *
 *
 *
 * (c)M.Schollmeyer
 */
void e_getkeyname( unsigned int key, char *name ) {

    struct KeyTable *kt;
    unsigned char *cp;

    static char *CODEBASED intro[] = { "num", "ctrl+", "shift+", "alt+" };

    if( (char)key == 0 ) {

        for( kt = kt3; kt->Code != 0xffff; ++kt ) {
            if( kt->Code == key )
                goto done;
        }
        strcpy( name, "<unknown key>" );
        return;

    } else {

        for( kt = kt1; kt->Code != 0xffff; ++kt )
        {
            if( kt->Code == key )
                goto done;
        }

        kt = &kt2[( 0xff & key ) - 1];
    }

done:
    cp = kt->Name;
    while( 1 ) {
        if( *cp <= 5 ) {
            if( *cp == 0 ) break;
            strncpy( name, intro[*cp-1], 7 );
            while( *name ) ++name;
        } else
            *name++ = *cp;
        ++cp;
    }
    *name = '\0';
}



static void error_readonly( void ) {

    DoErrorBox( HLP_READONLY, "Read-only files may\nnot be edited." );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_cancel( int key, int c )
{
    verbose = 1;
    status = 1;

    ClearTopicStack();

    if( ed.Flags & TEMPFILE ) {
        CLEARBIT( ed.Flags, TEMPFILE );
        _getfile();
    }

    CLEARBIT( ed.Flags, MARKED );
    ShowMenu();

    _redraw = _remark = _restat = TRUE;
    contstat = MSG_NULL;
    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_drawflags( void ) {
#define FLAGCOL 63
/* the flagboard appears like: LWIOAPR
    l = Learn
    w = WordWrap
    i = Indent
    o = OverType
    a = AutoSave
    p = Pseudofile
    r = ReadOnly
*/

    char learnmode = GetLearnState();
    int col = FLAGCOL;

    VideoOut( CR_FLAGDISABLE, 0xb3, col++, 0 );
    VideoOut( ( learnmode >= MACROMODE ? CR_FLAG : (
      learnmode == LEARNMODE ? CR_FLAGHILITE : CR_FLAGDISABLE ) ), 'L', col++, 0 );
    VideoOut( ed.WordWrap         ? CR_FLAG : CR_FLAGDISABLE,'W', col++, 0 );
    VideoOut( ed.Flags & INDENT   ? CR_FLAG : CR_FLAGDISABLE,'I', col++, 0 );
    VideoOut( ed.Flags & OVERTYPE ? CR_FLAG : CR_FLAGDISABLE,'O', col++, 0 );
    VideoOut( ed.Flags & AUTOSAVE ? CR_FLAG : CR_FLAGDISABLE,'A', col++, 0 );
    VideoOut( ed.Flags & PSEUDOFILE ? CR_FLAG : CR_FLAGDISABLE,'P', col++, 0 );
    VideoOut( ed.Flags & (READONLY|TEMPFILE) ? CR_FLAG : CR_FLAGDISABLE,'R', col++, 0 );
    VideoOut( CR_FLAGDISABLE, 0xb3, col++, 0 );

    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_status( int flag )
{
    if( verbose > 0 || flag ) {

        /* status is set by m_putmsg() when is prints a message, so we have
           to check here if we can refresh
         */
        if( status || flag ) {

            /* Now we _restat, set it to FALSE */
            _restat = FALSE;
            /* It's necessary to set status to 1 since the status line now
               contains the MSG_NULL argument and this should be refreshed
               permanently */

            m_putmsg( MSG_NULL, "%c%s cursor=(%03d,%05d) length=%05d   mark=(%03d,%05d)"
              , ed.Flags & (MODIFIED|LINECHANGED) ? '*' : ' '
              , ed.ShortFileName
              , ed.CursorX, ed.CursorY
              , mh.numlines + 1
              , ed.MarkX
              , ed.MarkY );
        }
        _setcursorpos();
    }

    return 1;
}

static void _setcursorpos( void ) {

    SetCursorPos( ed.CursorX - ed.WindowX , ed.CursorY - ed.WindowY + EDITW_TOP );
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_charins( int key, int c ) {

    return e_strnins( &(unsigned char)key, 1 );
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_strins( unsigned char _huge *str ) {

    return e_strnins( str, _fstrlen( str ) );
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_strnins( unsigned char _huge *str, unsigned int len ) {

    unsigned char *curr, *cp, *end;
    BOOL putthatline = FALSE;
    int i, ret;

    static BOOL _ins_wrapit( unsigned int );
    static int _ins_overtype( unsigned char _huge *, unsigned int );

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    if( ed.Flags & MARKED ) {
        /* There is a marked region, delete it */
        e_deletemr();
        /* This may look a little bit stange. If there is a marked region,
           it has been deleted above, and the undo info has been stored.
           So we have to put the line after inserting the text from 'str',
           so this linechange belongs to the group in undo mem, and
           things can be undone using one call to m_undo. Otherwise we
           would need one call so restore that line and one to restore
           the deleted region */
        putthatline = TRUE;
    }

    curr = &(currline[ EDIT_WIDTH - 1]);
    for( end = currline; *end != '\0'; ++end );

    if( ed.Flags & OVERTYPE ) {
        i = _ins_overtype( str, len );
        if( putthatline )
            m_putcurrentline();
        return i;
    }

    /* not OVERTYPE mode */

    if( (unsigned int)(curr - end) < len ) {
        line_too_long( ed.CursorY );
        _redraw = TRUE;
        return 0;
    }

    --verbose;

    SETBIT( ed.Flags, LINECHANGED );

    if( ed.CursorY == ed.MarkY && ed.CursorX <= ed.MarkX ) {
        _remark = TRUE;
        ed.MarkX += len;
    }

    cp = &(currline[ ed.CursorX ]);
    ed.CursorX += len;

    if( ed.CursorX >= ed.WindowX + EDITW_WIDTH ) {
        ed.WindowX += len;
        _redraw = TRUE;
    }

    if( end < cp ) {
        while( end < cp ) *end++ = ' ';
        *end = '\0';
    }

    curr = end + len;
    while( curr > cp ) {
        *curr = *( curr - len );
        --curr;
    }

    while( len ) {
        *cp++ = *str++;
        --len;
    }

    ret = TRUE;
    if( ed.WordWrap && ed.CursorX + len > ed.WordWrap ) {
        if( !_ins_wrapit( len ) ) {
            ret = FALSE;
            goto done;
        }
    }

    if( putthatline ) {
        if( !m_putcurrentline() ) {
            ret = FALSE;
            goto done;
        }
    }

done:
    ++verbose;
    _rewrite = TRUE;

    return ret;
}


static BOOL _ins_wrapit( unsigned int len ) {

    unsigned char *cp;
    register unsigned int cx, offset;
    int wrapx;

    wrapx = 0;
    /* first, skip leading blanks */
    for( cx = 0, cp = currline; *cp == ' '; ++cx, ++cp );

    /* scan forwards for spaces */
    while( *cp ) {
        if( *cp == ' ' ) {
            /* a space has been found. Mark this as a
               suitable place to break */
            do {
                ++cp;
                ++cx;
            } while( *cp == ' ' );
            wrapx = cx;
        } else {
            ++cp;
            ++cx;
        }
        if( wrapx && cx >= ed.WordWrap )
            break;
    }

    if( wrapx ) {
        offset = ed.CursorX - wrapx;
        ed.CursorX = wrapx;
        _redraw = TRUE;
        if( e_return( 1, 1 ) )
            ed.CursorX += offset;
        else
            return FALSE;
    }
    return TRUE;
}


static int _ins_overtype( unsigned char _huge *str, unsigned int len ) {

    char *curr, *cp, *end;
    BOOL skippedend = FALSE;

    curr = &(currline[ EDIT_WIDTH - 1]);
    end = currline;
    while( *end != '\0' ) ++end;

    if( ed.CursorX + len >= EDIT_WIDTH ) {
        line_too_long( ed.CursorY );
        _redraw = TRUE;
        return 0;
    }

    cp = &(currline[ ed.CursorX ]);
    ed.CursorX += len;

    if( ed.CursorX >= ed.WindowX + EDITW_WIDTH ) {
        ed.WindowX += len;
        _redraw = TRUE;
    }

    if( end < cp ) {
        while( end < cp ) *end++ = ' ';
        *end = '\0';
    }

    while( len ) {
        if( *cp == '\0' )
            skippedend = 1;
        *cp++ = *str++;
        --len;
    }
    if( skippedend )
        *cp = '\0';

    SETBIT( ed.Flags, LINECHANGED );

    _rewrite = TRUE;
    return 1;
}



static void line_too_long( unsigned int line ) {

    DoErrorBox( HLP_LINETOOLONG, _msg[LINE_TOO_LONG], line );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_markword( int key, int c ) {

    unsigned char *cp, *end, *top;

    for( cp = currline; *cp; ++cp );
    end = cp;

    cp = &currline[ed.CursorX];
    if( cp > end ) return 0;
    if( !_ischar(*cp) ) return 0;
    /* scan backwards */

    for( top = cp; ( top > currline) && _ischar(top[-1]) ; --top );
    for( end = cp;   end[1]          && _ischar(end[ 1]) ; ++end );
    ed.MarkY = ed.CursorY;
    ed.MarkX = top - currline;
    ed.CursorX = end - currline + 1;

    SETBIT( ed.Flags, MARKED );
    _remark = _restat = TRUE;
    return 1;
}

static int _ischar( int c ) {

    return( isalpha(c) || strchr( "A", c ) );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_tab( int key, int c ) {

    int ret, i, len;
    struct MarkRegion *mr;
    char buffer[EDIT_WIDTH];

    static char CODEBASED spaces[] = "        ";

    if( ed.TabStops ) {

        if( ed.Flags & MARKED ) {
            if( !m_putcurrentline() ) return 0;
            mr = m_getmr();
            if( mr->flags & MR_MULTILINE ) {
                for( i = mr->UpperY; i < mr->LowerY; ++i ) {
                    len = m_getline( i, buffer );
                    if( len + ed.TabStops > EDIT_WIDTH ) {
                        line_too_long( i );
                        break;
                    }
                    strcpy( buffer, spaces );
                    len = m_getline( i, buffer+ed.TabStops );
                    m_putline( buffer, i, EDIT_WIDTH );
                }
                _redraw = TRUE;
                _remark = TRUE;
                m_getcurrentline();
                return TRUE;
            }
        }
        if( ed.Flags & OVERTYPE ) {
            ret = e_right( 0, ed.TabStops - (ed.CursorX % ed.TabStops ) );
            if( ed.Flags & MARKED ) {
                CLEARBIT( ed.Flags, MARKED );
                _redraw = TRUE;
            }
            return ret;
        }

        return e_strnins( spaces, ed.TabStops - (ed.CursorX % ed.TabStops ) );
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_backtab( int key, int c ) {

    int ret = 0, i, len;
    char buffer[EDIT_WIDTH];
    struct MarkRegion *mr;

    if( ed.TabStops ) {
        if( ed.Flags & MARKED ) {
            if( !m_putcurrentline() ) return 0;
            mr = m_getmr();
            if( mr->flags & MR_MULTILINE ) {
                for( i = mr->UpperY; i < mr->LowerY; ++i ) {
                    m_getline( i, buffer );
                    for( len = 0; buffer[len] == ' ' && len < ed.TabStops; ++len );
                    m_putline( buffer+len, i, EDIT_WIDTH );
                }
                _redraw = TRUE;
                _remark = TRUE;
                m_getcurrentline();
                return TRUE;
            }
        }

        CLEARBIT( IntuitionBase->InputEvent->shift, IES_SHIFT );
        ret = e_left( 0, (ed.CursorX-1) % ed.TabStops + 1 );
        if( ed.Flags & MARKED ) {
            CLEARBIT( ed.Flags, MARKED );
            _redraw = TRUE;
        }
    }

    return ret;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_return( int key, int c )
{
    unsigned char *cp, *ccp;
    unsigned int indent;
    int cx, wx;

    static unsigned int _ret_indent( int );

    if( ed.Flags & OVERTYPE ) {

        if( !m_putcurrentline() )
            return 0;

        cx = ed.CursorX;
        wx = ed.WindowX;
        ed.CursorX = ed.WindowX = 0;
        while( c > 0 ) {
            if( ed.CursorY == mh.numlines ) {
                ed.CursorX = cx;
                ed.WindowX = wx;
                return 0;
            }
            ++ed.CursorY;
            if( ed.CursorY >= ed.WindowY + ed.EditHeight ) {
                _redraw = TRUE;
                ++ed.WindowY;
            }
            --c;
        }
        m_getcurrentline();
        CLEARBIT( ed.Flags, MARKED );
        _redraw = TRUE;
        return 1;
    }

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    while( c > 0 ) {
        if( !m_putline( currline, ed.CursorY, ed.CursorX ) ) {
            return 0;
        }

        if( !m_insertline( ed.CursorY+1 ) ) {
            return 0;
        }

        if( ed.MarkY == ed.CursorY && ed.MarkX >= ed.CursorX ) {
            ++ed.MarkY;
            ed.MarkX -= ed.CursorX;
        }

        ++ed.CursorY;

        indent = _ret_indent( key );

        /* scan for end of line */
        for( ccp = currline; *ccp; ++ccp );

        cp = currline + indent;

        /* look if passed end of line */
        if( ccp - currline <= ed.CursorX )
            *cp = '\0';
        else
            strcpy( cp, currline + ed.CursorX );

        ed.CursorX = indent;
        if( ed.WindowX >= ed.CursorX )
            ed.WindowX = _MAX( 0, (int)ed.CursorX - 5 );


        if( ed.CursorY >= ed.WindowY + ed.EditHeight ) {
            ++ed.WindowY;
        }

        CLEARBIT( ed.Flags, MARKED );
        _redraw = TRUE;

        --c;
    }

    if(!m_putcurrentline())
        return 0;
    return 1;
}


static unsigned int _ret_indent( int key ) {

    unsigned char *cp;
    register unsigned int indent;
    int texindent = 0;
    int i;

    cp = currline;
    indent = 0;
    if( ed.Flags & INDENT ) {
        while( *cp == ' ' ) {
            ++indent;
            ++cp;
        }

        if( indent >= ed.CursorX || *cp == '\0' ) {
            /* no chars until cursorx */
            cp = currline;
            indent = 0;
        } else if( key ) {
            /* look for LaTeX indents */
            /* shouldn't do that for interal calls */
            if( !strncmp( cp, "\\item", 5 )
             || !strncmp( cp, "\\begin", 6 ) )
                texindent = 1;
            if( !strncmp( cp, "\\end", 4 ) )
                texindent = -1;
        }
    }

    if( texindent ) {
        if( ed.TabStops )
            i = ed.TabStops - indent % ed.TabStops;
        else
            i = 0;

        if( texindent == 1 ) {
            for( ; i; --i ) {
                *cp++ = ' ';
                ++indent;
            }
        } else if( texindent == -1 ) {
            for( ; i; --i )
                if( cp > currline ) {
                    --cp;
                    --indent;
                }
        }
    }

    return indent;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_chardel( int key, int c )
{
    unsigned char *curr, *end;
    BOOL shift = FALSE;
    char buffer[EDIT_WIDTH];
    unsigned int len;

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    if( (IntuitionBase->InputEvent->shift & IES_SHIFT) && (ed.Flags & MARKED) ) {
        return e_cut( 0, 1 );
    }

    if( ed.Flags & MARKED ) {
        e_deletemr();
        return 1;
    }

    if( IntuitionBase->InputEvent->shift & IES_SHIFT ) {
        shift = TRUE;
    }

    if( ed.MarkY == ed.CursorY ) {
        if( ed.MarkX > ed.CursorX ) {
            _remark = TRUE;
            --ed.MarkX;
        }
    }

    ed.Flags |= LINECHANGED;

    end = currline;
    while( *end ) ++end;

    while( *end == ' ' ) --end;
    *end = '\0';

    curr = &(currline[ ed.CursorX ]);

    if( curr >= end && shift ) {
        if( ed.CursorY >= mh.numlines )
            return 0;
        len = m_getline( ed.CursorY+1, buffer );
        if( len + ed.CursorX >= EDIT_WIDTH ) {
            line_too_long( ed.CursorY+1 );
            return 0;
        }
        while( end < curr ) *end++ = ' ';
        strcpy( end, buffer );
        m_deleteline( ed.CursorY+1, 1 );
        _redraw = TRUE;
    } else {
        while( curr < end ) {
            *curr = *( curr + 1 );
            ++curr;
        }
    }

    _rewrite = TRUE;

    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_linedel( int key, int c ) {

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    while( c > 0 ) {
        if( mh.numlines ) {
            m_deleteline( ed.CursorY, 1 );
            if( ed.CursorY > mh.numlines )
                ed.CursorY = mh.numlines;

            if( ed.CursorY < ed.WindowY ) {
                --ed.WindowY;
            }

            m_getcurrentline();
        } else {
            _redraw = TRUE;
            return 0;
        }
        --c;
    }
    CLEARBIT( ed.Flags, MARKED );
    _redraw = TRUE;
    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_lineins( int key, int c ) {

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    while( c > 0 ) {
        /* m_insertline() doesn't put the current line, because e_return()
           also uses it, and e_return() doen't put the whole line.
           So we have to do it now.
         */
        if( !m_putcurrentline() )
            return 0;
        if( !m_insertline( ed.CursorY ) ) {
            _redraw = TRUE;
            return 0;
        }
        currline[0] = 0;
        SETBIT( ed.Flags, LINECHANGED );
        --c;
    }
    /* just to make a group for undo... */
    m_putcurrentline();
    CLEARBIT( ed.Flags, MARKED );
    _redraw = TRUE;
    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_backsp( int key, int c )
{
    unsigned char *cp, *src;
    unsigned char buffer[EDIT_WIDTH];
    unsigned int cx, col;

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    if( e_left( 0, 1 ) )
    {
        return( e_chardel( 0, 1 ) );
    }

    if( ed.CursorY == 0 )
        return 0;

    col = cx = m_getline( ed.CursorY-1, buffer );
    cp = buffer + cx;

    if( col + strlen(currline) >= EDIT_WIDTH ) {
        line_too_long( ed.CursorY-1 );
        return 0;
    }

    ed.CursorX = cx;
    if( ed.MarkY == ed.CursorY ) {
        ed.MarkX = _MIN( ed.MarkX + cx, EDIT_WIDTH-1 );
        --ed.MarkY;
    }

    src = currline;
    while( *src )
        *cp++ = *src++;
    *cp++ = 0;
    strcpy( currline, buffer );

    m_deleteline( ed.CursorY, 1 );
    --ed.CursorY;
    if( ed.CursorY < ed.WindowY )
        --ed.WindowY;
    if( ed.CursorX < ed.WindowX || ed.CursorX >= ed.WindowX + EDITW_WIDTH ) {
        ed.WindowX = _MAX( 0, (int)(ed.CursorX)-SCROLLCOLS );
    }

    SETBIT( ed.Flags, LINECHANGED );
    CLEARBIT( ed.Flags, MARKED );

    _redraw = TRUE;

    m_putcurrentline();
    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
void e_refresh( void ) {

    if( _redraw )
        e_redraw();

    if( _remark ) {
        HideMouse();
        e_drawmr();
        ShowMouse();
    }

    if( _rewrite ) {
        HideMouse();
        _writecurrline();
        ShowMouse();
    }

    if( _restat )
        e_status( 0 );
}

/*
 *    Name: e_redraw
 *  Return: TRUE
 * Purpose: this routine refreshes the edit window.
 *
 *
 *
 * (c)M.Schollmeyer
 */
#define HIADR(c) (*(((unsigned char _far *)c)+1))
#define LOADR(c) (*(unsigned char _far *)c)

void e_redraw( void )
{
    register int       row;
    int                len,
                       endwin;
    unsigned char     *buf,
                       buffer[EDIT_WIDTH];
    unsigned int _far *vp;
    unsigned int       ep;

    /* don't display, if verbose negative */
    if( verbose < 1 )
        return;

    _redraw = FALSE;

    /* should remark if window is redrawed */
    _remark = TRUE;

    HideMouse();

    FP_SEG( vp ) = VideoBase->Segment;
    ep = FP_OFF( vp ) = 2 * ( EDITW_TOP * SCREEN_WIDTH + EDITW_LEFT );

    /* main loop: loop once for each line */
    for( row = ed.WindowY, endwin = ed.WindowY+ed.EditHeight; row < endwin; ++row ) {
        /* we shouldn't redraw the current line from the file buffer
           since 'currline' could contain something different */
        if( row == ed.CursorY ) {
            /* now the current line is displayed, draw it... */
            _writecurrline();
            vp += SCREEN_WIDTH;
            ep += 2 * SCREEN_WIDTH;
        } else {
            /* it's any other line, draw it */
            ep += 2 * SCREEN_WIDTH;
            if( row <= mh.numlines ) {
                /* we didn't pass the last line, y-way */
                len = m_getline( row, buffer ) - ed.WindowX;
                /* 'len' now represents the cols to be displayed */
                buf = buffer + ed.WindowX;

                for( ; FP_OFF(vp) < ep && len > 0; --len, ++vp ) {
                    LOADR( vp ) = *buf++;
                }
            }

            /* now something happened that let us skip this line.
               if we passed ed.numlines or the line ended without
               reaching the end of screen, clear to end of screen */

            for( ; FP_OFF(vp) < ep; ++vp ) {
                /* clear this character */
                LOADR( vp ) = 0;
            }
        }
    }

    ShowMouse();
}


/*
 *    Name: _writecurrline
 *  Return: void
 * Purpose: redraws the current line (represented by ed.CursorY)
 *          on screen.
 *
 *
 * (c)M.Schollmeyer
 */
static void _writecurrline()
{
    unsigned char     *cp;
    int                row, col, len;
    unsigned int _far *vp;

    if( verbose < 1 )
        return;

    _rewrite = FALSE;

    row = ed.CursorY - ed.WindowY;
    len = strlen( currline ) - ed.WindowX;
    cp = currline + ed.WindowX;

    FP_SEG( vp ) = VideoBase->Segment;
    FP_OFF( vp ) = 2 * ( (EDITW_TOP+row) * SCREEN_WIDTH + EDITW_LEFT );

    for( col = 0; col < EDITW_WIDTH && len > 0; ++col, --len, ++vp )
        LOADR( vp ) = *cp++;
    for( ; col < EDITW_WIDTH; ++col, ++vp )
        LOADR( vp ) = 0;
}



/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
void e_deletemr( void ) {

    struct MarkRegion *mr;
    char *cp, *dst;
    char buffer[EDIT_WIDTH];
    int i, len;

    if( ed.Flags & MARKED ) {
        mr = m_getmr();
        CLEARBIT( ed.Flags, MARKED );
        if( mr->flags & MR_MULTILINE ) {
            SETBIT( ed.Flags, LINECHANGED );
            /* delete first line */
            if( mr->flags & MR_FORWARDS ) {
                m_getline( mr->UpperY, buffer );
            } else {
                strncpy( buffer, currline, EDIT_WIDTH );
            }
            buffer[mr->UpperX] = 0;

            /* delete mid lines */
            i = mr->LowerY - mr->UpperY;
            if( i > 0 ) m_deleteline( mr->UpperY, i );

            /* delete last line */
            if( mr->flags & MR_FORWARDS ) {
                len = strlen( currline ) - mr->LowerX;
                if( len > 0 )
                    strncat( buffer, currline + mr->LowerX, len );
                strncpy( currline, buffer, EDIT_WIDTH );
            } else {
                strncpy( currline, buffer, EDIT_WIDTH );
                m_getline( mr->UpperY, buffer );
                len = strlen( buffer ) - mr->LowerX;
                if( len > 0 )
                    strncat( currline, buffer + mr->LowerX, len );
            }
        } else {
            /* not MR_MULTILINE */
            len = strlen( currline );
            if( mr->UpperX < len ) {
                len -= mr->LowerX;
                for( dst = currline + mr->UpperX,
                      cp = currline + mr->LowerX; len > 0; --len )
                    *dst++ = *cp++;
                *dst = 0;
            }
            SETBIT( ed.Flags, LINECHANGED );
            _remark = _rewrite = TRUE;
        }
        ed.CursorY = mr->UpperY;
        e_goto( ed.WindowX, ed.WindowY, mr->UpperX, mr->UpperY );
        ed.MarkX = ClipMark.CursorX = ed.CursorX;
        ed.MarkY = ClipMark.CursorY = ed.CursorY;
        ClipMark.WindowX = ed.WindowX;
        ClipMark.WindowY = ed.WindowY;

        _redraw = TRUE;
    }
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
void e_drawmr( void ) {

    unsigned int _far *vp;
    int col, row;
    unsigned int ep;
    unsigned int attr;

    static void _draw_marked( void );

    /* don't display, if verbose negative */
    if( verbose < 1 )
        return;

    attr = VideoBase->Palette[CR_TEXT];

    _remark = FALSE;
    if( ed.Flags & MARKED )
        _draw_marked();
    else {
        /* display marker */

        FP_SEG( vp ) = VideoBase->Segment;
        ep = FP_OFF( vp ) = 2 * ( EDITW_TOP * SCREEN_WIDTH + EDITW_LEFT );

        ep += 2 * EDITW_WIDTH * ed.EditHeight;
        for( ; ep > FP_OFF( vp ); ++vp )
                HIADR(vp) = attr;

        if( ed.MarkY >= ed.WindowY && ed.MarkY < ed.WindowY+ed.EditHeight
         && ed.MarkX >= ed.WindowX && ed.MarkX < ed.WindowX+EDITW_WIDTH )
            VideoAttr( CR_MARK, ed.MarkX-ed.WindowX, ed.MarkY-ed.WindowY+EDITW_TOP );
    }
}

// #pragma optimize( "elg", off )
static void _draw_marked( void ) {

    struct MarkRegion *mr;
    unsigned int _far *vp;
    register int col, row;
    unsigned int ep;
    unsigned int attr = VideoBase->Palette[CR_TEXT];
    unsigned int markattr = VideoBase->Palette[CR_MARK];

    mr = m_getmr();

    FP_SEG( vp ) = VideoBase->Segment;
    ep = FP_OFF( vp ) = 2 * ( EDITW_TOP * SCREEN_WIDTH + EDITW_LEFT );


    /* draw white lines until first line */
    for( row = ed.WindowY; row < mr->UpperY && row < ed.WindowY + ed.EditHeight; ++row )
        for( col = ed.WindowX; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
            HIADR( vp ) = attr;

    if( row == mr->UpperY ) {
        /* draw first line */
        for( col = ed.WindowX; col < mr->UpperX && col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
            HIADR(vp) = attr;
        if( mr->flags & MR_MULTILINE ) {
            for( ; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
                HIADR(vp) = markattr;
        } else {
            if( col == mr->UpperX && col < ed.WindowX + EDITW_WIDTH ) {
                HIADR(vp) = markattr;
                ++vp;
                ++col;
            }
            for( ; col < mr->LowerX && col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
                HIADR(vp) = markattr;
            for( ; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
                HIADR(vp) = attr;
        }
        ++row;
    }
    /* draw marked lines */
    for( ; row < mr->LowerY && row < ed.WindowY + ed.EditHeight; ++row )
        for( col = ed.WindowX; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
            HIADR(vp) = markattr;

    if( row == mr->LowerY && row < ed.WindowY + ed.EditHeight ) {
        /* draw last line */
        for( col = ed.WindowX; col < mr->LowerX; ++col, ++vp )
            HIADR(vp) = markattr;
        for( ; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
            HIADR(vp) = attr;
        ++row;
    }
    for( ; row < ed.WindowY + ed.EditHeight; ++row )
        for( col = ed.WindowX; col < ed.WindowX + EDITW_WIDTH; ++col, ++vp )
            HIADR(vp) = attr;
}
// #pragma optimize( "", on )




/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_close( unsigned int num ) {

    if( ed.Flags & (LINECHANGED|MODIFIED) && num == ed.CurrFile
        || ( ed.Flags & PSEUDOFILE ) ) {
        if( ! m_save( SAVE_NORMAL ) )
            return 0;
    }

    Delete( GetProfileString( PR_VIRTFILE(num), (char *)&profnull ) );
    ClearProfileData( PR_VIRTFILE( num ) );
    ClearProfileData( PR_FILEKOORDS( num ) );
    ClearProfileData( PR_FILENAME( num ) );

    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_load( char *file ) {

    int                  fptr,
                         i;
    unsigned char        sfilename[SCREEN_WIDTH],
                         s[SCREEN_WIDTH];
    BOOL                 ret;

    MakeShortName( sfilename, file, SCREEN_WIDTH-19 );

    fptr = Open( file, ACCESS_READ );
    if( fptr == -1 ) {
        i = GetExtError();
        if( i == 0x53 || i == 0x03 || i == 0x15 ) {
            PutError( i, file );
            return 0;
        }

        if( _createrequest( sfilename ) == CANCEL ) {
            return 0;
        }

        fptr = Create( file, 0 );
        if( fptr == -1 ) {
            DoErrorBox( HLP_NOCREAT, "'%s':Could not create file.", sfilename );
            return 0;
        }
    } else if( verbose > 0 ) {
        m_putmsg( MSG_STNDRD, "Loading %s...", sfilename );
        /* status line should be restored... */
        _restat = TRUE;
        /* ...in any case */
        status = 1;
    }

    ret = m_load( fptr );
    Close( fptr );

    if( ret == TRUE ) {
        /* loaded successfully */
        CLEARBIT( ed.Flags, (MODIFIED|LINECHANGED|PSEUDOFILE|MARKED) );
        if( GetFileAttr( file ) & ATTR_READONY )
            SETBIT( ed.Flags, READONLY );
        else
            CLEARBIT( ed.Flags, READONLY );

        e_drawflags();
        MakeShortName( ed.ShortFileName, file, SHORTFILENAMELEN );
        if( ed.DiskFileName ) free( ed.DiskFileName );
        ed.DiskFileName = HeapStrDup( file );

        m_getcurrentline();

        _remark = _redraw = TRUE;
        return 1;
    }

    e_new();
    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_setfile( unsigned slot ) {

    if( _setfile( slot ) )
        return TRUE;

    /* try anything valid */
    _getfile();
    return FALSE;
}

static int _setfile( unsigned slot ) {

    char *fname;
    int ret;

    if( slot >= MAXFILES )
        return 0;

    fname = GetProfileString( PR_FILENAME( slot ), NULL );
    if( !fname ) {
        /* There is no file name, cancel operation */
        return FALSE;
    }

    _redraw = _restat = TRUE;

    ret = m_loadvirtual( slot );
    if( ret == TRUE ) {
        /* reloaded successfully */
        e_getkoords();
        e_goto( ed.WindowX, ed.WindowY, ed.CursorX, ed.CursorY );

        return TRUE;
    } else if( ret == CANCEL ) {
        /* the edit buffer hasn't been changed yet,
           there's no need to restore the edit buffer, just return */
        return FALSE;
    }

    /* reload didn't work, and the edit buffer could have been modified */
    if( e_load( fname ) ) {
        if( slot != ed.CurrFile ) {
            ed.LastFile = ed.CurrFile;
            ed.CurrFile = slot;
        }
        e_getkoords();
        e_goto( ed.WindowX, ed.WindowY, ed.CursorX, ed.CursorY );

        m_savevirtual();
        /* it's possible that m_savevirtual() didn't work since
           m_loadvirtual() didn't work as well, but that stuff
           isn't really necessary for work */
        return TRUE;
    }

    /* This file cannot be loaded, remove it from the file list */
    ClearProfileData( PR_VIRTFILE( slot ) );
    ClearProfileData( PR_FILEKOORDS( slot ) );
    ClearProfileData( PR_FILENAME( slot ) );

    return FALSE;
}

static void _getfile( void ) {

    unsigned int i;

    /* First, try to load ed.CurrFile */
    if( _setfile( ed.CurrFile ) )
        goto done;
    else
        ClearProfileData( PR_FILENAME( ed.CurrFile ) );

    /* Try to load ed.LastFile */
    if( _setfile( ed.LastFile ) )
        goto done;
    else
        ClearProfileData( PR_FILENAME( ed.CurrFile ) );

    /* Try any file */
    for( i = 0; i < MAXFILES; ++i ) {
        if( _setfile( i ) )
            goto done;
        else
            ClearProfileData( PR_FILENAME( i ) );
    }

    /* Nothing could be loaded */
    ed.CurrFile = ed.LastFile = 0;
    e_new();

    strcpy( ed.ShortFileName, UNTITLEDNAME );
    SETBIT(ed.Flags, PSEUDOFILE);
    e_drawflags();
done:
    return;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_new( void )
{
    currline[0] = (char)0;
    mh.numlines = 0;
    mh.endlines = mh.base;
    ed.CursorX  = 0;
    ed.CursorY  = 0;
    ed.WindowX  = 0;
    ed.WindowY  = 0;
    ed.MarkX    = 0;
    ed.MarkY    = 0;

    m_putline(currline, 0, 0);
    ed.Flags &= ~(MODIFIED|LINECHANGED|MARKED);
    free( ed.DiskFileName );
    ed.DiskFileName = NULL;

    _redraw = _remark = _restat = TRUE;
    return( 1 );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_savekoords( void ) {

    if( ! (ed.Flags & PSEUDOFILE ) ) {
        PutProfileData( PR_FILEKOORDS( ed.CurrFile ), (char *)&(ed.CursorX), 12 );
        return 1;
    }
    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_getkoords( void ) {

    unsigned int *koord;

    _redraw = _remark = TRUE;

    if( koord = (unsigned int *)GetProfileData( PR_FILEKOORDS( ed.CurrFile ), NULL ) ) {
        ed.CursorX = koord[0];
        ed.CursorY = koord[1];
        ed.WindowX = koord[2];
        ed.WindowY = koord[3];
        ed.MarkX   = koord[4];
        ed.MarkY   = koord[5];

        return 1;
    } else {
        ed.CursorX = 0;
        ed.CursorY = 0;
        ed.WindowX = 0;
        ed.WindowY = 0;
        ed.MarkX   = 0;
        ed.MarkY   = 0;

        return 0;
    }
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_next( void ) {

    unsigned int i, end = MAXFILES-1;

    if( ed.Flags & PSEUDOFILE ) {
        /* search from 0...MAXFILES */
        i = 0;
    } else {
        /* search from (ed.CurrFile+1)...ed.CurrFile */
        i = ed.CurrFile + 1;
    }

    while( 1 ) {
        if( end == 0 ) {
            DoErrorBox( HLP_NOALTBUF, _msg[NO_ALTERNATE_BUF] );
            return 0;
        }
        --end;
        if( i >= MAXFILES ) i = 0;
        if( GetProfileString( PR_FILENAME(i), NULL ) ) {
            if( m_save( SAVE_NORMAL ) ) {
                if( e_setfile( i ) ) {
                    return 1;
                } else
                    ClearProfileData( PR_FILENAME( i ) );
            } else
                return 0;
        }
        ++i;
    }
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_prev( void ) {

    int i, end = MAXFILES-1;

    if( ed.Flags & PSEUDOFILE ) {
        /* search from MAXFILES...0 */
        i = MAXFILES-1;
    } else {
        /* search from (ed.CurrFile-1)...ed.CurrFile */
        if( ed.CurrFile )
            i = ed.CurrFile - 1;
        else
            i = -1;
    }

    while( 1 ) {
        if( end == 0 ) {
            DoErrorBox( HLP_NOALTBUF, _msg[NO_ALTERNATE_BUF] );
            return 0;
        }
        --end;
        if( i < 0 ) i = MAXFILES - 1;
        if( GetProfileString( PR_FILENAME(i), NULL ) ) {
            if( m_save( SAVE_NORMAL ) ) {
                if( e_setfile( i ) ) {
                    return 1;
                } else
                    ClearProfileData( PR_FILENAME( i ) );
            } else
                return 0;
        }
        --i;
    }
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_last( void ) {

    int slot;

    if( ed.Flags & PSEUDOFILE )
        slot = ed.CurrFile;
    else
        slot = ed.LastFile;

    if( m_save( SAVE_NORMAL ) ) {
        e_savekoords();
        e_setfile( slot );
    }
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_up( int key, int c )
{
    _checkmark();

    if( (int)ed.CursorY - c >= 0 )
    {
        if( !m_putcurrentline() )
            return 0;
        ed.CursorY -= c;

        if( ed.CursorY < ed.WindowY )
        {
            ed.WindowY -= c;
            m_getcurrentline();
            _redraw = TRUE;
        }
        else
            m_getcurrentline();

        return( 1 );
    }
    return( 0 );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_down( int key, int c )
{
    _checkmark();

    if( ed.CursorY + c <= mh.numlines )
    {
        if( !m_putcurrentline() )
            return 0;
        ed.CursorY += c;

        if( ed.CursorY >= ed.WindowY + ed.EditHeight )
        {
            ed.WindowY += c;
            m_getcurrentline();
            _redraw = TRUE;
        }
        else
            m_getcurrentline();

        return( 1 );
    }
    return( 0 );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_left( int key, int c )
{
    _checkmark();

    while( c > 0 ) {
        --c;
        if( ed.CursorX ) {
            --ed.CursorX;
            if( ed.CursorX < ed.WindowX ) {
                ed.WindowX = _MAX( 0, (int)(ed.CursorX)-SCROLLCOLS );
                _redraw = TRUE;
            }
        } else
            return( 0 );
    }

    return( 1 );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_right( int key, int c )
{
    _checkmark();

    while( c > 0 ) {
        --c;
        if( ed.CursorX < EDIT_WIDTH-1 ) {
            ++ed.CursorX;
            if( ed.CursorX >= ed.WindowX + EDITW_WIDTH ) {
                ed.WindowX += SCROLLCOLS;
                _redraw = TRUE;
            }
        } else
            return 0;
    }

    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_pgup( int key, int c )
{
    if( ed.CursorY ) {

        if( !m_putcurrentline() )
            return 0;

        _checkmark();

        if( ed.CursorY <= SCROLLINES ) {
            ed.CursorY = 0;
        } else {
            ed.CursorY -= SCROLLINES;
        }
        if( ed.WindowY <= SCROLLINES ) {
            ed.WindowY = 0;
        } else {
            ed.WindowY -= SCROLLINES;
        }

        m_getcurrentline();
        _redraw = TRUE;
        return 1;
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_pgdown( int key, int c )
{
    if( ed.CursorY < mh.numlines ) {

        if( !m_putcurrentline() )
            return 0;

        _checkmark();

        if( mh.numlines >= ed.EditHeight ) {
            ed.WindowY = _MIN( ed.WindowY + SCROLLINES, mh.numlines -
                ed.EditHeight + 1 );
        }
        ed.CursorY = _MIN( ed.CursorY + SCROLLINES, mh.numlines );

        m_getcurrentline();
        _redraw = TRUE;
        return 1;
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_begfile( int key, int c ) {

    if( ed.CursorY || ed.WindowY ) {

        if( !m_putcurrentline() )
            return 0;

        _checkmark();

        ed.CursorY = ed.WindowY = 0;

        m_getcurrentline();

        _redraw = TRUE;
        return 1;
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_endfile( int key, int c ) {

    if( ed.CursorY < mh.numlines ) {

        if( !m_putcurrentline() )
            return 0;

        _checkmark();

        ed.CursorY = mh.numlines;
        if( mh.numlines >= ed.EditHeight ) {
            ed.WindowY = mh.numlines + 1 - ed.EditHeight;
        }

        m_getcurrentline();

        _redraw = TRUE;
        return 1;
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_home( int key, int c )
{
    int top = 0;

    for( top = 0; currline[top] == ' '; ++top );

    if( ed.CursorX )
    {
        _checkmark();

        if( ed.CursorX <= top ) {
            /* go to very top of line */
            ed.CursorX = 0;
            if( ed.WindowX )
            {
                ed.WindowX = 0;
                _redraw = TRUE;
            }
            return( 1 );
        }
        /* fall thru */
    }
    /* go to first char */
    e_goto( ed.WindowX, ed.WindowY, top, ed.CursorY );
    return( 1 );
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_end( int key, int c )
{
    unsigned char *cp;
    unsigned int cx, oldcx;

    cp = currline;

    _checkmark();

    if( ! *cp ) {
        /* empty line */
        oldcx = ed.CursorX;
        ed.CursorX = ed.WindowX = 0;
        _redraw = TRUE;
            /* return oldcx, the contents of ed.CursorX before
               we changed it, so if the cursor moved, returning true
               otherwise false */
        return oldcx;
    }

    /* Scan for end of line */
    while( *cp != 0 )
        ++cp;

    --cp;
    while( *cp == ' ' ) --cp;
    cx = (int)( cp - &(currline[ 0 ]) + 1 );
    /* cx is the new cursor position. If we're coming from behind, insert
       a blank character. */
    if (cx+1 < ed.CursorX || cx == ed.CursorX)
        ++cx;

    ed.CursorX = cx;
    if( ed.WindowX + EDITW_WIDTH <= ed.CursorX
    || ed.WindowX > ed.CursorX )
    {
        /* ed.CursorX must be casted to (int), otherwise the
           second _MAX() expression is never negative and
           horizontal scrolling to the left doesn't work
         */
        ed.WindowX = _MAX( 0, (int)ed.CursorX - EDITW_WIDTH + 1 );
        _redraw = TRUE;
    }
    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_homewin( int key, int c ) {

    if( m_putcurrentline() ) {
        _checkmark();

        ed.CursorX = ed.WindowX;
        ed.CursorY = ed.WindowY;
        m_getcurrentline();
        return 1;
    }
    return 0;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_endwin( int key, int c ) {

    if( m_putcurrentline() ) {
        _checkmark();
        ed.CursorX = ed.WindowX;
        ed.CursorY = _MIN( mh.numlines, ed.WindowY + ed.EditHeight - 1 );
        m_getcurrentline();
        return 1;
    }
    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_nextword( int key, int c ) {

    char *cp;
    char buffer[EDIT_WIDTH];
    int i, j;

    /* look for next word in the current line */
    for( cp = currline, i = ed.CursorX; *cp && i; ++cp, --i );
    if( *cp ) {
        /* cp now points to current pos */
        /* skip leading blanks */
        for( i = ed.CursorX+1; *cp == ' '; ++cp, ++i );
        for( ++cp;
            *cp && (*cp != ' ') && (*cp != '{') && (*cp != '\\'); ++i, ++cp );
        for( ; *cp == ' '; ++cp, ++i );
        if( *cp )
            return e_goto( ed.WindowX, ed.WindowY, i, ed.CursorY );
    }

    /* nothing found in current line */
    for( i = ed.CursorY+1; i <= mh.numlines; ++i ) {
        m_getline( i, buffer );
        /* skip leading blanks */
        for( j = 0, cp = buffer; *cp == ' '; ++cp, ++j );
        if( *cp )
            return e_goto( ed.WindowX, ed.WindowY, j, i );
    }
    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_prevword( int key, int c ) {

    char *cp;
    char buffer[EDIT_WIDTH];
    int i, j;

    /* look for prev word in the current line */
    for( cp = currline, i = 0; *cp && i<ed.CursorX; ++cp, ++i );
    if( cp > currline ) {
        /* cp now points to current pos */
        for( --cp,--i;cp > currline && (*cp != ' ') && (*cp != '}'); --i, --cp ) {
            if( *cp == '\\' ) {
                --cp;
                --i;
                break;
            }
        }
        /* skip trailing blanks */
        for( ; *cp == ' ' && cp >= currline; --cp, --i );
        if( cp > currline )
            return e_goto( ed.WindowX, ed.WindowY, i, ed.CursorY );
    }

    /* nothing found in current line */
    for( i = ed.CursorY-1; i >= 0 ; --i ) {
        m_getline( i, buffer );
        /* skip trailing blanks */
        for( j = 0, cp = buffer; *cp; ++cp, ++j );
        --cp;
        --j;
        while( (cp > buffer) && (*cp == ' ') ) --cp;
        if( cp > buffer )
            return e_goto( ed.WindowX, ed.WindowY, j, i );
    }
    return 0;
}


static void _checkmark( void ) {

    if( IntuitionBase->InputEvent->shift & IES_SHIFT ) {
        if( ! (ed.Flags & MARKED) ) {
            e_setmark( 0, 0 );
            SETBIT( ed.Flags, SHIFTMARKED );
        }
        _remark = TRUE;
    } else if( ed.Flags & SHIFTMARKED ) {
        CLEARBIT( ed.Flags, MARKED|SHIFTMARKED );
        _remark = TRUE;
    } else if( ed.Flags & MARKED ) {
        _remark = TRUE;
    }
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_goto( int wx, int wy, int cx, int cy )
{
    if( !m_putcurrentline() )
        return 0;

    if( cy < 0 )
    {
        cy = 0;
    }
    else if( cy > mh.numlines )
    {
        cy = mh.numlines;
    }
    if( wy < cy - ed.EditHeight + 1 || wy > cy )
    {
        _redraw = TRUE;
        wy = cy - ed.EditHeight/2;
    }
    if( wy < 0 )
    {
        _redraw = TRUE;
        wy = 0;
    }
    if( cx >= EDIT_WIDTH )
    {
        cx = 0;
    }
    if( wx < cx - EDITW_WIDTH + 1 || wx > cx )
    {
        _redraw = TRUE;
        wx = cx - EDITW_WIDTH/2;
    }
    if( wx < 0 )
    {
        _redraw = TRUE;
        wx = 0;
    }

    ed.CursorX = cx;
    ed.CursorY = cy;
    ed.WindowX = wx;
    ed.WindowY = wy;

        /* because all routines using e_load() are calling e_goto()
           after that, we can adjust MarkY here...
         */
    if( ed.MarkY > mh.numlines ) {
        _remark = TRUE;
        ed.MarkY = mh.numlines;
    }

    m_getcurrentline();

    _restat = TRUE;
    if( ed.Flags & MARKED )
        _remark = TRUE;

    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
// #pragma optimize( "elg", off )
int e_search( unsigned char *src, unsigned int flags, struct MarkRegion *mr )
{
    unsigned int line;
    int cc;
    int len, i, linelen, srclen, row, col;
    unsigned char *buf, *scan;

    static int _nstrcmp( unsigned char *, unsigned char *, unsigned int );

    if( flags & SEARCH_REGULAR )
        return 0;       /* regular expressions not implemented yet */

    if( !m_putcurrentline() )
        return 0;

    buf = AllocHeap( EDIT_WIDTH );
    if( buf == NULL )
        return 0;

    line = ed.CursorY;
    linelen = len = m_getline( line, buf );

    scan = buf;
    if( flags & SEARCH_PLUS ) {
        if( len > ed.CursorX ) {
            /* not passed end of line */
            /* set pointer to top of line plus cursorpos plus one
               to search for next apperance */
            scan += ed.CursorX + 1;
            len -= ed.CursorX + 1;
            if( flags & SEARCH_QUERY ) {
                --scan;
                ++len;
            }
        } else {
            /* passed end of line, skip this line */
            len = -1;
        }
    } else {
        /* reverse search */
        len = _MIN( len, ed.CursorX );
        if( ed.CursorX )
            scan += len - 1;
        else
            len = -1;
    }

    if( verbose > 0 ) {
        i = m_putmsg( MSG_SEARCH, "%s %s for ",
            flags & SEARCH_QUERY ? "Query s&r" : "Search",
            flags & SEARCH_PLUS ? "down" : "up" );

        m_putxmsg( CR_STATUSHILITE, i, "%-.*s", EDIT_WIDTH-i, src );

        status = 0;
    }

    srclen = strlen( src );

    while( 1 ) {
        if( ScanKeybrd( BK_ESC ) ) goto done;

        if( line % 50 == 0 ) {
            m_putxmsg( CR_STATUS, VideoBase->ScreenWidth-7, "(%d) ", line );
        }

        if( flags & SEARCH_PLUS && line > mr->LowerY ) goto done;
        else if( line < mr->UpperY ) goto done;

        while( len >= 0 ) {
            if( flags & SEARCH_PLUS ) {
                if( line == mr->LowerY ) {
                    if( linelen-len+srclen > mr->LowerX ) goto done;
                }
            } else {
                if( line == mr->UpperY ) {
                    if( len < mr->UpperX ) goto done;
                }
            }

            if( _nstrcmp( scan, src, flags & SEARCH_CASE ) == 0 ) {
                /* search string found */
                ed.CursorX = scan - buf;
                ed.CursorY = line;
                cc = ed.CursorX + strlen( src );
                if( ed.WindowX + EDITW_WIDTH < cc ) {
                    /* right of string right out of window */
                    /* first, try to set to end of found string */
                    ed.WindowX = cc - EDITW_WIDTH;
                }
                if( ed.WindowX > ed.CursorX ) {
                    /* left of string left out of window */
                    /* or string longer than EDIT_WIDTH */
                    ed.WindowX = ed.CursorX;
                }
                if( ed.CursorY >= ed.WindowY + ed.EditHeight
                 || ed.CursorY < ed.WindowY )
                    ed.WindowY = _MAX( 0, (int)ed.CursorY - ed.EditHeight/2 );
                m_getcurrentline();
                CLEARBIT( ed.Flags, MARKED );

                _redraw = _remark = TRUE;

                if( verbose > 0 ) {
                    e_refresh();
                    _setcursorpos();
                    for( row = ed.CursorY-ed.WindowY+EDITW_TOP, col = ed.CursorX - ed.WindowX,
                        i = strlen( src ); i; --i, ++col ) {
                            if( col >= 0 && col < EDITW_WIDTH )
                                VideoAttr( CR_TEXTHILITE, col, row );
                    }
                    _remark = TRUE;
                }

                free( buf );
                m_putxmsg( CR_STATUS, VideoBase->ScreenWidth-7, "(%d) ", line );
                return 1;
            }

            if( flags & SEARCH_PLUS )
                ++scan;
            else
                --scan;

            --len;
        }
        /* nothing found in this line */
        if( flags & SEARCH_PLUS ) {
            ++line;
            if( line > mh.numlines ) break;
            linelen = len = m_getline( line, buf );
            scan = buf;
        } else {
            if( line == 0 ) break;
            --line;
            linelen = len = m_getline( line, buf );
            scan = buf + len;
        }
    }

done:
    free( buf );
    _redraw = _remark = TRUE;
    if( !(flags & SEARCH_FILES) )
        e_refresh();
    return 0;
}
// #pragma optimize( "", on )


static int _nstrcmp( unsigned char *hp, unsigned char *np, unsigned int noignorecase ) {

    if( noignorecase ) {
        while( *hp == *np ) {
            ++hp;
            ++np;
            if( *np == '\0' )
                return 0;
        }
        return 1;
    } else {
        while( tolower(*hp) == tolower(*np) ) {
            ++hp;
            ++np;
            if( *np == '\0' )
                return 0;
        }
        return 1;
    }
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
// #pragma optimize( "elg", off )
int e_replace( int num, unsigned char *by, unsigned int reason ) {

    unsigned char *cp, *end;
    unsigned int editorwrap;
    int i, rcx = strlen( by );

    /* we will change this line, so set the flag */
    SETBIT( ed.Flags, LINECHANGED );

    /* get base pointer */
    cp = currline;
    i = 0;
    /* scan until end of line */
    while( *cp ) {
        ++i;
        ++cp;
    }
    /* save end pointer */
    end = cp;

    if( i <= ed.CursorX ) {
        /* the cursor is behind end of line,
           just insert the replace string */
        editorwrap = ed.WordWrap;
        ed.WordWrap = 0;
        i = e_strins( by );
        ed.WordWrap = editorwrap;
        return i;
    }

    /* get pointer to start */
    cp = &(currline[ed.CursorX]);
    if( *by == '\0' )
        goto byempty;

    while( num ) {
        *cp++ = *by++;
        --num;
        if( *by == '\0' ) {
byempty:
            /* end of replace string, kill last chars */
            do {
                *cp = *(cp+num);
                ++cp;
            }while( *(cp+num) );
            *cp ='\0';
            goto done;
        }
    }
    /* replace space exhausted, shift right and continue */
    num = strlen( by );
    while( end >= cp ) {
        *(end+num) = *end;
        --end;
    }
    while( num ) {
        *cp++ = *by++;
        --num;
    }
done:
    if( reason & SEARCH_PLUS ) {
        if( rcx )
            ed.CursorX += rcx;
        /* I know, we shouldn't shift ed.CursorX, since e_paste doesn't
           shift it, too. But if the searched string would be 'EasyTeX'
           and the replace string '\EasyTeX' search woud request twice */
    } else {
        /* First possible occurence of search string can be
           (ed.CursorX-num), so we shift left here, could be necessary if
           we replace 'aa' by 'aaa' in a string like 'aaaa', otherwise
           we wold replace the just replaced string. */
        if( ed.CursorX > num )
            ed.CursorX -= num+1;
        else {
            if( !m_putcurrentline() ) return 0;
            _rewrite = TRUE;
            if( ed.CursorY == 0 ) return 0;
            --ed.CursorY;
            m_getcurrentline();
            ed.CursorX = m_framesize( ed.CursorY );
        }
    }

    _rewrite = TRUE;
    return 1;
}
// #pragma optimize( "", on )

/*
 *    Name: e_getclipboardstruct, e_savecboard, e_loadcboard, e_copy,
 *          e_cut, e_paste, e_jpaste
 *  Return:
 * Purpose: clipboard routines.
 *
 *
 *
 * (c)M.Schollmeyer
 */
struct Clipboard cboard = { CLIP_EMPTY, 0, 0, NULL };
#define _CB_COPY 1
#define _CB_CUT  0
static char *CODEBASED could_not_load = "Could not load clipboard.";
static char *CODEBASED could_not_save = "Could not save clipboard.";

static int _cbwork( int );

void e_getclipboardstruct( struct Clipboard *cb ) {

    *cb = cboard;
}


int e_savecboard( void ) {

    int fptr;
    char *cboardname;
    char tmpname[BUFSIZE];

    if( cboard.flags & CLIP_EXTERN )
        return 1;

    cboardname = GetProfileString( PR_CLIPBOARDNAME, NULL );
    if( cboard.flags & CLIP_EMPTY ) {
        if( cboardname )
            Delete( cboardname );
        return 1;
    }

    if( cboardname ) {
        strcpy( tmpname, cboardname );
        fptr = Open( tmpname, ACCESS_WRITE );
    } else
        fptr = CreateTmp( tmpname, 0 );

    if( fptr == -1 ) {
        DoErrorBox( HLP_NOSAVCBOARD, "Could not save clipboard." );
        return 0;
    }

    _Write( fptr, (char _far *)&cboard, sizeof( cboard ) );
    _Write( fptr, cboard.buffer, (unsigned)cboard.size );
    Close( fptr );
    PutProfileString( PR_CLIPBOARDNAME, tmpname );
    return 1;
}

int e_loadcboard( void ) {

    char *cboardname;
    int fptr;
    unsigned char _far *buf;

    cboardname = GetProfileString( PR_CLIPBOARDNAME, NULL );
    if( cboardname == NULL )
        return 0;

    fptr = Open( cboardname, ACCESS_READ );
    if( fptr == -1 ) {
        DoErrorBox( HLP_NOLOADCBOARD, could_not_load );
        goto error;
    }

    _Read( fptr, (char _far *)&cboard, sizeof( cboard ) );
    if( cboard.flags & CLIP_EXTERN ) {
        Close( fptr );
        return 1;
    }

    buf = AllocHeap( cboard.size );
    if( buf == 0L ) {
        DoErrorBox(  HLP_NOLOADCBOARD, "%s\n%s", could_not_load, errmsg(EM_OUTOFMEM) );
        goto error;
    }
    cboard.buffer = buf;

    _Read( fptr, buf, (unsigned)cboard.size );
    Close( fptr );
    return 1;

error:
    if( fptr != -1 ) Close( fptr );
    ClearProfileData( PR_CLIPBOARDNAME );
    return 0;
}


int e_copy( int key, int c )
{
    return _cbwork( _CB_COPY );
}

int e_cut( int key, int c )
{
    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    return _cbwork( _CB_CUT );
}

    // CL sais this one is too large for global optimization
// #pragma optimize( "egl", off )
static int _cbwork( int flag ) {

    unsigned long size = 0L;     /* size of clipboard */
    int           i,
                  j,
                  linelen,
                  fptr = -1;

    struct MarkRegion *mr;

    unsigned long         _huge *desc;
    unsigned char         _huge *buf,
                          _huge *clp,
                                *cp,
                                *buffer,
                                *dstbuf = NULL;

    char *cboardname;
    char tmpname[BUFSIZE];

    if( !(ed.Flags & MARKED ) ) {
        e_invalidarg(0);
        return FALSE;
    }

    if( cboard.buffer )
        free( cboard.buffer );
    cboard.buffer = NULL;

    CLEARBIT( cboard.flags, CLIP_STREAM );
    SETBIT( cboard.flags, CLIP_EMPTY );

    /* get coordinates for work */
    mr = m_getmr();

    if( !( mr->flags & MR_MULTILINE ) ) {
        /* stream clipboard */
        cp = currline;
        i = strlen( cp );
        mr->UpperX = _MIN( i, mr->UpperX );
        mr->LowerX = _MIN( i, mr->LowerX );
        i = mr->LowerX - mr->UpperX;

        if( i == 0 ) {
            SETBIT( cboard.flags, CLIP_EMPTY );
            CLEARBIT( ed.Flags, MARKED );
            return 0;
        }

        cboard.buffer = AllocHeap( EDIT_WIDTH+1 );
        if( cboard.buffer ) {
            clp = cboard.buffer;
            SETBIT( cboard.flags, CLIP_STREAM );
            CLEARBIT( cboard.flags, CLIP_EMPTY );
            *clp++ = SMALLFRAME|ALLOCFRAME;
            *clp++ = i;
            cboard.size = i + 2;
            cp += mr->UpperX;
            while( i ) {
                --i;
                *clp++ = *cp++;
            }
            if( flag == _CB_CUT ) {
                SETBIT( ed.Flags, MARKED );
                /* This was really a nasty bug! I had a e_chardel(0,1);
                   instead of the e_deletemr(); before, but that made
                   a deadcycle when e_chardel() has been called either
                   by pressing Shift+del or Shift+Backsp */
                e_deletemr();
            } else if( mr->flags & MR_MARKED ) {
                CLEARBIT( ed.Flags, MARKED );
                _redraw = TRUE;
            }
            return 1;
        } else {
            DoErrorBox( HLP_NOSAVCBOARD, "%s\n%s",
             could_not_save, errmsg(EM_OUTOFMEM) );
            return 0;
        }
    }

    /* multiline clipboard */
    if( !m_putcurrentline() )
        return 0;

    if( ! ( buffer = AllocHeap( EDIT_WIDTH+1 ) ) ) {
        return 0;
    }

    /* get size for clipboard */

    i = m_getline( mr->UpperY, buffer );
    mr->UpperX = _MIN( i, mr->UpperX );
    size = i - mr->UpperX + 2;
    i = m_getline( mr->LowerY, buffer );
    mr->LowerX = _MIN( i, mr->LowerX );
    size += mr->LowerX + 2;

    desc = mh.desc - mr->UpperY - 1;

    for( i = mr->LowerY - mr->UpperY - 1; i > 0; --i ) {
        size += *(mh.base + *desc + 1) + 2;
        --desc;
    }

    cboard.size = size;
    cboard.lines = mr->LowerY - mr->UpperY + 1;

    if( size > MAXCBSIZE ) {
        if( ! ( dstbuf = AllocHeap( EDIT_WIDTH+1 ) ) ) {
            free( buffer );
            return 0;
        }

        cboardname = GetProfileString( PR_CLIPBOARDNAME, NULL );

        if( cboardname ) {
            strcpy( tmpname, cboardname );
            fptr = Open( tmpname, ACCESS_WRITE );
        } else
            fptr = CreateTmp( tmpname, 0 );

        if( fptr == -1 ) {
            DoErrorBox( HLP_NOSAVCBOARD, could_not_save );
            free( buffer );
            free( dstbuf );
            return 0;
        }

        SETBIT( cboard.flags, CLIP_EXTERN );
        CLEARBIT( cboard.flags, CLIP_EMPTY );
        _Write( fptr, (char _far *)&cboard, sizeof( cboard ) );
        PutProfileString( PR_CLIPBOARDNAME, tmpname );

        clp = dstbuf;
    } else {
        CLEARBIT( cboard.flags, CLIP_EXTERN );
        cboard.buffer = AllocHeap( (unsigned int)(size) );
        if( cboard.buffer == NULL ) {
            free( buffer );
            if( dstbuf ) free( dstbuf );
            return 0;
        }
        clp = cboard.buffer;
        CLEARBIT( cboard.flags, CLIP_EMPTY );
    }

    /* copy first line */
    linelen = m_getline( mr->UpperY, buffer );
    linelen  -= mr->UpperX;
    *clp++ = SMALLFRAME|ALLOCFRAME;
    *clp++ = linelen;
    cp = buffer + mr->UpperX;

    for( i = linelen; i ; --i )
        *clp++ = *cp++;

    if( cboard.flags & CLIP_EXTERN ) {
        /* store line */
        _Write( fptr, dstbuf, linelen+2 );
    }

    /* copy mid lines */
    desc = mh.desc - mr->UpperY - 1;
    /* loop once per line */
    for( i = mr->LowerY - mr->UpperY - 1; i > 0; --i ) {
        if( cboard.flags & CLIP_EXTERN )
            clp = dstbuf;

        /* loop once per byte */
        for( buf = mh.base + *desc, linelen = j = *(mh.base + *desc + 1)+2;j; --j )
            *clp++ = *buf++;

        if( cboard.flags & CLIP_EXTERN )
            _Write( fptr, dstbuf, linelen );

        --desc;
    }

    /* copy last line */
    if( cboard.flags & CLIP_EXTERN )
        clp = dstbuf;

    m_getline( mr->LowerY, buffer );
    linelen = mr->LowerX;
    *clp++ = FREEFRAME|SMALLFRAME;
    *clp++ = (char)linelen;
    cp = buffer;
    for( i = linelen; i ; --i )
        *clp++ = *cp++;

    if( cboard.flags & CLIP_EXTERN ) {
        /* store line */
        _Write( fptr, dstbuf, linelen+2 );
    }

    if( flag == _CB_CUT ) {
        SETBIT( ed.Flags, MARKED );
        e_deletemr();
    }

    if( mr->flags & MR_MARKED ) {
        CLEARBIT( ed.Flags, MARKED );
        _redraw = TRUE;
    }

    free( buffer );
    if( dstbuf ) free( dstbuf );
    Close( fptr );
    return 1;
}
// #pragma optimize( "", on )


#define PASTE_N_JMP 1
#define PASTE_STD   2
static int _cbpaste( int );

int e_jpaste( int key, int c ) {

    return _cbpaste( PASTE_N_JMP );
}

int e_paste( int key, int c ) {

    return _cbpaste( PASTE_STD );
}

static int _cbpaste( int flag ) {

    unsigned char _huge *buf;
    int i, ret = 0, fptr = -1;
    unsigned long editorflags = ed.Flags;
    unsigned int editorwrap = ed.WordWrap;
    char *cboardname;
    char *linebuf = NULL;
    unsigned char linelen;

    if( cboard.flags & CLIP_EMPTY )
        return 0;

    if( ed.Flags & (READONLY|TEMPFILE) ) {
        error_readonly();
        return 0;
    }

    ClipMark.CursorX = ed.CursorX;
    ClipMark.CursorY = ed.CursorY;
    ClipMark.WindowX = ed.WindowX;
    ClipMark.WindowY = ed.WindowY;

    buf = cboard.buffer;

    --verbose;

    CLEARBIT( ed.Flags, INDENT );
    ed.WordWrap = 0;

    if( cboard.flags & CLIP_STREAM ) {
        ret = e_strnins( buf+2, *(buf+1) );
        if( flag == PASTE_STD ) {
            ed.CursorX = ClipMark.CursorX;
            ed.CursorY = ClipMark.CursorY;
            ed.WindowX = ClipMark.WindowX;
            ed.WindowY = ClipMark.WindowY;
        }
    } else {
        if( cboard.flags & CLIP_EXTERN ) {
            cboardname = GetProfileString( PR_CLIPBOARDNAME, NULL );
            if( cboardname == NULL )
                return 0;

            fptr = Open( cboardname, ACCESS_READ );
            if( fptr == -1 ) {
                DoErrorBox( HLP_NOLOADCBOARD, could_not_load );
                return 0;
            }
            if( ! ( linebuf = AllocHeap( EDIT_WIDTH+1 ) ) ) {
                Close( fptr );
                return 0;
            }
            _Read( fptr, (char _far *)&cboard, sizeof( cboard ) );
            buf = linebuf;
        }

        if( (unsigned long)mh.desc -
            sizeof(unsigned char _huge *) * mh.numlines <=
                (unsigned long)mh.endlines + cboard.size
              + sizeof(unsigned char _huge *) * cboard.lines ) {
            DoErrorBox( HLP_NOPASTE, "Could not paste,\nout of memory.\nTry 'Pack'." );
            ++verbose;
            ed.WordWrap = editorwrap;
            ed.Flags = editorflags;
            if( fptr != -1 ) Close( fptr );
            if( linebuf ) free( linebuf );
            return 0;
        }
        i = cboard.lines - 1;
        while( i >= 0 ) {
            if( cboard.flags & CLIP_EXTERN ) {
                /* first byte is garbage */
                _Read( fptr, &linelen, sizeof(unsigned char) );
                _Read( fptr, &linelen, sizeof(unsigned char) );
                _Read( fptr, linebuf, (unsigned)linelen );
                buf = linebuf;
            } else {
                linelen = buf[1];
                buf += 2;
            }
            if(! ( ret = e_strnins( buf, linelen ) ) )
                goto done;
            if( i )
                if(! ( ret = e_return( 0, 1 ) ) )
                    goto done;

            if( !(cboard.flags & CLIP_EXTERN) )
               buf += buf[-1];
            --i;
        }
        /* e_return() assigned the line descriptor table entry of the
           current line to 0L, so we have to put the line here since
           ed.CursorY is modified below
         */
done:
        if( flag == PASTE_STD ) {
            if( !m_putcurrentline() )
                return 0;
            ed.CursorX = ClipMark.CursorX;
            ed.CursorY = ClipMark.CursorY;
            ed.WindowX = ClipMark.WindowX;
            ed.WindowY = ClipMark.WindowY;
            m_getcurrentline();
        }
    }
    ++verbose;
    ed.WordWrap = editorwrap;
    /* we saved ed.Flags above, so we have to set LINECHANGED here
        since e_strnins() processed this line
     */
    ed.Flags = editorflags|MODIFIED|LINECHANGED;
    CLEARBIT( ed.Flags, MARKED );

    _redraw = _restat = TRUE;

    if( fptr != -1 ) Close( fptr );
    return ret;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_merge( char *file ) {

    int fptr;
    unsigned char buffer;
    unsigned int editorwrap = ed.WordWrap;
    unsigned long editorflags = ed.Flags;

    fptr = Open( file, ACCESS_READ );
    if( fptr == -1 ) {
        PutError( -1, file );
        return 0;
    }

    --verbose;
    CLEARBIT( ed.Flags, INDENT );
    m_beginundogroup();

    while( _Read( fptr, (char _far *)&buffer, 1 ) == 1 ) {
        switch( buffer ) {
            case 0x0d: e_return( 0, 1 );
                       Seek( fptr, 1L, SEEK_CURRENT );
                       break;
            case 0xff: break;
            default:   e_charins( (int)buffer, 1 );
                       break;
        }
    }

    ++verbose;
    ed.WordWrap = editorwrap;
    ed.Flags = editorflags|LINECHANGED|MODIFIED;
    m_putcurrentline();
    m_endundogroup();
    return 1;
}


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_insmode( int key, int c ) {

    ed.Flags ^= OVERTYPE;
    return e_drawflags();
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_setmark( int key, int c ) {

    ed.MarkX = ed.CursorX;
    ed.MarkY = ed.CursorY;
    SETBIT( ed.Flags, MARKED );

    /* It doesn't matter that we clear SHIFTMARKED here (e_left()... call
       e_setmark() as well to set the marker, but set the SHIFTMARKED
       flag afterwards). The reason why we have to clear SHIFTMARKED here
       is that, if e_setmark is called from somewhere outside, it means
       normal (nonSHIFTMAKED) marking */
    CLEARBIT( ed.Flags, SHIFTMARKED );

    _remark = _restat = TRUE;
    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_remark( int key, int c ) {

    SETBIT( ed.Flags, MARKED );

    _remark = TRUE;
    return 1;
}



/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_gomark( int key, int c ) {

    e_goto( ed.WindowX, ed.WindowY, ed.MarkX, ed.MarkY );
    _restat = TRUE;
    return 1;
}



/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_swapdm( int key, int c ) {

    unsigned int x, y;

    x = ed.MarkX;
    y = ed.MarkY;

    ed.MarkX = ed.CursorX;
    ed.MarkY = ed.CursorY;

    e_goto( ed.WindowX, ed.WindowY, x, y );
    _restat = _remark = TRUE;
    return 1;
}



/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_markline( int key, int c ) {

    ed.CursorX = 0;
    ed.MarkY = ed.CursorY;
    ed.MarkX = strlen( currline );

    if( ed.WindowX )
        ed.WindowX = 0;

    SETBIT( ed.Flags, MARKED );
    _remark = TRUE;
    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_delend( int key, int c ) {

    // delete all characters until end of line
    currline[ed.CursorX] = '\0';
    if( ed.MarkY == ed.CursorY && ed.MarkX > ed.CursorX )
        ed.MarkX = ed.CursorX;
    SETBIT(ed.Flags, LINECHANGED|MODIFIED);
    if( ed.Flags & MARKED ) {
        CLEARBIT( ed.Flags, MARKED );
        _redraw = TRUE;
    } else {
        _rewrite = TRUE;
    }
    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_delbeg( int key, int c ) {

    if( e_left( 0, 1 ) ) {
        do {
            if( !e_chardel( 0, 1 )  )
                break;
        } while( e_left( 0, 1 ) );

        return 1;
    }

    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
// #pragma optimize( "elg", off )
int e_format( int key, int c ) {

    unsigned char _huge *buf, _huge *bptr;
    unsigned int ul, bl, lm, oldwx, oldwy, oldcx, oldcy, oldmx, oldmy,
                 line, len;
    unsigned long size;
    char tmpbuf[EDIT_WIDTH], *cp;
    unsigned long editorflags = ed.Flags;

    if( !(ed.Flags & MARKED) ) {
        e_invalidarg(0);
        return 0;
    }

    --verbose;

    /* save the current line */
    if( !m_putcurrentline() ) return 0;

    ul = _MIN( ed.CursorY, ed.MarkY );
    bl = _MAX( ed.CursorY, ed.MarkY );

    oldwx = ed.WindowX;
    oldwy = ed.WindowY;
    oldcx = ed.CursorX;
    oldcy = ed.CursorY;
    oldmx = ed.MarkX;
    oldmy = ed.MarkY;

    /* first, get size for buffer */
    for( size = 0L, line = ul; line <= bl; ++line ) {
        size += (unsigned long)m_getline( line, tmpbuf ) + 2L;
    }

    if( size > 0xffff ) {
        DoErrorBox( HLP_PARTOOLARGE, "Your paragraph is too large, sorry" );
        return 0;
    }

    /* try to allocate buffer */
    buf = AllocHeap( (unsigned)size );
    if( buf == 0L )
        return 0;

    /* fill buffer with lines */

    for( size = 0L, bptr = buf, line = ul; line <= bl; ++line ) {
        len = m_getline( line, tmpbuf );
        for( cp = tmpbuf; *cp; ) {
            while( *cp == ' ' || *cp == '\t' ) ++cp;
            while( *cp != ' ' && *cp != '\t' && *cp ) {
                *bptr++ = *cp++;
                ++size;
            }
            *bptr++ = ' ';
            ++size;
        }
    }
    *(bptr-1) = '\0';

    /* get left margin */
    lm = ed.CursorX;
    if( lm >= ed.WordWrap ) {
        DoErrorBox( HLP_PARERROR1,
                    "The cursor should be left\nof what the wordwrap value is" );
        goto done;
    }

    /* delete lines */
    m_deleteline( ul, bl - ul );

    for( cp = currline, len = lm; len; ++cp, --len ) *cp = ' ';
    *cp = '\0';
    ed.CursorY = ul;
    ed.CursorX = lm;

    SETBIT( ed.Flags, INDENT );
    CLEARBIT( ed.Flags, OVERTYPE|MARKED );

    for( bptr = buf; size; --size, ++bptr ) {
        if( !e_strnins( bptr, 1 ) ) break;
    }

    ed.Flags = editorflags|LINECHANGED|MODIFIED;
    CLEARBIT( ed.Flags, MARKED );

    ed.MarkX = oldmx;
    ed.MarkY = oldmy;
    e_goto( oldwx, oldwy, oldcx, oldcy );

done:
    free( buf );

    ++verbose;
    _redraw = TRUE;
    return 1;
}
// #pragma optimize( "", on )


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_learn( int key, int c ) {

    e_drawflags();
    return 1;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_launch( int key, int c ) {

    int slot;

    slot = ((unsigned int)key>>8) - 0x3f;
    LaunchMacro( GetProfileData( PR_MACROS(slot), NULL ), slot );
    e_drawflags();
    return 1;
}

/*
 *  The following are special LaTeX functions
 */
static int _env_work( int );

static int _te_enumerate( void );
static int _te_tabular( void );
static int _te_picture( void );

static int _descitem( void );
static int _normitem( void );
static int _capitem( void );
static int _secitem( void );
static int _putitem( void );
static int _bibitem( void );

static int isbgchar( char * );
static int isegchar( char * );
static char *_latexenv( char *, int );
static int _balance( char *, int * );

    static struct {
        char key;
        char *envname;
        int (*special)( void );
        int (*itemfunc)( void );
    } CODEBASED envtab[] = {
{'a',"array", NULL, NULL },
{'b',"abstract", NULL, NULL },
{'c',"center", NULL, NULL },
{'d',"description", NULL, _descitem },
{'e',"enumerate", NULL, _normitem },
{'f',"figure", NULL, _capitem },
{'g',"flushright", NULL, NULL },
{'h',"flushleft", NULL, NULL },
{'i',"itemize", NULL, _normitem },
{'k',"fussypar", NULL, NULL },
{'l',"list", NULL, NULL },
{'m',"math", NULL, NULL },
{'n',"minipage", NULL, NULL },
{'o',"document", NULL, _secitem },
{'p',"picture", _te_picture, _putitem },
{'q',"quote", NULL, NULL },
{'r',"abstract", NULL, NULL },
{'s',"sloppypar", NULL, NULL },
{'t',"tabular", _te_tabular, NULL },
{'u',"letter", NULL, NULL },
{'v',"verbatim", NULL, NULL },
{'w',"equation", NULL, NULL },
{'x',"appendix", NULL, _secitem },
{'y',"displaymath", NULL, NULL },
{'z',"eqnarray", NULL, NULL },
{'1',"quotation", NULL, NULL },
{'2',"samepage", NULL, NULL },
{'3',"tabbing", NULL, NULL },
{'4',"table", NULL, NULL },
{'5',"thebibliography", NULL, _bibitem },
{'6',"theindex", NULL, _normitem },
{'7',"titlepage", NULL, NULL },
{'8',"trivlist", NULL, NULL },
{'9',"verse", NULL, NULL },
{'\0', "" , NULL, NULL },
};

static char *CODEBASED send = "\\end{%s}";
static char *CODEBASED sbegin = "\\begin{%s}";


static char *CODEBASED bgchars[] = { "{", "[", "(", "\\begin", NULL };
static char *CODEBASED egchars[] = { "}", "]", ")", "\\end", NULL };
#define LATEXENVNUM 3
#define EW_BEGIN    1
#define EW_END      2
#define EW_BEGINEND 3

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int e_balance( int key, int c ) {

    char buffer[EDIT_WIDTH];
    int offset, y;

    offset = _balance( buffer, &y );
    if( offset ) {
        /* this char is unbalanced */
        e_strins( _latexenv(buffer,offset) );
        _setcursorpos();
        m_putmsg( MSG_STNDRD, "(%d):%-.*s", y, SCREEN_WIDTH-8, buffer );
        return 0;
    }
    return 0;
}

/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int et_lastitem( int key, int c ) {

    char buffer[EDIT_WIDTH], bal[EDIT_WIDTH], *cp;
    unsigned int y, offset, i;

    offset = _balance( bal, &y );
    if( offset == LATEXENVNUM+1 ) {
        for( i = 0; envtab[i].key; ++i ) {
            sprintf( buffer, send, envtab[i].envname );
            if( strcmp( buffer, _latexenv(bal, offset) ) == 0 ) {
                /* This is a nonbalanced environment */
                if( envtab[i].itemfunc )
                    return (*(envtab[i].itemfunc))();
                else
                    break;
            }
        }
    }
    m_putmsg( MSG_STNDRD, "(%d): no item: %-.*s", y, SCREEN_WIDTH-18, bal );
    return 0;
}

/* _latexenv() converts a group level delimiter into a defined format.
   everything except "\\begin" or "\\end" will be ignored. These two
   delimiters should be treated more sophisticated:
   "\\begin {math} ...\\end{math}" should be recognized though the two
   strings are not equal.
   _latexenv() returns the corresponding end group delimiter.
*/
static char *_latexenv( char *cp, int i ) {

    char *eptr;
    static char CODEBASED arg[EDIT_WIDTH];
    char *delim;

    arg[0] = '\0';

    if( i == LATEXENVNUM+1 ) {
        strcpy( arg, "\\end" );
        eptr = cp;
        /* skip command, can be either "\\begin" or "\\end" */
        if( !strncmp( cp, egchars[LATEXENVNUM], strlen(egchars[LATEXENVNUM]) ) )
            eptr += strlen(egchars[LATEXENVNUM]);
        else if( !strncmp( cp, bgchars[LATEXENVNUM], strlen(bgchars[LATEXENVNUM]) ) )
            eptr += strlen(bgchars[LATEXENVNUM]);

        /* now eptr points to the first char behind the command,
           skip blanks if any */
        while( *eptr == ' ' ) ++eptr;
        /* eptr points to argument, get length */
        /* the first char of the argument determines the type wich is
           stored in delim */
        if( *eptr == '{' ) delim = "}";   /* argument ends with "}" */
        else               delim = " \\"; /* argument ends with
                                             either " " or "\\"  */
        /* now get length of argument */
        for( i = 0; !strchr(delim, eptr[i]) && eptr[i] ; ++i );
        /* check if argument is braced, if so, add the brace to argument */
        if( delim[0] == '}' ) ++i;
        strncat( arg, eptr, i );
        arg[i+4] = '\0';
    } else
        strcpy(arg, egchars[i-1] );

    return arg;
}


static int isbgchar( char *c ) {

    int taboff;

    for( taboff = 0; bgchars[taboff]; ++taboff ) {
        if( strncmp( bgchars[taboff], c, strlen(bgchars[taboff] ) ) == 0 )
            return (taboff+1);
    }
    return 0;
}

static int isegchar( char *c ) {

    int taboff;

    for( taboff = 0; egchars[taboff]; ++taboff ) {
        if( strncmp( egchars[taboff], c, strlen( egchars[taboff] ) ) == 0 )
            return (taboff+1);
    }
    return 0;
}


int et_begin( int key, int c ) {

    if( IntuitionBase->InputEvent->shift & IES_SHIFT )
        return _env_work( EW_BEGINEND );
    else
        return _env_work( EW_BEGIN );
}

int et_end( int key, int c ) {

    return _env_work( EW_END );
}

static int _te_tabular( void ) {

    if( e_up( 0,1 ) )
        if( e_end( 0,1 ) )
            if( e_strins( "{}" ) )
                return e_left( 0,1 );
    return FALSE;
}

static int _te_picture( void ) {

    if( e_up( 0,1 ) )
        if( e_end( 0,1 ) )
            if( e_strins( "(,)" ) )
                return e_left( 0, 2 );
    return FALSE;
}


static int _descitem( void ) {

    if( e_strins( "\\item[]" ) )
        return e_left( 0, 1 );
    return FALSE;
}

static int _normitem( void ) {

    return e_strins( "\\item " );
}

static int _capitem( void ) {

    if( e_strins( "\\caption{}" ) )
        return e_left( 0, 1 );
    return FALSE;
}

static int _secitem( void ) {

    if( e_strins( "\\section{}" ) )
        return e_left( 0, 1 );
    return FALSE;
}

static int _putitem( void ) {

    if( e_strins( "\\put(,){}" ) )
        return e_left( 0, 4 );
    return FALSE;
}

static int _bibitem( void ) {

    if( e_strins( "\\bibitem{}" ) )
        return e_left( 0, 1 );
    return FALSE;
}

/* This function scans the file backwards to search for nunbalanced
   characters. The actual parameter should be a buffer where the
   end group character is copied. The return value is an integer
   representing an offset to the egchars array or 0 if nothing found */
static int _balance( char *dst, int *y ) {

    char buffer[EDIT_WIDTH], msgbuf[EDIT_WIDTH], *cp, *ccp;
    unsigned int len, i;
    void *stack = NULL;

    if( !m_putcurrentline() ) return 0;

    *y = ed.CursorY;
    /* get next line */
    len = m_getline( *y, buffer );
    cp = &buffer[_MIN((int)len-1,(int)ed.CursorX-1)];

    m_putmsg( MSG_STNDRD, "balancing..." );

    for ever {
        if( *y % 50 == 0 ) {
            m_putxmsg( CR_STATUS, VideoBase->ScreenWidth-7, "(%d) ", *y );
        }
        /* scan every line */
        while( cp >= buffer ) {
            if( i = isbgchar( cp ) ) {
                /* it's a begin group charcter which must be on stack */
                if( stack == NULL ) {
                    /* stack empty, this char is unbalanced */
                    strncpy( dst, cp, EDIT_WIDTH-1 );
                    dst[EDIT_WIDTH-1]='\0';
                    return i;
                }
                if( strncmp(_latexenv(cp,i),(char *)stack,strlen((char *)stack)) ) {
                    m_putmsg( MSG_STNDRD, "(%d): Non matching levels: %-.*s", *y, SCREEN_WIDTH-31, cp );
                    FreeStack( stack );
                    return 0;
                }
                /* pop from stack */
                stack = PopStack( stack );
            } else if( i = isegchar( cp ) ) {
                /* push corresponding end group on stack */
                ccp = _latexenv(cp,i);
                stack = PushStack( stack, (APTR)ccp, strlen(ccp)+1 );
            }
            --cp;
        }
        if( (*y == 0) || ScanKeybrd( BK_ESC ) ) {
            m_putmsg( MSG_STNDRD, "No unbalanced characters found." );
            FreeStack( stack );
            return 0;
        }
        /* get next line */
        len = m_getline( --*y, buffer );
        cp = &buffer[len-1];
    }
}

// #pragma optimize( "elg", off )
static int _env_work( int flag ) {

    char *buffer;
    int i, j;
    struct InputEvent *ie;
    int cx;
    unsigned int editorwrap;

    buffer = AllocHeap( 100 );
    if( buffer == 0L )
        return 0;

    PushTopic( HLP_TEXENV );
    m_putmsg( MSG_STNDRD, "F1=Help -- Press the Environment Key" );

    do {
        ie = ReadInput();
    } while( ie->key == 0 );

    PopTopic();
    e_status( 1 );

    for( i = 0; envtab[i].key; ++i ) {
        if( (char)ie->key == envtab[i].key ) {
            if( flag == EW_BEGINEND ) {
                editorwrap = ed.WordWrap;
                ed.WordWrap = 0;
                if( e_lineins( 0, 2 ) ) {
                    cx = ed.CursorX;
                    sprintf( buffer, sbegin, envtab[i].envname );
                    if( e_strins( (unsigned char _huge *)buffer ) ) {
                        ed.CursorX = cx;
                        e_down( 0, 2 );
                        sprintf( buffer, send, envtab[i].envname );
                        if( e_strins( (unsigned char _huge *)buffer ) ) {
                            ed.CursorX = cx;
                            if( e_up( 0, 1 ) ) {
                                if( ed.Flags & INDENT )
                                    e_tab( 0, 1 );
                                ed.WordWrap = editorwrap;
                                free( buffer );
                                if( envtab[i].special )
                                    return (*(envtab[i].special))();
                                else
                                    return 1;
                            }
                        }
                    }
                }
                ed.WordWrap = editorwrap;
                free( buffer );
                _restat = TRUE;
                return 0;

            } else if( flag == EW_END ) {

                sprintf( buffer, send, envtab[i].envname );

                if( ed.Flags & INDENT )
                    e_backtab( 0, 1 );

                i = e_strins( (unsigned char _huge *)buffer );

                free( buffer );
                if( i ) {
                    if( e_return( 0, 1 ) ) {
                        return 1;
                    }
                }

            } else if( flag == EW_BEGIN ) {

                sprintf( buffer, sbegin, envtab[i].envname );

                j = e_strins( (unsigned char _huge *)buffer );

                free( buffer );
                if( j ) {
                    if( e_return( 0, 1 ) ) {
                        if( envtab[i].special )
                            return (*(envtab[i].special))();
                        else
                            return e_tab(0,1);
                    }
                }

            }
            return 0;
        }
    }

    free( buffer );
    return 0;
}
// #pragma optimize( "", on )


/*
 *    Name:
 *  Return:
 * Purpose:
 *
 *
 *
 * (c)M.Schollmeyer
 */
int et_index( int key, int c ) {

    struct MarkRegion *mr = m_getmr();
    char buffer[EDIT_WIDTH];
    int i;
    if( mr->flags & MR_MULTILINE && mr->flags & MR_MARKED ) {
        e_invalidarg( 0 );
        return 0;
    }

    strcpy( buffer, "\\index{" );

    if( mr->flags & MR_MARKED ) {
        i = mr->LowerX - mr->UpperX;
        strncat( buffer, currline+mr->UpperX, i );
        buffer[i+7] = '\0';
        CLEARBIT( ed.Flags, MARKED );
        _remark = TRUE;
    }

    strcat( buffer, "}" );

    if( mr->flags & MR_MARKED ) {
        e_goto( ed.WindowX, ed.WindowY, mr->LowerX, ed.CursorY );
        return e_strins( buffer );
    } else {
        if( e_strins( buffer ) )
            return e_left( 0,1 );
        return 0;
    }
}

void e_invalidarg( int topic ) {

    DoErrorBox( topic ? topic : HLP_INVARG, "Invalid Argument" );
}

/* end of file edit.c */
