/*
 * SOURCE:  makehelp.c
 * PROJECT: makehelp
 *
 * PURPOSE: main module
 *
 */
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <stdio.h>
#include <stdarg.h>

#include "compress.h"
#include "help.h"

/*  makehelp converts the ascii source code file to the help database.

    the ascii source code has the following format:

    @p LABELNAME Title of Label
    @pr REFERENCE Title of Reference
    @dt @c43test@c00
    ; This is a comment
    @l LABELNAME
    This string is displayed normal
    These words are @c31blue@c00 and @c34red@c00

    This is a @!reference@REFERENCE@
    See also: @*REFERENCE@

    You can also print the Comment introducer '@;'
    and the AT-sign '@@'
    ;
    @l REFERENCE
    The difference between References and Labels is
    that References will not appear in the relocation
    table.

    the help database has the following format:

    4 bytes magic chars (defined as HELP_MAGIC)
    256 bytes frequency values
    relocation table:
        4 bytes size of table (in bytes)
        2 bytes number of entries
        entries:
            4 bytes offset of topic
            n bytes topic name (null-terminated string)
              maximum length defined in MAX_TOPICNAME
    topics:
        4 bytes size of decompressed entry (in bytes)
        2 bytes number of lines (includes title)
        4 bytes size of compressed entry (in bits)

        2 bytes length of title (in bytes)
        n bytes topic title (null terminated string)
          maximum length defined in MAX_TOPICTITLE
        lines:
            2 bytes length of line (in bytes)
            n bytes line information
              information:
                code 01bf: change color background to b
                                        foreground to f
                             b=0,f=0 change to default
                code 02: start button
                code 03hhhh: end button
                             hhhh: offset of reference in reloc table
                                   (unsigned)

*/


#define NUMTABS 8
#define MAX_TOPICS 1000
#define MAX_UBYTE 0xff
#define MAX_MACROLEVEL 16
#define MAX_LINELEN 250

typedef unsigned char ubyte;

            // statics

struct TopicStruc {
    ubyte name[MAX_TOPICNAME+1];
    ubyte title[MAX_TOPICTITLE+1];
    unsigned long offset;
    char istopic;       /* 1 for topic, 0 for reference */
    char resolved;      /* 1 for resolved, 0 for not */
    unsigned long size;
    unsigned numlines;
};

struct TopicStruc *topic[MAX_TOPICS];

int currtopic = -1;
int currline = 0;
int currprot = -1;

int warnings = 0, errors = 0, srcline;

static char *unex_eol = "unexpected end of line";
static char *out_of_mem = "out of memory";
static ubyte *all_cmds = "@;dcp*!l";
static char *tmpfilebasnam = "makehelp.!!!";

static char tmpfilenam[_MAX_PATH];
static char srcfilenam[_MAX_PATH];
static char *nullstring = "";

ubyte _far *macros[MAX_UBYTE];

    // function prototypes
int main( int, char ** );
int _setcolor( ubyte _far **, ubyte _far ** );

int atoh( ubyte );

int readln( FILE *, ubyte * );

void printerr( unsigned, unsigned, char *, ... );
#define WARNING __LINE__, 0
#define ERROR   __LINE__, 1
#define FATAL   __LINE__, 2

// #pragma optimize( "elg", off )
int main( int argc, char **argv ) {

                                     /* file handles             */
    FILE         *source = NULL,     /*         source file      */
                 *dst = NULL,        /*         destination file */
                 *tmp = NULL;        /*         temporary file   */

    static ubyte   in[2*MAX_LINELEN], /* line buffers             */
                   out[1024];

    ubyte    _far *inptr,             /* general purpose pointers */
             _far *outptr,
             _far *cp;

    unsigned long magic = HELP_MAGIC,
                  size,
                  maxtopicsize = 0;

    unsigned      len;
    int           i,
                  j,
                  k,
                  numtopics = 0,     /* all topics (labels+references) */
                  numlabels = 0,     /* all labels */
                  currfile;          /* current source file */

    ubyte _far    *oldinptr[MAX_MACROLEVEL+1];
    unsigned      macrolevel = 0;


    printf( "MAKEhelp Version 2.1  (c)1991,1992 ITLR\n\n" );

    ResetCount();

    /* first, check arguments */
    if( argc < 3 ) {
        printerr( FATAL, "Bad arguments!  Usage: makehelp {source}... destination" );
        exit( 99 );
    }

    /* open destination file */
    dst = fopen( argv[argc-1], "wb" );
    if( dst == NULL ) {
        printerr( FATAL, "unable to open destination file: '%s'\n", argv[argc-1] );
        goto error;
    }

    /* open temporary file */
    tmpfilenam[0] = 0;
    if( outptr = getenv( "TMP" ) ) {
        strcpy( tmpfilenam, outptr );
        if( tmpfilenam[strlen(tmpfilenam)-1] != '\\' )
            strcat( tmpfilenam, "\\" );
    }
    strcat( tmpfilenam, tmpfilebasnam );

    tmp = fopen( tmpfilenam, "w+b" );
    if( tmp == NULL ) {
        printerr( FATAL, "unable to open temporary file: '%s'\n", tmpfilenam );
        goto error;
    }
    printf( "temporary file '%s' has been created\n", tmpfilenam );

    size = 0L;  /* will be size of relocation table */

    /* open next file */
    for( currfile = 1; currfile < argc-1; ++currfile ) {
        if( source ) fclose( source );
        source = fopen( argv[currfile], "r" );
        if( source == NULL ) {
            printerr( FATAL, "unable to open source file: '%s'", argv[currfile] );
            goto error;
        }
        strcpy( srcfilenam, argv[currfile] );
        srcline = 0;
        printf( "reading source file '%s'...\n", srcfilenam );

        while( 1 ) {
    nextline:
            /* read line to buffer */
            if( readln( source, in ) == -1 ) break;
            ++srcline;

            inptr = in;
            outptr = out;

            while( 1 ) {
                switch( *inptr ) {
                    case '\0': /* end of scan line */
                        if( macrolevel ) {
                            --macrolevel;
                            inptr = oldinptr[macrolevel];
                            break;
                        } else
                            goto writeline;
                    case ';': /* comment */
                        if( inptr == in )
                            goto nextline;
                        while( *inptr ) ++inptr;
                        break;
                    case '@': /* command */
                        switch( *++inptr ) {
                            case '@': /* fall thru */
                            case ';': /* print out verbose */
                                *outptr++ = *inptr++;
                                break;

                            case '\0':
                                /* read line to buffer */
                                if( readln( source, in ) == -1 )
                                    goto nextline;
                                ++srcline;
                                inptr = in;
                                break;

                            case 'd': /* define */
                                if( *++inptr == 0 ) {
                                    printerr( WARNING, "define: %s", unex_eol );
                                    goto nextline;
                                }
                                i = (int)(*inptr);
                                if( strchr( all_cmds, i ) != NULL ) {
                                    printerr( ERROR, "define: char '%c' is a command", *inptr );
                                    goto nextline;
                                }

                                ++inptr;
                                while( isspace( *inptr ) ) ++inptr;
                                if( *inptr == 0 ) {
                                    /* undefined */
                                    if( macros[i] )
                                        free( macros[i] );
                                    else
                                        printerr( ERROR, "define: %s", unex_eol );
                                    goto nextline;
                                }
                                if( macros[i] )
                                    printerr( WARNING, "'%c' redefinition", (ubyte)i );
                                macros[i] = strdup( inptr );
                                if( macros[i] == NULL )
                                    printerr( ERROR, "define: %s", out_of_mem );

                                goto nextline;

                            case 'c': /* set color */
                                _setcolor( &outptr, &inptr );
                                break;

                            case 'p': /* prototype */
                                if( ++currprot >= MAX_TOPICS ) {
                                    printerr( FATAL, "compiler limit: number of topics exeeds %d", MAX_TOPICS );
                                    exit( 99 );
                                }
                                topic[currprot] = malloc( sizeof(struct TopicStruc) );
                                if( topic[currprot] == NULL ) {
                                    printerr( FATAL, "allocstruct: %s", out_of_mem );
                                    exit( 99 );
                                }
                                memset( topic[currprot], 0, sizeof(struct TopicStruc) );

                                ++inptr;
                                if( *inptr == 'r' ) {
                                    topic[currprot]->istopic = 0;
                                    ++inptr;
                                } else
                                    topic[currprot]->istopic = 1;

                                topic[currprot]->resolved = 0;

                                while( isspace( *inptr ) ) ++inptr;
                                if( *inptr == 0 ) {
                                    printerr( ERROR, "prototype: %s", unex_eol );
                                    break;
                                }
                                cp = inptr;
                                while( *cp && !isspace( *cp ) ) ++cp;
                                if( *cp == 0 )
                                    i = 1;
                                else
                                    i = 0;

                                *cp = 0;
                                if( strlen( inptr ) > MAX_TOPICNAME ) {
                                    printerr( WARNING, "prototype: '%s': name too long", inptr );
                                    inptr[MAX_TOPICNAME] = 0;
                                }
                                strcpy( topic[currprot]->name, inptr );
                                if( !i ) {
                                    ++cp;
                                    while( isspace( *cp ) ) ++cp;
                                }
                                if( i || (*cp == 0) ) {
                                    printerr( WARNING, "prototype: %s", unex_eol );
                                    strcpy( topic[currprot]->title, "???" );
                                    break;
                                }
                                if( strlen( cp ) > MAX_TOPICTITLE ) {
                                    printerr( WARNING, "prototype: '%s': title too long", cp );
                                    inptr[MAX_TOPICTITLE] = 0;
                                }
                                strcpy( topic[currprot]->title, cp );

                                topic[currprot]->size = 0L;

                                if( topic[currprot]->istopic )
                                    size += sizeof(long) +
                                            strlen(topic[currprot]->name) +
                                            sizeof(ubyte);
                                else
                                    size += sizeof(long) +
                                            sizeof(ubyte);

                                for( i = 0; i < currprot; ++i ) {
                                    if( strcmp( topic[i]->name, topic[currprot]->name ) == 0 ) {
                                        printerr( WARNING, "prototype: '%s': redeclared", topic[i]->name );
                                    }
                                }
                                goto nextline;

                            case '*': /* auto reference */
                                *outptr++ = 02;
                                cp = ++inptr;
                                /* scan for end of name */
                                j = 0;
                                while( *cp != '@' ) {
                                    if( *cp == 0 ) {
                                        printerr( WARNING, "reference: %s", unex_eol );
                                        j = 1;
                                        break;
                                    }
                                    ++cp;
                                }
                                *cp = 0;
                                /* scan in topic list */
                                for( i = 0; i <= currprot; ++i ) {
                                    if( strcmp( topic[i]->name, inptr ) == 0 ) {
                                        /* reference found */
                                        break;
                                    }
                                }
                                if( i == currprot+1 ) {
                                    /* reference not found */
                                    printerr( WARNING, "'%s': no prototype given", inptr );
                                    strcpy( outptr, "???" );
                                    i = 0;
                                } else {
                                    strcpy( outptr, topic[i]->title );
                                }
                                while( *outptr ) ++outptr;

                                *outptr++ = 03;
                                *(unsigned *)outptr = (unsigned)i;
                                outptr += sizeof(unsigned);

                                if( !j )
                                    inptr = cp + 1;
                                else
                                    inptr = cp;

                                break;

                            case '!': /* manual reference */
                                *outptr++ = 02;
                                /* copy string */
                                ++inptr;
                                /* scan for end of name */
                                while( 1 ) {
                                    if( *inptr == 0 ) {
                                        printerr( WARNING, "reference: %s", unex_eol );
                                        --inptr;
                                        break;
                                    } else if( *inptr == '@' ) {
                                        ++inptr;
                                        if( *inptr == 'c' ) {
                                            if( _setcolor( &outptr, &inptr ) == 0 ) {
                                                /* set to default illegal here */
                                                printerr( ERROR, "default color set illegal inside reference" );
                                            }
                                        }
                                        else
                                            break;
                                    } else
                                        *outptr++ = *inptr++;
                                }
                                *outptr++ = 03;
                                /* copy ref name */
                                cp = inptr;
                                /* scan for end of name */
                                j = 0;
                                while( *cp != '@' ) {
                                  if( *cp == 0 ) {
                                    printerr( ERROR, "reference: %s", unex_eol );
                                    j = 1;
                                    break;
                                  }
                                  ++cp;
                                }
                                *cp = 0;

                                /* scan in topic list */
                                for( i = 0; i <= currprot; ++i ) {
                                    if( strcmp( topic[i]->name, inptr ) == 0 ) {
                                        /* reference found */
                                        break;
                                    }
                                }
                                if( i == currprot+1 ) {
                                    /* reference not found */
                                    printerr( WARNING, "'%s': no prototype given", inptr );
                                    i = 0;
                                }

                                *(unsigned *)outptr = (unsigned)i;
                                outptr += sizeof(unsigned);

                                if( !j )
                                    inptr = cp + 1;
                                else
                                    inptr = cp;

                                break;

                            case 'l': /* label */
                                ++numlabels;
                                cp = inptr + 1;
                                while( isspace(*cp) ) ++cp;
                                if( *cp == 0 ) {
                                    printerr( WARNING, "label: %s", unex_eol );
                                    break;
                                }

                                for( i = 0; i <= currprot; ++i ) {
                                    if( strcmp( topic[i]->name, cp ) == 0 ) {
                                        /* reference found */
                                        break;
                                    }
                                }
                                if( i == currprot+1 ) {
                                    /* reference not found */
                                    printerr( ERROR, "'%s': no prototype given", cp );
                                    i = 0;
                                } else {
                                    if( topic[i]->resolved == 1 )
                                        printerr( ERROR, "'%s': redefinition", cp );
                                }
                                topic[i]->resolved = 1;

                                if( currtopic != -1 ) {
                                    topic[currtopic]->numlines = currline;
                                    if( topic[currtopic]->size > maxtopicsize )
                                        maxtopicsize = topic[currtopic]->size;
                                }
                                currtopic = i;
                                currline = 0;

                                printf( "%d\r", currtopic );

                                strcpy( out, topic[currtopic]->title );
                                outptr += strlen( topic[currtopic]->title )+1;
                                topic[currtopic]->offset = ftell( tmp );

                                goto writeline;

                            default:  /* unknown command */
                                      /* look if it is a macro */
                                      i = (int)(*inptr);
                                      if( macros[i] ) {
                                        if( macrolevel >= MAX_MACROLEVEL ) {
                                            printerr( ERROR, "define: level exeeds %d", MAX_MACROLEVEL );
                                            break;
                                        }
                                        oldinptr[macrolevel++] = ++inptr;
                                        inptr = macros[i];
                                      } else
                                          printerr( WARNING, "unknown command: '%c'", *inptr );
                                      break;
                          }
                          break;
                    default: /* any char */
                        *outptr++ = *inptr++;
                }
            }
    writeline:
            if( currtopic == -1 ) {
                printerr( ERROR, "missing topic, line skipped" );
                goto nextline;
            }


            i = (int)(outptr-(ubyte _far *)out);
            fwrite( &i, sizeof(unsigned), 1, tmp );
            fwrite( out, sizeof(ubyte), i, tmp );

            AddCount( (char _far *)&i, sizeof(int) );
            AddCount( out, (unsigned long)i );

            topic[currtopic]->size += i + sizeof(unsigned);
            ++currline;
        }
    }

    if( currprot == -1 ) {
        printerr( FATAL, "no topics found" );
        goto error;
    }
    numtopics = currprot+1;

    for( i = 0; i < numtopics; ++i ) {
        if( topic[i]->resolved == 0 )
            printerr( ERROR, "'%s': unresolved topic", topic[i]->name );
    }

    printf( "%d topics found\n", numtopics );

    topic[currtopic]->numlines = currline;

    /* next pass: compress labels */

    if( BuildTree() == CE_NOMEM ) {
        printerr( FATAL, "build huffman tree: %s", out_of_mem );
        exit( 99 );
    }

    maxtopicsize += sizeof(unsigned);
    inptr = malloc( (unsigned)maxtopicsize );
    if( inptr == NULL ) {
        printerr( FATAL, "compress file: %s", out_of_mem );
        exit( 99 );
    }
    outptr = malloc( (unsigned)(maxtopicsize+maxtopicsize/8+1) );
    if( outptr == NULL ) {
        printerr( FATAL, "compress file: %s", out_of_mem );
        exit( 99 );
    }

    fwrite( &magic, sizeof(unsigned long), 1, dst );
    fwrite( GetTree(), sizeof(unsigned long), 256, dst );
    fwrite( &size, sizeof(unsigned long), 1, dst );
    fwrite( &numtopics, sizeof(unsigned), 1, dst );
    fseek( dst, size, SEEK_CUR );

    for( currtopic = 0; currtopic < numtopics; ++currtopic ) {
        printf( ">%-*s\r", MAX_TOPICNAME, topic[currtopic]->name );
        size = topic[currtopic]->size + 1;
        fseek( tmp, topic[currtopic]->offset, SEEK_SET );
        fread( inptr, sizeof(ubyte), (unsigned)size, tmp );
        topic[currtopic]->offset = ftell( dst );
        fwrite( &size, sizeof(unsigned long), 1, dst );
        fwrite( &(topic[currtopic]->numlines), sizeof(unsigned), 1, dst );
        size = Encode( outptr, maxtopicsize+maxtopicsize/8+1,
                       inptr, size );
        fwrite( &size, sizeof(unsigned long), 1, dst );
        fwrite( outptr, sizeof(ubyte), (unsigned)(size/8+1), dst );
    }

    /* next pass: write reloc table */

    fseek( dst, 10L + 4L*256L, SEEK_SET );

    for( currtopic = 0; currtopic < numtopics; ++currtopic ) {
        fwrite( &(topic[currtopic]->offset), sizeof(unsigned long), 1, dst );
        if( topic[currtopic]->istopic )
            fwrite( topic[currtopic]->name, sizeof(ubyte), strlen(topic[currtopic]->name)+1, dst );
        else
            fwrite( nullstring, sizeof(ubyte), 1, dst );
    }


error:
    if( dst && !errors ) {
        fseek( dst, 0L, SEEK_END );
        printf( "done. (%u topics, %lu bytes)\n\n", currtopic, ftell( dst ) );
    }

    printf( "      %d warnings, %d errors.\n\n", warnings, errors );

    if( dst ) fclose( dst );
    if( source ) fclose( source );
    if( tmp ) {
        fclose( tmp );
        remove( tmpfilenam );
    }

    if( errors )
        remove( argv[argc-1] );

    exit( 0 );
}
// #pragma optimize( "", on )


int _setcolor( ubyte _far **out, ubyte _far **in ) {

    ubyte _far *outptr, _far *inptr;
    int color = 0;

    outptr = *out;
    inptr = *in;

    *outptr++ = 01;
    if( *++inptr == 0 ) {
        printerr( WARNING, "color set: %s", unex_eol );
    } else {
        color = atoh( *inptr );
        if( *++inptr == 0 )
            printerr( WARNING, "color set: %s", unex_eol );
        else {
            color = (color << 4) | atoh( *inptr );
            ++inptr;
        }
    }
    *outptr++ = color;

    *out = outptr;
    *in = inptr;

    return color;
}


int atoh( ubyte c ) {

    c = tolower( c );

    if( c >= '0' && c <= '9' )
        return (c-'0');
    if( c >= 'a' && c <= 'f' )
        return (c-'a'+10);

    printerr( WARNING, "illegal hex char: '%c'", c );
    return 0;
}


int readln( FILE *fptr, ubyte *buffer ) {

    ubyte *cp, *dest;
    int xpos = 0;

    dest = buffer;
    cp = buffer+MAX_LINELEN;
    if( fgets( cp, MAX_LINELEN, fptr ) == NULL )
        return -1;

    /* expand tabs */
    while( *cp && *cp != '\n' ) {
        if( *cp == '\x09' ) {
            ++cp;
            do {
                *dest++ = ' ';
                ++xpos;
            } while( xpos % NUMTABS );
        } else {
            *dest++ = *cp++;
            ++xpos;
        }
    }
    *dest = '\0';
done:
    return xpos;
}




static char *errintro[] = { "warning", "error", "fatal error" };
void printerr( unsigned line, unsigned type, char *fmt, ... ) {

    va_list marker;

    printf( "%s(%d): %s M%04u: ", srcfilenam, srcline, errintro[type], line );

    if( type ) ++errors;
    else       ++warnings;

    va_start( marker, fmt );
    vprintf( fmt, marker );
    va_end( marker );
    puts( "" );
}

/* end of file makehelp.c */
