/* ln.c -- create symbolic links.

   Copyright (C) 1994, 1995 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <symlink.h>
#include <ansidecl.h>
#include "ln.h"


static void EXFUN (dump_temp, (void));
static void EXFUN (copy_file, (void));

static void *EXFUN (xmalloc, (size_t bytes));
static char *EXFUN (base_name, (char *name, const char *ext));
static int EXFUN (confirm, (const char *format, ...));
static void EXFUN (run_error, (int sys, const char *format, ...));
static void EXFUN (usage, (void));

static char magic[LN_TEMP_LENGTH] = LN_TEMP_MAGIC;

static char *exec_file = 0;		/* Pointer to beginning of MAGIC. */
static char *exec_buf = 0;		/* Stores TEMPLATE in core. */
static int exec_size = 0;		/* Size of EXEC_BUF. */

static char *orig_file = 0;		/* Original file name. */
static char *link_file = 0;		/* Link file name. */

static int backup = 0;			/* Create backup file. */
static int force = 0;			/* Override existing file. */
static int interactive = 0;		/* Ask for confirmation. */
static char *suffix = "~";		/* Default backup suffix. */
static int symbolic = 0;		/* Make a symbolic link. */
static int verbose = 0;			/* Be verbose. */
static char *template = 0;		/* Template type. */

static char *prog_name;			/* How I am called. */


int
DEFUN (main, (ac, av),
int ac AND
char **av)
{
  struct stat s;
  int c;

  prog_name = base_name (av[0], ".exe");

  opterr = 0;

  while (1)
    {
      c = getopt (ac, av, "fist:v");
      if (c == -1)
	break;
      switch (c)
	{
	case 'f':
	  force = 1;
	  interactive = 0;
	  break;
	case 'i':
	  force = 0;
	  interactive = 1;
	  break;
	case 's':
	  symbolic = 1;
	  break;
	case 't':
	  template = optarg;
	  break;
	case 'v':
	  ++verbose;
	  break;
	case '?':
	default:
	  usage ();
	}
    }

  if (ac != optind + 2)
    usage ();

  orig_file = av[optind + 0];
  link_file = av[optind + 1];

  if (lstat (link_file, &s) == 0)
    {
      if (S_ISDIR (s.st_mode))
	run_error (0, "`%s' is a directory\n", link_file);

      if (interactive)
	{
	  if (!confirm ("Overwrite file `%s'? ", link_file))
	    run_error (0, 0);

	  force = 1;
	}

      if (!force)
	run_error (0, "%s: File exists\n", link_file);

      if (remove (link_file) == -1)
	run_error (1, link_file);
    }

  if (verbose)
    fprintf (stdout, "%s -> %s\n", orig_file, link_file);

  if (template)
    {
      dump_temp ();

      if (symbolic && _symlink (orig_file, link_file) == -1)
	run_error (1, link_file);
    }
  else if (symbolic)
    {
      if (symlink (orig_file, link_file) == -1)
	run_error (1, link_file);
    }
  else
    copy_file ();

  return 0;
}


static void
DEFUN_VOID (dump_temp)
{
  struct stat s;
  int c, h;
  char *p, *t;

  p = strpbrk (template, "/\\.:");
  if (!p)
    {
      t = xmalloc (strlen (LN_TEMP_FORMAT) + strlen (template) + 1);
      sprintf (t, LN_TEMP_FORMAT, template);
      template = t;
    }

  if (!p || access (template, 04) == -1)
    {
      p = getenv ("DPATH");
      if (!p)
	{
	  p = xmalloc (strlen ("DPATH") + 1 + strlen (LN_TEMP_PATH) + 1);
	  sprintf (p, "%s=%s", "DPATH", LN_TEMP_PATH);

	  if (putenv (p) == -1)
	    run_error (1, "DPATH");
	}

      t = xmalloc (_MAX_PATH);
      _searchenv (template, "DPATH", t);
      if (*t)
	template = t;
      else
	free (t);
    }

  h = open (template, O_RDONLY | O_BINARY);
  if (h == -1)
    run_error (1, template);

  c = fstat (h, &s);
  if (c == -1)
    run_error (1, template);
  if (!S_ISREG (s.st_mode))
    run_error (0, "%s: Not a regular file\n", template);

  exec_size = s.st_size;
  exec_buf = xmalloc (exec_size);

  c = read (h, exec_buf, exec_size);
  if (c != exec_size)
    run_error (1, template);

  c = close (h);
  if (c == -1)
    run_error (1, template);

  exec_file = memchr (exec_buf, *magic, exec_size);
  while (exec_file && strncmp (exec_file, magic, strlen (magic)) != 0)
    exec_file = memchr (++exec_file, *magic, exec_buf + exec_size - exec_file);

  if (!exec_file)
    {
      exec_file = magic;

      if (verbose)
	fprintf (stderr, "%s:warning: Cannot find magic string in `%s'\n",
	  prog_name, template);
    }

  memset (exec_file, 0, LN_TEMP_LENGTH);

  c = _fullpath (exec_file, orig_file, LN_TEMP_LENGTH);
  if (c == -1)
    run_error (1, orig_file);

  *exec_file = tolower (*exec_file);	/* Downcase drive letter, too. */

  c = stat (orig_file, &s);
  if (c == -1)
    run_error (1, orig_file);

  h = open (link_file, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY,
    s.st_mode & (S_IREAD | S_IWRITE));
  if (h == -1)
    run_error (1, link_file);

  c = write (h, exec_buf, exec_size);
  if (c != exec_size)
    run_error (1, link_file);

  c = close (h);
  if (c == -1)
    run_error (1, link_file);
}


#define CHUNK (64 * 1024)

static void
DEFUN_VOID (copy_file)
{
  struct stat s;
  int c, r, w;
  char *b;

  c = stat (orig_file, &s);
  if (c == -1)
    run_error (1, orig_file);

  r = open (orig_file, O_RDONLY | O_BINARY);
  if (r == -1)
    run_error (1, orig_file);

  w = open (link_file, O_WRONLY | O_CREAT | O_TRUNC | O_BINARY,
    s.st_mode & (S_IREAD | S_IWRITE));
  if (w == -1)
    run_error (1, link_file);

  b = xmalloc (CHUNK);

  while (eof (r) == 0)
    {
      c = read (r, b, CHUNK);
      if (c == -1)
	run_error (1, orig_file);

      c = write (w, b, c);
      if (c == -1)
	run_error (1, link_file);
    }

  free (b);

  c = close (r);
  if (c == -1)
    run_error (1, orig_file);

  c = close (w);
  if (c == -1)
    run_error (1, link_file);
}


static void *
DEFUN (xmalloc, (bytes),
size_t bytes)
{
  void *mem;

  mem = malloc (bytes);
  if (mem == 0)
    run_error (0, "Out of memory\n");

  return mem;
}


static char *
DEFUN (base_name, (name, ext),
char *name AND
const char *ext)
{
  char *begin, *dot;

  begin = strrchr (name, '\\');
  if (begin == 0)
    begin = strrchr (name, '/');
  if (begin == 0)
    begin = name;
  else
    ++begin;

  if (ext)
    {
      dot = strrchr (begin, *ext);
      if (dot && stricmp (dot, ext) == 0)
	*dot = 0;
    }

  return begin;
}


static int
DEFUN_VAR (confirm, (format, VA_LIST),
const char *format AND
VA_DECL)
{
  va_list a;
  int c, i;

  if (!isatty (fileno (stdin)))
    return 0;

  fputs (prog_name, stdout);
  fputs (": ", stdout);

  VA_START (a, format);
  vfprintf (stdout, format, a);
  va_end (a);

  fflush (stdout);
  fflush (stdin);

  c = fgetc (stdin);
  clearerr (stdin);

  if (c != EOF && c != '\n')
    do
      {
	i = fgetc (stdin);
	clearerr (stdin);
      }
    while (i != EOF && i != '\n');

  return (c == 'y');
}


static void
DEFUN_VAR (run_error, (sys, format, VA_LIST),
int sys AND
const char *format AND
VA_DECL)
{
  va_list a;

  if (format)
    {
      fputs (prog_name, stderr);
      fputs (": ", stderr);

      VA_START (a, format);
      vfprintf (stderr, format, a);
      va_end (a);

      if (sys)
	fprintf (stderr, ": %s\n", sys_errlist[errno]);
    }

  _exit (1);
}


static void
DEFUN_VOID (usage)
{
  printf ("Usage:  %s [-f] [-i] [-s] [-t <temp>] [-v] <orig> <dest>\n\n"
    "\t-f  remove existing destinations\n"
    "\t-i  prompt whether to remove destinations\n"
    "\t-s  make symbolic links instead of hard links\n"
    "\t-t  use <temp> as the link template (implies -s)\n"
    "\t-v  print name of each file before linking\n",
    prog_name);

  exit (1);
}
