/**
 * Copyright (c) 2014-2017 EclipseSource Muenchen GmbH and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Stefan Dirix - initial API and implementation
 */
package org.eclipse.emf.ecp.emf2web.json.generator.xtend;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.util.Arrays;
import java.util.Collection;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecp.emf2web.json.generator.xtend.JsonGenerator;
import org.eclipse.emf.ecp.emf2web.util.ReferenceHelper;
import org.eclipse.emf.ecp.view.spi.categorization.model.VAbstractCategorization;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorization;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategorizationElement;
import org.eclipse.emf.ecp.view.spi.categorization.model.VCategory;
import org.eclipse.emf.ecp.view.spi.label.model.VLabel;
import org.eclipse.emf.ecp.view.spi.model.VContainedElement;
import org.eclipse.emf.ecp.view.spi.model.VContainer;
import org.eclipse.emf.ecp.view.spi.model.VControl;
import org.eclipse.emf.ecp.view.spi.model.VDomainModelReference;
import org.eclipse.emf.ecp.view.spi.model.VElement;
import org.eclipse.emf.ecp.view.spi.model.VView;

/**
 * The class which handles the conversion from ecore files to qbForm files.
 */
@SuppressWarnings("all")
public class FormsJsonGenerator extends JsonGenerator {
  private final static String TYPE = "type";
  
  private final static String ELEMENTS = "elements";
  
  private final static String CONTROL = "Control";
  
  private final static String CATEGORIZATION = "Categorization";
  
  private final static String CATEGORY = "Category";
  
  private final static String SCOPE = "scope";
  
  private final static String LABEL = "label";
  
  private ReferenceHelper refHelper;
  
  public FormsJsonGenerator(final ReferenceHelper refHelper) {
    this.refHelper = refHelper;
  }
  
  public JsonElement createJsonElement(final EObject object) {
    return this.createJsonFormsElement(object);
  }
  
  private JsonElement _createJsonFormsElement(final EObject object) {
    throw new UnsupportedOperationException(
      "Cannot create a JSON Forms element for EObjects that are not instanceof VView, VControl or VContainer.");
  }
  
  private JsonElement _createJsonFormsElement(final VLabel label) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObj = new JsonObject();
      this.withType(jsonObj, "Label");
      String _name = label.getName();
      _xblockexpression = this.with(jsonObj, "text", _name);
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VCategorizationElement categorizationElement) {
    JsonObject _jsonObject = new JsonObject();
    JsonObject _withType = this.withType(_jsonObject, FormsJsonGenerator.CATEGORIZATION);
    EList<VAbstractCategorization> _categorizations = categorizationElement.getCategorizations();
    return this.withElements(_withType, _categorizations);
  }
  
  private JsonElement _createJsonFormsElement(final VCategorization categorization) {
    JsonObject _jsonObject = new JsonObject();
    JsonObject _withType = this.withType(_jsonObject, FormsJsonGenerator.CATEGORIZATION);
    String _label = categorization.getLabel();
    JsonObject _withLabel = this.withLabel(_withType, _label);
    EList<VAbstractCategorization> _categorizations = categorization.getCategorizations();
    return this.withElements(_withLabel, _categorizations);
  }
  
  private JsonElement _createJsonFormsElement(final VCategory category) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObj = new JsonObject();
      this.withType(jsonObj, FormsJsonGenerator.CATEGORY);
      String _label = category.getLabel();
      this.withLabel(jsonObj, _label);
      final VContainedElement contained = category.getComposite();
      JsonObject _switchResult = null;
      boolean _matched = false;
      if (!_matched) {
        if (contained instanceof VContainer) {
          _matched=true;
          EList<VContainedElement> _children = ((VContainer)contained).getChildren();
          _switchResult = this.withElements(jsonObj, _children);
        }
      }
      if (!_matched) {
        throw new UnsupportedOperationException("Category must contain a VContainer element");
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VView view) {
    JsonElement _switchResult = null;
    EList<VContainedElement> _children = view.getChildren();
    int _size = _children.size();
    switch (_size) {
      case 0:
        _switchResult = new JsonObject();
        break;
      case 1:
        EList<VContainedElement> _children_1 = view.getChildren();
        VContainedElement _get = _children_1.get(0);
        _switchResult = this.createJsonFormsElement(_get);
        break;
      default:
        EList<VContainedElement> _children_2 = view.getChildren();
        _switchResult = this.createJsonFormsElement(_children_2);
        break;
    }
    return _switchResult;
  }
  
  private JsonElement _createJsonFormsElement(final VControl control) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      this.withType(jsonObject, FormsJsonGenerator.CONTROL);
      String _displayLabel = this.getDisplayLabel(control);
      this.withLabel(jsonObject, _displayLabel);
      String _ref = this.getRef(control);
      _xblockexpression = this.withScope(jsonObject, _ref);
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final VContainer container) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      String _type = this.getType(container);
      this.withType(jsonObject, _type);
      String _name = container.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        String _label = container.getLabel();
        String _displayLabel = this.getDisplayLabel(_label);
        this.withLabel(jsonObject, _displayLabel);
      }
      EList<VContainedElement> _children = container.getChildren();
      _xblockexpression = this.withElements(jsonObject, _children);
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonFormsElement(final Collection<? extends VElement> elements) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      _xblockexpression = this.withVerticalLayout(jsonObject, elements);
    }
    return _xblockexpression;
  }
  
  private JsonObject withType(final JsonObject jsonObject, final String type) {
    return this.with(jsonObject, FormsJsonGenerator.TYPE, type);
  }
  
  private JsonObject withScope(final JsonObject jsonObject, final String ref) {
    return this.with(jsonObject, FormsJsonGenerator.SCOPE, ref);
  }
  
  private JsonObject withLabel(final JsonObject jsonObject, final String label) {
    return this.with(jsonObject, FormsJsonGenerator.LABEL, label);
  }
  
  private JsonObject withElements(final JsonObject jsonObject, final Collection<? extends VElement> elements) {
    return this.with(jsonObject, FormsJsonGenerator.ELEMENTS, elements);
  }
  
  private JsonObject withVerticalLayout(final JsonObject jsonObject, final Collection<? extends VElement> elements) {
    JsonObject _xblockexpression = null;
    {
      this.withType(jsonObject, "VerticalLayout");
      _xblockexpression = this.withElements(jsonObject, elements);
    }
    return _xblockexpression;
  }
  
  private String getType(final VElement vElement) {
    EClass _eClass = vElement.eClass();
    return _eClass.getName();
  }
  
  private String getRef(final VControl control) {
    VDomainModelReference _domainModelReference = control.getDomainModelReference();
    return this.refHelper.getStringRepresentation(_domainModelReference);
  }
  
  private String getDisplayLabel(final VControl control) {
    VDomainModelReference _domainModelReference = control.getDomainModelReference();
    return this.refHelper.getLabel(_domainModelReference);
  }
  
  private String getDisplayLabel(final String containerLabel) {
    String _xblockexpression = null;
    {
      String _trim = containerLabel.trim();
      boolean _startsWith = _trim.startsWith("%");
      if (_startsWith) {
        String _trim_1 = containerLabel.trim();
        String label = _trim_1.substring(1);
        final String[] split = label.split("(?<!(^|[A-Z]))(?=[A-Z])|(?<!^)(?=[A-Z][a-z])");
        String _get = split[0];
        int _length = _get.length();
        boolean _greaterThan = (_length > 0);
        if (_greaterThan) {
          String _get_1 = split[0];
          final char[] charArray = _get_1.toCharArray();
          char _get_2 = charArray[0];
          char _upperCase = Character.toUpperCase(_get_2);
          charArray[0] = _upperCase;
          String _string = new String(charArray);
          split[0] = _string;
        }
        final StringBuilder sb = new StringBuilder();
        for (final String str : split) {
          {
            sb.append(str);
            sb.append(" ");
          }
        }
        String _string_1 = sb.toString();
        return _string_1.trim();
      }
      _xblockexpression = containerLabel;
    }
    return _xblockexpression;
  }
  
  private JsonElement createJsonFormsElement(final Object categorization) {
    if (categorization instanceof VCategorization) {
      return _createJsonFormsElement((VCategorization)categorization);
    } else if (categorization instanceof VCategory) {
      return _createJsonFormsElement((VCategory)categorization);
    } else if (categorization instanceof VCategorizationElement) {
      return _createJsonFormsElement((VCategorizationElement)categorization);
    } else if (categorization instanceof VLabel) {
      return _createJsonFormsElement((VLabel)categorization);
    } else if (categorization instanceof VControl) {
      return _createJsonFormsElement((VControl)categorization);
    } else if (categorization instanceof VContainer) {
      return _createJsonFormsElement((VContainer)categorization);
    } else if (categorization instanceof VView) {
      return _createJsonFormsElement((VView)categorization);
    } else if (categorization instanceof Collection) {
      return _createJsonFormsElement((Collection<? extends VElement>)categorization);
    } else if (categorization instanceof EObject) {
      return _createJsonFormsElement((EObject)categorization);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(categorization).toString());
    }
  }
}
