/**
 * Copyright (c) 2014-2015 EclipseSource Muenchen GmbH and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Stefan Dirix - initial API and implementation
 * Philip Langer - re-implementation based on Gson
 * Florian Zoubek - bug fixing
 */
package org.eclipse.emf.ecp.emf2web.json.generator.xtend;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecp.emf2web.json.generator.xtend.JsonGenerator;
import org.eclipse.emf.ecp.emf2web.util.xtend.TypeMapper;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * The class which handles the conversion from ecore files to qbForm files.
 */
@SuppressWarnings("all")
public class EcoreJsonGenerator extends JsonGenerator {
  private final static String TYPE = "type";
  
  private final static String OBJECT = "object";
  
  private final static String REQUIRED = "required";
  
  private final static String PROPERTIES = "properties";
  
  private final static String ADDITIONAL_PROPERTIES = "additionalProperties";
  
  private final HashSet<EClass> visitedClasses = new HashSet<EClass>();
  
  public JsonElement createJsonElement(final EObject object) {
    JsonElement _xsynchronizedexpression = null;
    synchronized (this.visitedClasses) {
      JsonElement _xblockexpression = null;
      {
        this.visitedClasses.clear();
        _xblockexpression = this.createJsonSchemaElement(object);
      }
      _xsynchronizedexpression = _xblockexpression;
    }
    return _xsynchronizedexpression;
  }
  
  private JsonElement _createJsonSchemaElement(final EClass eClass) {
    JsonObject _xblockexpression = null;
    {
      this.visitedClasses.add(eClass);
      JsonObject _jsonObject = new JsonObject();
      final JsonObject jsonObject = this.withObjectType(_jsonObject);
      EList<EStructuralFeature> _eAllStructuralFeatures = eClass.getEAllStructuralFeatures();
      this.withProperties(jsonObject, _eAllStructuralFeatures);
      this.with(jsonObject, EcoreJsonGenerator.ADDITIONAL_PROPERTIES, Boolean.valueOf(false));
      EList<EStructuralFeature> _eAllStructuralFeatures_1 = eClass.getEAllStructuralFeatures();
      final Function1<EStructuralFeature, Boolean> _function = new Function1<EStructuralFeature, Boolean>() {
        public Boolean apply(final EStructuralFeature it) {
          return Boolean.valueOf(it.isRequired());
        }
      };
      Iterable<EStructuralFeature> _filter = IterableExtensions.<EStructuralFeature>filter(_eAllStructuralFeatures_1, _function);
      _xblockexpression = this.withRequiredProperties(jsonObject, _filter);
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonSchemaElement(final EAttribute attribute) {
    JsonObject _jsonObject = new JsonObject();
    EClassifier _eType = attribute.getEType();
    int _upperBound = attribute.getUpperBound();
    return this.withTypeProperties(_jsonObject, _eType, _upperBound);
  }
  
  private JsonObject withTypeProperties(final JsonObject jsonObject, final EClassifier eClassifier, final int upper) {
    JsonObject _xblockexpression = null;
    {
      if (((upper > 1) || (upper == (-1)))) {
        this.withType(jsonObject, "array");
        JsonObject _jsonObject = new JsonObject();
        JsonObject _withTypeProperties = this.withTypeProperties(_jsonObject, eClassifier);
        this.with(jsonObject, "items", _withTypeProperties);
      } else {
        this.withTypeProperties(jsonObject, eClassifier);
      }
      _xblockexpression = jsonObject;
    }
    return _xblockexpression;
  }
  
  private JsonObject withTypeProperties(final JsonObject jsonObject, final EClassifier eClassifier) {
    JsonObject _xblockexpression = null;
    {
      String _jsonType = this.jsonType(eClassifier);
      this.withType(jsonObject, _jsonType);
      boolean _isDateType = TypeMapper.isDateType(eClassifier);
      if (_isDateType) {
        this.with(jsonObject, "format", "date-time");
      } else {
        boolean _isEnumType = TypeMapper.isEnumType(eClassifier);
        if (_isEnumType) {
          final EEnum eEnum = ((EEnum) eClassifier);
          final JsonArray literalArray = new JsonArray();
          EList<EEnumLiteral> _eLiterals = eEnum.getELiterals();
          final Function1<EEnumLiteral, String> _function = new Function1<EEnumLiteral, String>() {
            public String apply(final EEnumLiteral it) {
              return it.getName();
            }
          };
          List<String> _map = ListExtensions.<EEnumLiteral, String>map(_eLiterals, _function);
          for (final String name : _map) {
            JsonPrimitive _jsonPrimitive = new JsonPrimitive(name);
            literalArray.add(_jsonPrimitive);
          }
          this.with(jsonObject, "enum", literalArray);
        }
      }
      _xblockexpression = jsonObject;
    }
    return _xblockexpression;
  }
  
  private String jsonType(final EClassifier eClassifier) {
    String _switchResult = null;
    boolean _matched = false;
    if (!_matched) {
      boolean _isBooleanType = TypeMapper.isBooleanType(eClassifier);
      if (_isBooleanType) {
        _matched=true;
        _switchResult = "boolean";
      }
    }
    if (!_matched) {
      boolean _isIntegerType = TypeMapper.isIntegerType(eClassifier);
      if (_isIntegerType) {
        _matched=true;
        _switchResult = "integer";
      }
    }
    if (!_matched) {
      boolean _isNumberType = TypeMapper.isNumberType(eClassifier);
      if (_isNumberType) {
        _matched=true;
        _switchResult = "number";
      }
    }
    if (!_matched) {
      _switchResult = "string";
    }
    return _switchResult;
  }
  
  private JsonElement _createJsonSchemaElement(final EReference reference) {
    JsonElement _xblockexpression = null;
    {
      final JsonObject jsonObject = new JsonObject();
      JsonElement _xifexpression = null;
      boolean _or = false;
      int _upperBound = reference.getUpperBound();
      boolean _greaterThan = (_upperBound > 1);
      if (_greaterThan) {
        _or = true;
      } else {
        int _upperBound_1 = reference.getUpperBound();
        boolean _equals = (_upperBound_1 == (-1));
        _or = _equals;
      }
      if (_or) {
        JsonObject _xblockexpression_1 = null;
        {
          this.withType(jsonObject, "array");
          EClass _eReferenceType = reference.getEReferenceType();
          JsonElement _createJsonSchemaElement = this.createJsonSchemaElement(_eReferenceType);
          _xblockexpression_1 = this.with(jsonObject, "items", _createJsonSchemaElement);
        }
        _xifexpression = _xblockexpression_1;
      } else {
        EClass _eReferenceType = reference.getEReferenceType();
        _xifexpression = this.createJsonSchemaElement(_eReferenceType);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private JsonElement _createJsonSchemaElement(final EObject eObject) {
    throw new UnsupportedOperationException(
      ("Cannot create a Json Schema element for EObjects that are not " + 
        "EClasses, EEnums, EAttributes, or EReferences."));
  }
  
  private JsonObject withObjectType(final JsonObject object) {
    return this.withType(object, EcoreJsonGenerator.OBJECT);
  }
  
  private JsonObject withType(final JsonObject jsonObject, final String type) {
    return this.with(jsonObject, EcoreJsonGenerator.TYPE, type);
  }
  
  private JsonObject withRequiredProperties(final JsonObject jsonObject, final Iterable<EStructuralFeature> requiredProperties) {
    JsonObject _xblockexpression = null;
    {
      final JsonArray requiredPropertiesArray = new JsonArray();
      final Function1<EStructuralFeature, String> _function = new Function1<EStructuralFeature, String>() {
        public String apply(final EStructuralFeature it) {
          return it.getName();
        }
      };
      Iterable<String> _map = IterableExtensions.<EStructuralFeature, String>map(requiredProperties, _function);
      for (final String name : _map) {
        JsonPrimitive _jsonPrimitive = new JsonPrimitive(name);
        requiredPropertiesArray.add(_jsonPrimitive);
      }
      boolean _isEmpty = IterableExtensions.isEmpty(requiredPropertiesArray);
      boolean _not = (!_isEmpty);
      if (_not) {
        this.with(jsonObject, EcoreJsonGenerator.REQUIRED, requiredPropertiesArray);
      }
      _xblockexpression = jsonObject;
    }
    return _xblockexpression;
  }
  
  private JsonObject withProperties(final JsonObject jsonObject, final Collection<? extends EStructuralFeature> features) {
    JsonObject _xblockexpression = null;
    {
      final JsonObject propertyObject = new JsonObject();
      for (final EStructuralFeature feature : features) {
        boolean _isNotCircle = this.isNotCircle(feature);
        if (_isNotCircle) {
          final JsonElement jsonElement = this.createJsonSchemaElement(feature);
          String _name = feature.getName();
          propertyObject.add(_name, jsonElement);
        }
      }
      _xblockexpression = this.with(jsonObject, EcoreJsonGenerator.PROPERTIES, propertyObject);
    }
    return _xblockexpression;
  }
  
  private boolean isNotCircle(final EStructuralFeature feature) {
    boolean _xblockexpression = false;
    {
      if ((feature instanceof EReference)) {
        EClassifier _eType = ((EReference)feature).getEType();
        boolean _contains = this.visitedClasses.contains(_eType);
        return (!_contains);
      }
      _xblockexpression = true;
    }
    return _xblockexpression;
  }
  
  private JsonElement createJsonSchemaElement(final EObject attribute) {
    if (attribute instanceof EAttribute) {
      return _createJsonSchemaElement((EAttribute)attribute);
    } else if (attribute instanceof EReference) {
      return _createJsonSchemaElement((EReference)attribute);
    } else if (attribute instanceof EClass) {
      return _createJsonSchemaElement((EClass)attribute);
    } else if (attribute != null) {
      return _createJsonSchemaElement(attribute);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(attribute).toString());
    }
  }
}
