/*******************************************************************************
 * Copyright (c) 2008-2011 Chair for Applied Software Engineering,
 * Technische Universitaet Muenchen.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Otto von Wesendonk - initial API and implementation
 ******************************************************************************/
package org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.conflicts;

import java.text.MessageFormat;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.DecisionManager;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.ConflictContext;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.ConflictDescription;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.ConflictOption;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.ConflictOption.OptionType;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.conflict.VisualConflict;
import org.eclipse.emf.emfstore.internal.client.model.changeTracking.merging.util.DecisionUtil;
import org.eclipse.emf.emfstore.internal.server.conflictDetection.ConflictBucket;

/**
 * Conflict with an {@link org.eclipse.emf.emfstore.internal.server.model.versioning.operations.CreateDeleteOperation}
 * involved.
 *
 * @author wesendon
 */
public class DeletionConflict extends VisualConflict {

	/**
	 * Default constructor.
	 *
	 * @param conflictBucket the conflict
	 * @param meCausingDelete true, if deleting operation was generated by merging user
	 * @param decisionManager decisionmanager
	 */
	public DeletionConflict(ConflictBucket conflictBucket, boolean meCausingDelete,
		DecisionManager decisionManager) {
		super(conflictBucket, decisionManager, meCausingDelete,
			false);
		init();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ConflictContext initConflictContext() {
		return new ConflictContext(getDecisionManager(), getLeftOperation(), getTheirOperation());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ConflictDescription initConflictDescription(ConflictDescription description) {
		if (isLeftMy()) {
			description.setDescription(DecisionUtil.getDescription("deletionconflict.my", getDecisionManager() //$NON-NLS-1$
				.isBranchMerge()));
		} else {
			description.setDescription(DecisionUtil.getDescription("deletionconflict.their", getDecisionManager() //$NON-NLS-1$
				.isBranchMerge()));
		}

		description.add("modelelement", getLeftOperation().getModelElementId()); //$NON-NLS-1$
		description.add("firstother", getRightOperation().getModelElementId()); //$NON-NLS-1$
		description.add("otherinvolved", generateOthers()); //$NON-NLS-1$
		description.setImage("delete.gif"); //$NON-NLS-1$

		return description;
	}

	private String generateOthers() {
		if (getRightOperations().size() > 1) {
			return MessageFormat.format(Messages.DeletionConflict_AndOtherElements, getRightOperations().size() - 1);
		}
		return StringUtils.EMPTY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void initConflictOptions(List<ConflictOption> options) {
		final ConflictOption myOption = new ConflictOption(StringUtils.EMPTY, OptionType.MyOperation);
		myOption.addOperations(getMyOperations());
		final ConflictOption theirOption = new ConflictOption(StringUtils.EMPTY, OptionType.TheirOperation);
		theirOption.addOperations(getTheirOperations());

		if (isLeftMy()) {
			myOption.setOptionLabel(deleteMsg());
			theirOption.setOptionLabel(keepMsg());
		} else {
			myOption.setOptionLabel(keepMsg());
			theirOption.setOptionLabel(deleteMsg());
		}

		options.add(myOption);
		options.add(theirOption);
	}

	private String keepMsg() {
		final EObject modelElement = getDecisionManager().getModelElement(getRightOperation().getModelElementId());
		String result = Messages.DeletionConflict_Recover + DecisionUtil.getClassAndName(modelElement);
		result += generateOthers();
		return result;
	}

	private String deleteMsg() {
		final EObject modelElement = getDecisionManager().getModelElement(getLeftOperation().getModelElementId());
		return Messages.DeletionConflict_Delete + DecisionUtil.getClassAndName(modelElement);
	}
}
