/*******************************************************************************
 * Copyright (c) 2014 E.D.Willink and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     R.Dvorak and others - QVTo debugger framework
 *     E.D.Willink - revised API for OCL/QVTi debugger framework
 *******************************************************************************/
package org.eclipse.qvtd.debug.ui.launching;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.qvtd.debug.launching.QVTiLaunchConstants;
import org.eclipse.qvtd.debug.ui.QVTdDebugUIPlugin;
import org.eclipse.qvtd.pivot.qvtbase.Domain;
import org.eclipse.qvtd.pivot.qvtbase.Rule;
import org.eclipse.qvtd.pivot.qvtbase.Transformation;
import org.eclipse.qvtd.pivot.qvtbase.TypedModel;
import org.eclipse.qvtd.pivot.qvtcorebase.BottomPattern;
import org.eclipse.qvtd.pivot.qvtcorebase.CoreDomain;
import org.eclipse.qvtd.pivot.qvtimperative.Mapping;
import org.eclipse.qvtd.pivot.qvtimperative.evaluation.QVTiEnvironmentFactory;
import org.eclipse.qvtd.xtext.qvtimperative.utilities.QVTiXtextEvaluator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;

public class MainTab extends AbstractMainTab implements QVTiLaunchConstants
{
	protected class TransformationModifyListener implements ModifyListener
	{
		@Override
		public void modifyText(ModifyEvent e) {
/*			if (modelPath.isDisposed()) {
				return;
			}
			String modelName = modelPath.getText();
			@SuppressWarnings("null")@NonNull URI contextURI = URI.createURI(modelName, true);
			URI elementsURI = contextURI.trimFragment();
			try {
				Resource resource = getMetaModelManager().getExternalResourceSet().getResource(elementsURI, true);
		        if (resource == null) {
		        	throw new IOException("There was an error loading the model file. ");
		        }
				List<String> elements = new ArrayList<String>();
				for (TreeIterator<EObject> tit = resource.getAllContents(); tit.hasNext(); ) {
					EObject eObject = tit.next();
					String displayString = DomainUtil.getLabel(eObject);
					URI uri = EcoreUtil.getURI(eObject);
					elements.add(displayString);
					element2uri.put(displayString, uri);
					}
				Collections.sort(elements);
				elementCombo.setItems(elements.toArray(new String[elements.size()]));
			}
			catch (Exception ex) {
				setErrorMessage("Failed to load '" + elementsURI + "': " + ex.toString());
			}
			if (!initializing) {
				updateLaunchConfigurationDialog();
			} */
			if (txPath.isDisposed()) {
				return;
			}
			String txName = txPath.getText();
			@SuppressWarnings("null")@NonNull URI txURI = URI.createURI(txName, true);
			try {
				@SuppressWarnings("null")@NonNull Group inputsGroup2 = inputsGroup;
				@SuppressWarnings("null")@NonNull Group outputsGroup2 = outputsGroup;
				QVTiEnvironmentFactory envFactory = new QVTiEnvironmentFactory(null, getMetaModelManager());
				QVTiXtextEvaluator xtextEvaluator = new QVTiXtextEvaluator(envFactory, txURI);
				Transformation transformation = xtextEvaluator.getTransformation();
				Set<TypedModel> inputs = new HashSet<TypedModel>();
				Set<TypedModel> outputs = new HashSet<TypedModel>();
				Map<String, String> inputMap = new HashMap<String, String>();
				Map<String, String> outputMap = new HashMap<String, String>();
				for (Rule rule : transformation.getRule()) {
					if (rule instanceof Mapping) {
						Mapping mapping = (Mapping)rule;
						for (Domain domain : mapping.getDomain()) {
							if (domain instanceof CoreDomain) {
								CoreDomain coreDomain = (CoreDomain)domain;
								BottomPattern bottomPattern = coreDomain.getBottomPattern();
								TypedModel typedModel = coreDomain.getTypedModel();
								String name = typedModel.getName();
								if (bottomPattern.getRealizedVariable().isEmpty()) {
									if (inputs.add(typedModel)) {
										inputMap.put(name, getDefaultPath(inputsGroup2, name));
									}
								}
								else {
									if (outputs.add(typedModel)) {
										outputMap.put(name, getDefaultPath(outputsGroup2, name));
									}
								}
							}
						}
					}
				}
				for (String key : outputMap.keySet()) {
					inputMap.remove(key);
				}
				updateParametersGroup(inputsGroup2, SWT.NONE, inputMap);
				updateParametersGroup(outputsGroup2, SWT.SAVE, outputMap);
			}
			catch (Exception ex) {
				setErrorMessage("Failed to load '" + txName + "': " + ex.toString());
			}
			updateLaunchConfigurationDialog();
		}
	}

	protected Text txPath;
	protected Button txBrowseWS;
	protected Button txBrowseFile;
	protected Group inputsGroup;
	protected Group outputsGroup;

	@Override
	public boolean canSave() {
		assert !initializing;
		ResourceSet resourceSet = getMetaModelManager().getExternalResourceSet();
		URIConverter uriConverter = resourceSet.getURIConverter();
		String txName = txPath.getText();
		URI txURI = URI.createURI(txName, true);
		boolean txExists = uriConverter.exists(txURI, null);
		if (!txExists){
			setErrorMessage("Selected file " + txName + " does not exist");
			return false;
		}
		setErrorMessage(null);
		return true;			
	}

	@SuppressWarnings("null")
	public void createControl(Composite parent) {
		Composite control = createForm(parent);
		txPath.addModifyListener(new TransformationModifyListener());
		LaunchingUtils.prepareBrowseWorkspaceButton(txBrowseWS, txPath, false);
		LaunchingUtils.prepareBrowseFileSystemButton(txBrowseFile, txPath, false);
		updateParametersGroup(inputsGroup, SWT.NONE, EMPTY_MAP);
		updateParametersGroup(outputsGroup, SWT.SAVE, EMPTY_MAP);
		control.setBounds(0, 0, 300, 300);
		control.layout();
		control.pack();
	}

	/**
	 * @wbp.parser.entryPoint
	 */
	public Composite createForm(Composite parent) {
		Composite control = new Composite(parent, SWT.NONE);
		setControl(control);
		control.setLayout(new GridLayout(1, false));
		control.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));

		Group txGroup = new Group(control, SWT.NONE);
		txGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		txGroup.setText("Transformation");
		txGroup.setLayout(new GridLayout(3, false));

		txPath = new Text(txGroup, SWT.BORDER);
		GridData gd_txPath = new GridData(SWT.FILL, SWT.CENTER, true, false, 1, 1);
		gd_txPath.minimumWidth = 100;
		txPath.setLayoutData(gd_txPath);
		txBrowseWS = new Button(txGroup, SWT.NONE);
		txBrowseWS.setText("Browse Workspace...");
		txBrowseFile = new Button(txGroup, SWT.NONE);
		txBrowseFile.setText("Browse File...");

		inputsGroup = new Group(control, SWT.NONE);
		inputsGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		inputsGroup.setText("Inputs");
		inputsGroup.setLayout(new GridLayout(1, false));

		outputsGroup = new Group(control, SWT.NONE);
		outputsGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));
		outputsGroup.setText("Outputs");
		outputsGroup.setLayout(new GridLayout(1, false));
		return control;
	}

	protected String getDefaultPath(@NonNull Group group, String name) {
		if (name != null) {
			for (Control child : group.getChildren()) {
				if (child instanceof ParameterRow) {
					ParameterRow row = (ParameterRow)child;
					if (name.equals(row.name.getText())) {
						return row.path.getText();
					}
				}
			}
		}
		return "";
	}

	@Override
	public Image getImage() {
		return QVTdDebugUIPlugin.getDefault().createImage("icons/QVTiModelFile.gif");
	}

	public void initializeFrom(ILaunchConfiguration configuration) {
		assert !initializing;
		try {
			initializing = true;
			txPath.setText(configuration.getAttribute(TX_KEY, ""));
			Map<String, String> inMap = configuration.getAttribute(IN_KEY, EMPTY_MAP);
			Map<String, String> outMap = configuration.getAttribute(OUT_KEY, EMPTY_MAP);
			if (inMap != null) {
				refreshParametersGroup(inputsGroup, SWT.NONE, inMap);
			}
			if (outMap != null) {
				refreshParametersGroup(outputsGroup, SWT.SAVE, outMap);
			}
		} catch (CoreException e) {
			//Ignore
		} finally {
			initializing = false;
			updateLaunchConfigurationDialog();
		}
	}

	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(TX_KEY, txPath.getText());
		Map<String, String> inputMap = new HashMap<String, String>();
		for (Control child : inputsGroup.getChildren()) {
			if (child instanceof ParameterRow) {
				ParameterRow row = (ParameterRow)child;
				inputMap.put(row.name.getText(), row.path.getText());
			}
		}
		configuration.setAttribute(IN_KEY, inputMap);
		Map<String, String> outputMap = new HashMap<String, String>();
		for (Control child : outputsGroup.getChildren()) {
			if (child instanceof ParameterRow) {
				ParameterRow row = (ParameterRow)child;
				outputMap.put(row.name.getText(), row.path.getText());
			}
		}
		configuration.setAttribute(OUT_KEY, outputMap);
	}

	protected void refreshParametersGroup(@NonNull Group group, int style, @NonNull Map<String, String> map) {
		List<String> keys = new ArrayList<String>(map.keySet());
		Collections.sort(keys);
		Control[] children = group.getChildren();
		for (int i = 0; i < children.length; i++) {
			ParameterRow row = (ParameterRow)children[i];
			String text = row.name.getText();
			String path = map.get(text);
			if (path != null) {
				row.path.setText(path);
			}
		}
		group.layout();
	}

	protected void setDefaults(@NonNull ILaunchConfigurationWorkingCopy configuration, @NonNull IFile iFile) {
		configuration.setAttribute(TX_KEY, iFile.getFullPath().toString());
		configuration.setAttribute(IN_KEY, EMPTY_MAP);
		configuration.setAttribute(OUT_KEY, EMPTY_MAP);
	}

	protected void updateParametersGroup(@NonNull Group group, int style, @NonNull Map<String, String> map) {
		List<String> keys = new ArrayList<String>(map.keySet());
		Collections.sort(keys);
		Control[] children = group.getChildren();
		int iMax = Math.min(children.length, keys.size());
		int i = 0;
		for (; i < iMax; i++) {
			ParameterRow row = (ParameterRow)children[i];
			String string = keys.get(i);
			row.name.setText(string);
			row.path.setText(map.get(string));
		}
		for (; i < keys.size(); i++) {
			@SuppressWarnings("null")@NonNull String string = keys.get(i);
			@SuppressWarnings("null")@NonNull String parameterPath = map.get(string);
			new ParameterRow(this, group, style, string, parameterPath);
		}
		for (; i < children.length; i++) {
			children[i].dispose();
		}
		group.layout();
	}
}
