/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.geo;

import java.util.function.Function;
import org.apache.lucene.geo.GeoUtils;
import org.apache.lucene.geo.Polygon;
import org.apache.lucene.geo.Polygon2D;
import org.apache.lucene.geo.Rectangle;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.util.NumericUtils;
import org.apache.lucene.util.SloppyMath;

public final class GeoEncodingUtils {
    public static final short BITS = 32;
    private static final double LAT_SCALE = 2.3860929422222223E7;
    private static final double LAT_DECODE = 4.190951585769653E-8;
    private static final double LON_SCALE = 1.1930464711111112E7;
    private static final double LON_DECODE = 8.381903171539307E-8;

    private GeoEncodingUtils() {
    }

    public static int encodeLatitude(double latitude) {
        GeoUtils.checkLatitude(latitude);
        if (latitude == 90.0) {
            latitude = Math.nextDown(latitude);
        }
        return (int)Math.floor(latitude / 4.190951585769653E-8);
    }

    public static int encodeLatitudeCeil(double latitude) {
        GeoUtils.checkLatitude(latitude);
        if (latitude == 90.0) {
            latitude = Math.nextDown(latitude);
        }
        return (int)Math.ceil(latitude / 4.190951585769653E-8);
    }

    public static int encodeLongitude(double longitude) {
        GeoUtils.checkLongitude(longitude);
        if (longitude == 180.0) {
            longitude = Math.nextDown(longitude);
        }
        return (int)Math.floor(longitude / 8.381903171539307E-8);
    }

    public static int encodeLongitudeCeil(double longitude) {
        GeoUtils.checkLongitude(longitude);
        if (longitude == 180.0) {
            longitude = Math.nextDown(longitude);
        }
        return (int)Math.ceil(longitude / 8.381903171539307E-8);
    }

    public static double decodeLatitude(int encoded) {
        double result = (double)encoded * 4.190951585769653E-8;
        assert (result >= -90.0 && result < 90.0);
        return result;
    }

    public static double decodeLatitude(byte[] src, int offset) {
        return GeoEncodingUtils.decodeLatitude(NumericUtils.sortableBytesToInt(src, offset));
    }

    public static double decodeLongitude(int encoded) {
        double result = (double)encoded * 8.381903171539307E-8;
        assert (result >= -180.0 && result < 180.0);
        return result;
    }

    public static double decodeLongitude(byte[] src, int offset) {
        return GeoEncodingUtils.decodeLongitude(NumericUtils.sortableBytesToInt(src, offset));
    }

    public static DistancePredicate createDistancePredicate(double lat, double lon, double radiusMeters) {
        Rectangle boundingBox = Rectangle.fromPointDistance(lat, lon, radiusMeters);
        double axisLat = Rectangle.axisLat(lat, radiusMeters);
        double distanceSortKey = GeoUtils.distanceQuerySortKey(radiusMeters);
        Function<Rectangle, PointValues.Relation> boxToRelation = box -> GeoUtils.relate(box.minLat, box.maxLat, box.minLon, box.maxLon, lat, lon, distanceSortKey, axisLat);
        Grid subBoxes = GeoEncodingUtils.createSubBoxes(boundingBox, boxToRelation);
        return new DistancePredicate(subBoxes.latShift, subBoxes.lonShift, subBoxes.latBase, subBoxes.lonBase, subBoxes.maxLatDelta, subBoxes.maxLonDelta, subBoxes.relations, lat, lon, distanceSortKey);
    }

    public static PolygonPredicate createPolygonPredicate(Polygon[] polygons, Polygon2D tree) {
        Rectangle boundingBox = Rectangle.fromPolygon(polygons);
        Function<Rectangle, PointValues.Relation> boxToRelation = box -> tree.relate(box.minLat, box.maxLat, box.minLon, box.maxLon);
        Grid subBoxes = GeoEncodingUtils.createSubBoxes(boundingBox, boxToRelation);
        return new PolygonPredicate(subBoxes.latShift, subBoxes.lonShift, subBoxes.latBase, subBoxes.lonBase, subBoxes.maxLatDelta, subBoxes.maxLonDelta, subBoxes.relations, tree);
    }

    private static Grid createSubBoxes(Rectangle boundingBox, Function<Rectangle, PointValues.Relation> boxToRelation) {
        int minLat = GeoEncodingUtils.encodeLatitudeCeil(boundingBox.minLat);
        int maxLat = GeoEncodingUtils.encodeLatitude(boundingBox.maxLat);
        int minLon = GeoEncodingUtils.encodeLongitudeCeil(boundingBox.minLon);
        int maxLon = GeoEncodingUtils.encodeLongitude(boundingBox.maxLon);
        if (maxLat < minLat || !boundingBox.crossesDateline() && maxLon < minLon) {
            return new Grid(1, 1, 0, 0, 0, 0, new byte[0]);
        }
        long minLat2 = (long)minLat - Integer.MIN_VALUE;
        long maxLat2 = (long)maxLat - Integer.MIN_VALUE;
        int latShift = GeoEncodingUtils.computeShift(minLat2, maxLat2);
        int latBase = (int)(minLat2 >>> latShift);
        int maxLatDelta = (int)(maxLat2 >>> latShift) - latBase + 1;
        assert (maxLatDelta > 0);
        long minLon2 = (long)minLon - Integer.MIN_VALUE;
        long maxLon2 = (long)maxLon - Integer.MIN_VALUE;
        if (boundingBox.crossesDateline()) {
            maxLon2 += 0x100000000L;
        }
        int lonShift = GeoEncodingUtils.computeShift(minLon2, maxLon2);
        int lonBase = (int)(minLon2 >>> lonShift);
        int maxLonDelta = (int)(maxLon2 >>> lonShift) - lonBase + 1;
        assert (maxLonDelta > 0);
        byte[] relations = new byte[maxLatDelta * maxLonDelta];
        for (int i = 0; i < maxLatDelta; ++i) {
            for (int j = 0; j < maxLonDelta; ++j) {
                int boxMinLat = (latBase + i << latShift) + Integer.MIN_VALUE;
                int boxMinLon = (lonBase + j << lonShift) + Integer.MIN_VALUE;
                int boxMaxLat = boxMinLat + (1 << latShift) - 1;
                int boxMaxLon = boxMinLon + (1 << lonShift) - 1;
                relations[i * maxLonDelta + j] = (byte)boxToRelation.apply(new Rectangle(GeoEncodingUtils.decodeLatitude(boxMinLat), GeoEncodingUtils.decodeLatitude(boxMaxLat), GeoEncodingUtils.decodeLongitude(boxMinLon), GeoEncodingUtils.decodeLongitude(boxMaxLon))).ordinal();
            }
        }
        return new Grid(latShift, lonShift, latBase, lonBase, maxLatDelta, maxLonDelta, relations);
    }

    private static int computeShift(long a, long b) {
        assert (a <= b);
        int shift = 1;
        long delta;
        while ((delta = (b >>> shift) - (a >>> shift)) < 0L || delta >= 64L) {
            ++shift;
        }
        return shift;
    }

    public static class DistancePredicate
    extends Grid {
        private final double lat;
        private final double lon;
        private final double distanceKey;

        private DistancePredicate(int latShift, int lonShift, int latBase, int lonBase, int maxLatDelta, int maxLonDelta, byte[] relations, double lat, double lon, double distanceKey) {
            super(latShift, lonShift, latBase, lonBase, maxLatDelta, maxLonDelta, relations);
            this.lat = lat;
            this.lon = lon;
            this.distanceKey = distanceKey;
        }

        public boolean test(int lat, int lon) {
            int lat2 = lat - Integer.MIN_VALUE >>> this.latShift;
            if (lat2 < this.latBase || lat2 >= this.latBase + this.maxLatDelta) {
                return false;
            }
            int lon2 = lon - Integer.MIN_VALUE >>> this.lonShift;
            if (lon2 < this.lonBase) {
                lon2 += 1 << 32 - this.lonShift;
            }
            assert (Integer.toUnsignedLong(lon2) >= (long)this.lonBase);
            assert (lon2 - this.lonBase >= 0);
            if (lon2 - this.lonBase >= this.maxLonDelta) {
                return false;
            }
            byte relation = this.relations[(lat2 - this.latBase) * this.maxLonDelta + (lon2 - this.lonBase)];
            if (relation == PointValues.Relation.CELL_CROSSES_QUERY.ordinal()) {
                return SloppyMath.haversinSortKey(GeoEncodingUtils.decodeLatitude(lat), GeoEncodingUtils.decodeLongitude(lon), this.lat, this.lon) <= this.distanceKey;
            }
            return relation == PointValues.Relation.CELL_INSIDE_QUERY.ordinal();
        }
    }

    private static class Grid {
        static final int ARITY = 64;
        final int latShift;
        final int lonShift;
        final int latBase;
        final int lonBase;
        final int maxLatDelta;
        final int maxLonDelta;
        final byte[] relations;

        private Grid(int latShift, int lonShift, int latBase, int lonBase, int maxLatDelta, int maxLonDelta, byte[] relations) {
            if (latShift < 1 || latShift > 31) {
                throw new IllegalArgumentException();
            }
            if (lonShift < 1 || lonShift > 31) {
                throw new IllegalArgumentException();
            }
            this.latShift = latShift;
            this.lonShift = lonShift;
            this.latBase = latBase;
            this.lonBase = lonBase;
            this.maxLatDelta = maxLatDelta;
            this.maxLonDelta = maxLonDelta;
            this.relations = relations;
        }
    }

    public static class PolygonPredicate
    extends Grid {
        private final Polygon2D tree;

        private PolygonPredicate(int latShift, int lonShift, int latBase, int lonBase, int maxLatDelta, int maxLonDelta, byte[] relations, Polygon2D tree) {
            super(latShift, lonShift, latBase, lonBase, maxLatDelta, maxLonDelta, relations);
            this.tree = tree;
        }

        public boolean test(int lat, int lon) {
            int lat2 = lat - Integer.MIN_VALUE >>> this.latShift;
            if (lat2 < this.latBase || lat2 >= this.latBase + this.maxLatDelta) {
                return false;
            }
            int lon2 = lon - Integer.MIN_VALUE >>> this.lonShift;
            if (lon2 < this.lonBase) {
                lon2 += 1 << 32 - this.lonShift;
            }
            assert (Integer.toUnsignedLong(lon2) >= (long)this.lonBase);
            assert (lon2 - this.lonBase >= 0);
            if (lon2 - this.lonBase >= this.maxLonDelta) {
                return false;
            }
            byte relation = this.relations[(lat2 - this.latBase) * this.maxLonDelta + (lon2 - this.lonBase)];
            if (relation == PointValues.Relation.CELL_CROSSES_QUERY.ordinal()) {
                return this.tree.contains(GeoEncodingUtils.decodeLatitude(lat), GeoEncodingUtils.decodeLongitude(lon));
            }
            return relation == PointValues.Relation.CELL_INSIDE_QUERY.ordinal();
        }
    }
}

