/*******************************************************************************
 * Copyright (c) 2015, 2017 Pivotal Software, Inc. and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Apache License v2.0 which accompanies this distribution. 
 * 
 * The Eclipse Public License is available at 
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * and the Apache License v2.0 is available at 
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You may elect to redistribute this code under either of these licenses.
 *  
 *  Contributors:
 *     Pivotal Software, Inc. - initial API and implementation
 *     IBM - Switching to NOT provide a default WAR package here
 *     		(default implementation is done through JavaWebApplicationDelegate)
 ********************************************************************************/
package org.eclipse.cft.server.core.internal.client;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.cloudfoundry.client.lib.CloudFoundryOperations;
import org.cloudfoundry.client.lib.UploadStatusCallback;
import org.cloudfoundry.client.lib.archive.ApplicationArchive;
import org.cloudfoundry.client.lib.domain.CloudApplication;
import org.eclipse.cft.server.core.CFApplicationArchive;
import org.eclipse.cft.server.core.internal.ApplicationAction;
import org.eclipse.cft.server.core.internal.CloudErrorUtil;
import org.eclipse.cft.server.core.internal.CloudFoundryPlugin;
import org.eclipse.cft.server.core.internal.CloudFoundryServer;
import org.eclipse.cft.server.core.internal.Messages;
import org.eclipse.cft.server.core.internal.application.ApplicationUtil;
import org.eclipse.cft.server.core.internal.application.CachingApplicationArchive;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.osgi.util.NLS;
import org.eclipse.wst.server.core.IModule;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.internal.Server;

/**
 * This action is the primary operation for uploading and starting an
 * application to a CF server.
 * <p/>
 * It does NOT publish the application if it doesn't exist in the server. It is
 * meant to start and update an application that already exists.
 * <p/>
 * 
 * Several primary steps are performed when deploying an application:
 * <p/>
 * 1. Create an archive file containing the application's resources. Incremental
 * publishing is may be used here to create an archive containing only those
 * files that have been changed.
 * <p/>
 * 2. Set local WTP module states to indicate the an application's contents have
 * been pushed (i.e. "published")
 * <p/>
 * 3. Start the application in the server, if specified by the deployment
 * configuration.
 * <p/>
 * 4. Set local WTP module states to indicate whether an application has
 * started, or is stopped if an error occurred while starting it.
 * <p/>
 * 5. Invoke callbacks to notify listeners that an application has been started.
 * One of the notification is to the CF console to display the app logs in the
 * CF console.
 * <p/>
 */
@SuppressWarnings("restriction")
public class StartOperation extends RestartOperation {

	/**
	 * 
	 */
	final protected boolean incrementalPublish;

	/**
	 * 
	 * @param waitForDeployment
	 * @param incrementalPublish
	 * @param modules
	 * @param cloudFoundryServerBehaviour
	 * @param alwaysStart if true, application will always start. if false,
	 */
	public StartOperation(CloudFoundryServerBehaviour behaviour, boolean incrementalPublish, IModule[] modules,
			boolean clearConsole) {
		super(behaviour, modules, clearConsole);
		this.incrementalPublish = incrementalPublish;
	}

	@Override
	public String getOperationName() {
		return incrementalPublish ? Messages.PushApplicationOperation_UPDATE_APP_MESSAGE
				: Messages.PushApplicationOperation_PUSH_MESSAGE;
	}

	@Override
	protected DeploymentConfiguration getDefaultDeploymentConfiguration() {
		return new DeploymentConfiguration(ApplicationAction.START);
	}

	@Override
	protected void performDeployment(CloudFoundryApplicationModule appModule, IProgressMonitor monitor)
			throws CoreException {
		final Server server = (Server) getBehaviour().getServer();
		final CloudFoundryServer cloudServer = getBehaviour().getCloudFoundryServer();

		final String deploymentName = appModule.getDeploymentInfo().getDeploymentName();

		// This request does three things:
		// 1. Checks if the application external or mapped to a local
		// project. If mapped to a local project
		// it creates an archive of the application's content
		// 2. If an archive file was created, it pushes the archive
		// file.
		// 3. While pushing the archive file, a check is made to see if
		// the application exists remotely. If not, the application is
		// created in the
		// CF server.

		IModule actualModule = getModules() != null && getModules().length > 0 ? getModules()[0] : null;
		if (actualModule != null && !actualModule.isExternal()) {

			String generatingArchiveLabel = NLS.bind(Messages.CONSOLE_GENERATING_ARCHIVE,
					appModule.getDeployedApplicationName());
			getBehaviour().printlnToConsole(appModule, generatingArchiveLabel);

			SubMonitor subMonitor = SubMonitor.convert(monitor, 100);

			subMonitor.subTask(generatingArchiveLabel);
			CFApplicationArchive applicationArchive = getBehaviour().generateApplicationArchiveFile(
					appModule.getDeploymentInfo(), appModule, getModules(), server, incrementalPublish,
					subMonitor.newChild(20));
			if (applicationArchive == null) {
				// An app archive must be always available, so if we reached
				// this point and we have none
				// then we must throw an exception.
				String message = NLS.bind(Messages.ERROR_StartOperation_UNSUPPORTED_MODULE_TYPE, new String[] {
						deploymentName, actualModule.getModuleType().getId(), cloudServer.getServer().getId() });
				throw new CoreException(new Status(IStatus.ERROR, CloudFoundryPlugin.PLUGIN_ID,
						message)); 
			}

			// Tell webtools the module has been published
			getBehaviour().resetPublishState(getModules());

			// update server publish status
			IModule[] serverModules = server.getModules();
			boolean allSynched = true;
			for (IModule serverModule : serverModules) {
				int modulePublishState = server.getModulePublishState(new IModule[] { serverModule });
				if (modulePublishState == IServer.PUBLISH_STATE_INCREMENTAL
						|| modulePublishState == IServer.PUBLISH_STATE_FULL) {
					allSynched = false;
				}
			}

			if (allSynched) {
				server.setServerPublishState(IServer.PUBLISH_STATE_NONE);
			}

			subMonitor.worked(10);

			final CFApplicationArchive applicationArchiveFin = applicationArchive;
			final CloudFoundryApplicationModule appModuleFin = appModule;
			// Now push the application resources to the server

			new BehaviourRequest<Void>(getOperationName() + " - " + deploymentName, getBehaviour()) { //$NON-NLS-1$
				@Override
				protected Void doRun(final CloudFoundryOperations client, SubMonitor progress)
						throws CoreException, OperationCanceledException {

					getBehaviour().printlnToConsole(appModuleFin, getRequestLabel());

					// Check for cancel here prior to pushing the
					// application
					if (progress.isCanceled()) {
						throw new OperationCanceledException(
								Messages.bind(Messages.OPERATION_CANCELED, getRequestLabel()));
					}
					pushApplication(client, appModuleFin, applicationArchiveFin, progress);

					CloudFoundryPlugin.trace("Application " + deploymentName //$NON-NLS-1$
							+ " pushed to Cloud Foundry server."); //$NON-NLS-1$

					cloudServer.moduleAdditionCompleted(getFirstModule());

					return null;
				}

			}.run(subMonitor.newChild(70));

			getBehaviour().printlnToConsole(appModule, Messages.CONSOLE_APP_PUSHED_MESSAGE);

		}

		super.performDeployment(appModule, monitor);
	}

	/**
	 * This performs the primary operation of creating an application and then
	 * pushing the application contents to the server. These are performed in
	 * separate requests via the CF client. If the application does not exist,
	 * it is first created through an initial request. Once the application is
	 * created, or if it already exists, the next step is to upload (push) the
	 * application archive containing the application's resources. This is
	 * performed in a second separate request.
	 * <p/>
	 * To avoid replacing the deployment info in the app module, the mapping to
	 * the most recent {@link CloudApplication} in the app module is NOT updated
	 * with newly created application. It is up to the caller to set the mapping
	 * in {@link CloudFoundryApplicationModule}
	 * @param client
	 * @param appModule valid Cloud module with valid deployment info.
	 * @param monitor
	 * @throws CoreException if error creating the application
	 */
	protected void pushApplication(CloudFoundryOperations client, final CloudFoundryApplicationModule appModule,
			CFApplicationArchive applicationArchive, final IProgressMonitor monitor) throws CoreException {
		updateIfManifestChanged(client, appModule, monitor);
		pushArchive(client, appModule, applicationArchive, monitor);
	}

	protected void pushArchive(CloudFoundryOperations client, final CloudFoundryApplicationModule appModule,
			CFApplicationArchive applicationArchive, final IProgressMonitor monitor) throws CoreException {
		String appName = appModule.getDeploymentInfo().getDeploymentName();

		// [95636410] - verify that the application actually exists.
		// Otherwise a cryptic error may be thrown and the user may not
		// know that the upload failed because the application no longer exists
		try {
			getBehaviour().getCloudApplication(appName, monitor);
		}
		catch (CoreException e) {
			if (CloudErrorUtil.isNotFoundException(e)) {
				throw CloudErrorUtil
						.toCoreException(NLS.bind(Messages.ERROR_FAILED_TO_PUSH_APP_DOES_NOT_EXIST, appName), e);
			}
			else {
				throw e;
			}
		}

		try {
			// Now push the application content.
			if (applicationArchive != null) {
				// Handle the incremental publish case separately as it
				// requires
				// a partial war file generation of only the changed
				// resources
				// AFTER
				// the server determines the list of missing file names.
				try {
					if (applicationArchive instanceof CachingApplicationArchive) {
						final CachingApplicationArchive cachingArchive = (CachingApplicationArchive) applicationArchive;
						ApplicationArchive v1ArchiveWrapper = ApplicationUtil.asV1ApplicationArchive(cachingArchive);
						client.uploadApplication(appName, v1ArchiveWrapper, new UploadStatusCallback() {

							public void onProcessMatchedResources(int length) {

							}

							public void onMatchedFileNames(Set<String> matchedFileNames) {
								cachingArchive.generatePartialWarFile(matchedFileNames);
							}

							public void onCheckResources() {

							}

							public boolean onProgress(String status) {
								return false;
							}
						});

						// Once the application has run, do a clean up of the
						// sha1
						// cache for deleted resources

					}
					else {
						ApplicationArchive v1ArchiveWrapper = ApplicationUtil
								.asV1ApplicationArchive(applicationArchive);

						client.uploadApplication(appName, v1ArchiveWrapper, new UploadStatusCallback() {

							public void onProcessMatchedResources(int length) {

							}

							public void onMatchedFileNames(Set<String> matchedFileNames) {

							}

							public void onCheckResources() {

							}

							public boolean onProgress(String status) {
								return false;
							}
						});
					}
					// Check for cancel
					if (monitor.isCanceled()) {
						throw new OperationCanceledException(
								Messages.bind(Messages.OPERATION_CANCELED, getOperationName()));
					}
				}
				finally {
					try {
						applicationArchive.close();
					}
					catch (CoreException e) {
						// Don't let errors in closing the archive stop the
						// publish operation
						CloudFoundryPlugin.logError(e);
					}
				}
			}
			else {
				throw CloudErrorUtil.toCoreException(
						"Failed to deploy application " + appModule.getDeploymentInfo().getDeploymentName() + //$NON-NLS-1$
								" since no deployable war or application archive file was generated."); //$NON-NLS-1$
			}
		}
		catch (IOException e) {
			throw new CoreException(CloudFoundryPlugin.getErrorStatus(
					"Failed to deploy application " + //$NON-NLS-1$
							appModule.getDeploymentInfo().getDeploymentName() + " due to " + e.getMessage(), //$NON-NLS-1$
					e));
		}
	}
	
	protected void updateIfManifestChanged(CloudFoundryOperations client, CloudFoundryApplicationModule appModule,
			IProgressMonitor monitor)  {
		// See if the information in the existing Cloud app has changed compared to the info in the appModule
		try {
			CloudApplication app = getBehaviour().getCloudApplication(appModule.getDeployedApplicationName(), monitor);
			if (app == null) {
				// App no longer exists. Nothing to update
				return;
			} else {
				List<String> changedProps = V1CFPropertiesUpdateFromManifest.v1UpdateFromManifest(client, getBehaviour().getCloudFoundryServer(), appModule, app, monitor);
			    if (changedProps != null && !changedProps.isEmpty()) {
					getBehaviour().printlnToConsole(appModule,
							NLS.bind(Messages.StartOperation_MANIFEST_PROPERTIES_CHANGED,
									changedProps.toString()));
			    }
			}
		}
		catch (Throwable e) {
			CloudFoundryPlugin.logError(e);
		}
	}
}
