/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.utils;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.eclipse.epp.internal.logging.aeri.ide.l10n.LogMessages.WARN_RESPONSE_UPLOAD_REPORT_FAILED;
import static org.eclipse.epp.internal.logging.aeri.ide.utils.Formats.format;

import java.net.UnknownHostException;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.epp.internal.logging.aeri.ide.IIdeFactory;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEvent;
import org.eclipse.epp.internal.logging.aeri.ide.ILogEventGroup;
import org.eclipse.epp.internal.logging.aeri.ide.IServerDescriptor;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.logging.aeri.core.IModelFactory;
import org.eclipse.epp.logging.aeri.core.IProblemState;
import org.eclipse.epp.logging.aeri.core.IServerConnection;
import org.eclipse.epp.logging.aeri.core.ProblemStatus;
import org.eclipse.epp.logging.aeri.core.util.Logs;
import org.eclipse.jdt.annotation.Nullable;

public class UploadReportsScheduler {

    @Nullable
    Job uploadJob = null;
    ConcurrentLinkedQueue<ILogEventGroup> uploadQueue = new ConcurrentLinkedQueue<>();

    public IProblemState send(ILogEvent event, IProgressMonitor monitor) {
        IServerDescriptor server = event.getServer();
        IServerConnection cx = server.getConnection();
        if (!server.isActive()) {
            // we skip reports for not yet enabled end-points! TODO should we get so far with this?
            String msg = Formats.format(Messages.RESPONSE_REPORT_UPLOADED, server.getName(), event.getStatus().getMessage());
            IProblemState response = IModelFactory.eINSTANCE.createProblemState();
            response.setStatus(ProblemStatus.FAILURE);
            response.setMessage(msg);
            event.setResponse(response);
            Logs.debug(msg);
        }
        try {
            IProblemState response = cx.submit(event.getStatus(), event.getContext(), SubMonitor.convert(monitor, 1));
            checkNotNull(response, "Server returned null response"); //$NON-NLS-1$
            event.setResponse(response);
        } catch (UnknownHostException e) {
            Logs.debug("Failed to connect to server ''{0}''", e, server.getName());
            IProblemState response = IModelFactory.eINSTANCE.createProblemState();
            response.setStatus(ProblemStatus.FAILURE);
            response.setMessage(
                    format(Messages.RESPONSE_REPORT_UPLOAD_FAILED, server.getName(), "Unknown Host Exception: " + e.getMessage()));
            event.setResponse(response);
        } catch (Exception e) {
            Logs.log(WARN_RESPONSE_UPLOAD_REPORT_FAILED, e, server.getName(), e.getMessage());
            IProblemState response = IModelFactory.eINSTANCE.createProblemState();
            response.setStatus(ProblemStatus.FAILURE);
            response.setMessage(format(Messages.RESPONSE_REPORT_UPLOAD_FAILED, server.getName(), e.getMessage()));
            event.setResponse(response);
        }
        monitor.done();
        return event.getResponse();
    }

    public void send(ILogEventGroup group, IProgressMonitor monitor) {
        SubMonitor sub = SubMonitor.convert(monitor, group.getEvents().size());
        for (ILogEvent event : group.getEvents()) {
            send(event, sub.newChild(1));
        }
        monitor.done();
    }

    public void schedule(ILogEvent event) {
        ILogEventGroup group = IIdeFactory.eINSTANCE.createLogEventGroup();
        group.setStatus(event.getStatus());
        group.getEvents().add(event);
        group.setStatus(event.getStatus());
        schedule(group);
    }

    public void schedule(ILogEventGroup group) {
        if (group == null) {
            Logs.warn("Got a NULL log event group. Please comment on bug 494165 if you know how to reproduce this error.");
            return;
        }
        uploadQueue.add(group);
        if (uploadJob != null && uploadJob.getState() == Job.RUNNING) {
            return;
        }
        uploadJob = new UploadJob(Messages.JOB_NAME_UPLOADING_REPORTS);
        uploadJob.schedule();
    }

    public void schedule(Iterable<ILogEventGroup> groups) {
        for (ILogEventGroup group : groups) {
            schedule(group);
        }
    }

    private final class UploadJob extends Job {
        private UploadJob(String name) {
            super(name);
        }

        @Override
        protected IStatus run(IProgressMonitor monitor) {
            while (!uploadQueue.isEmpty()) {
                ILogEventGroup group = uploadQueue.poll();
                send(group, monitor);
            }
            return Status.OK_STATUS;
        }

    }
}
