/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.logging.aeri.core.util;

import java.util.Set;

import org.eclipse.core.internal.runtime.Log;
import org.eclipse.core.internal.runtime.PlatformLogWriter;
import org.eclipse.core.internal.runtime.RuntimeLog;

import com.google.common.collect.ImmutableSet;

@SuppressWarnings("restriction")
public class NoStackTrace extends Throwable {

    private static final long serialVersionUID = 1L;

    public NoStackTrace() {
        super("This event was logged without a stack trace. A synthetic stack trace was hence inserted.");
    }

    @Override
    public synchronized Throwable fillInStackTrace() {
        super.fillInStackTrace();
        filterKnownLogClasses();
        return this;
    }

    private void filterKnownLogClasses() {
        StackTraceElement[] orgTrace = getStackTrace();
        StackTraceElement[] newTrace = Statuses.removeClassesFromTop(orgTrace, FILTERED_CLASSES);
        setStackTrace(newTrace);
    }

    /**
     * Example of a usual stack trace we like to clean up:
     *
     * <pre>
     *   at org.eclipse.epp.internal.logging.aeri.ui.LogListener.logging
     *   at org.eclipse.core.internal.runtime.RuntimeLog.logToListeners
     *   at org.eclipse.core.internal.runtime.PlatformLogWriter.logged
     *   at org.eclipse.osgi.internal.log.ExtendedLogReaderServiceFactory.safeLogged
     *   at org.eclipse.osgi.internal.log.ExtendedLogReaderServiceFactory.logPrivileged
     *   at org.eclipse.osgi.internal.log.ExtendedLogReaderServiceFactory.log
     *   at org.eclipse.osgi.internal.log.ExtendedLogServiceFactory.log
     *   at org.eclipse.osgi.internal.log.ExtendedLogServiceImpl.log
     *   at org.eclipse.osgi.internal.log.LoggerImpl.log
     *   at org.eclipse.core.internal.runtime.PlatformLogWriter.logging
     *   at org.eclipse.core.internal.runtime.RuntimeLog.log
     *   at org.eclipse.core.internal.jobs.JobManager.endJob
     *   at org.eclipse.core.internal.jobs.WorkerPool.endJob
     *   at org.eclipse.core.internal.jobs.Worker.run
     * </pre>
     */
    private static final Set<String> FILTERED_CLASSES = ImmutableSet.of(
            // org.eclipse.epp.logging.aeri:
            LogListener.class.getName(),
            // org.eclipse.core.runtime:
            Log.class.getName(), RuntimeLog.class.getName(), PlatformLogWriter.class.getName(),
            // org.eclipse.osgi:
            "org.eclipse.osgi.internal.log.ExtendedLogReaderServiceFactory$3",
            "org.eclipse.osgi.internal.log.ExtendedLogReaderServiceFactory", "org.eclipse.osgi.internal.log.ExtendedLogServiceFactory",
            "org.eclipse.osgi.internal.log.ExtendedLogServiceImpl", "org.eclipse.osgi.internal.log.LoggerImpl");
}
