/**
 * Copyright (c) 2010-2019, Geza Kulcsar, Abel Hegedus, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.testing.core;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.Enumerator;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.testing.core.api.JavaObjectAccess;
import org.eclipse.viatra.query.testing.snapshot.BooleanSubstitution;
import org.eclipse.viatra.query.testing.snapshot.CustomEMFSubstitution;
import org.eclipse.viatra.query.testing.snapshot.DateSubstitution;
import org.eclipse.viatra.query.testing.snapshot.DoubleSubstitution;
import org.eclipse.viatra.query.testing.snapshot.EMFSubstitution;
import org.eclipse.viatra.query.testing.snapshot.EnumSubstitution;
import org.eclipse.viatra.query.testing.snapshot.FloatSubstitution;
import org.eclipse.viatra.query.testing.snapshot.InputSpecification;
import org.eclipse.viatra.query.testing.snapshot.IntSubstitution;
import org.eclipse.viatra.query.testing.snapshot.LongSubstitution;
import org.eclipse.viatra.query.testing.snapshot.MatchRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSetRecord;
import org.eclipse.viatra.query.testing.snapshot.MatchSubstitutionRecord;
import org.eclipse.viatra.query.testing.snapshot.MiscellaneousSubstitution;
import org.eclipse.viatra.query.testing.snapshot.QuerySnapshot;
import org.eclipse.viatra.query.testing.snapshot.SerializedJavaObjectSubstitution;
import org.eclipse.viatra.query.testing.snapshot.SnapshotFactory;
import org.eclipse.viatra.query.testing.snapshot.StringSubstitution;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Helper methods for dealing with snapshots and match set records.
 */
@SuppressWarnings("all")
public class SnapshotHelper {
  private final Map<String, JavaObjectAccess> accessMap;
  
  private final Map<EClass, Function<EObject, String>> customEObjectSerializerMap;
  
  public SnapshotHelper() {
    this(Maps.<String, JavaObjectAccess>newHashMap());
  }
  
  /**
   * Initializes a {@link SnapshotHelper} with a map containing {@link JavaObjectAccess} objects for
   * serialization and deserialization of plain Java types.
   * 
   * @since 1.6
   * 
   * @deprecated
   * Use @link #SnapshotHelper(Map<String, JavaObjectAccess>,  Map<EClass, Function<EObject,String>>) instead
   */
  @Deprecated
  public SnapshotHelper(final Map<String, JavaObjectAccess> accessMap) {
    this.accessMap = accessMap;
    this.customEObjectSerializerMap = Maps.<EClass, Function<EObject, String>>newHashMap();
  }
  
  /**
   * @since 2.2
   */
  public SnapshotHelper(final Map<String, JavaObjectAccess> accessMap, final Map<EClass, Function<EObject, String>> customMap) {
    this.accessMap = accessMap;
    this.customEObjectSerializerMap = customMap;
  }
  
  /**
   * Returns the actual value of the substitution based on its type
   */
  public Object derivedValue(final MatchSubstitutionRecord substitution) {
    Object _switchResult = null;
    boolean _matched = false;
    if (substitution instanceof BooleanSubstitution) {
      _matched=true;
      _switchResult = Boolean.valueOf(((BooleanSubstitution)substitution).isValue());
    }
    if (!_matched) {
      if (substitution instanceof DateSubstitution) {
        _matched=true;
        _switchResult = ((DateSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof DoubleSubstitution) {
        _matched=true;
        _switchResult = Double.valueOf(((DoubleSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof EMFSubstitution) {
        _matched=true;
        _switchResult = ((EMFSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof EnumSubstitution) {
        _matched=true;
        _switchResult = ((EnumSubstitution)substitution).getValueLiteral();
      }
    }
    if (!_matched) {
      if (substitution instanceof FloatSubstitution) {
        _matched=true;
        _switchResult = Float.valueOf(((FloatSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof IntSubstitution) {
        _matched=true;
        _switchResult = Integer.valueOf(((IntSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof LongSubstitution) {
        _matched=true;
        _switchResult = Long.valueOf(((LongSubstitution)substitution).getValue());
      }
    }
    if (!_matched) {
      if (substitution instanceof MiscellaneousSubstitution) {
        _matched=true;
        _switchResult = ((MiscellaneousSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof StringSubstitution) {
        _matched=true;
        _switchResult = ((StringSubstitution)substitution).getValue();
      }
    }
    if (!_matched) {
      if (substitution instanceof SerializedJavaObjectSubstitution) {
        _matched=true;
        _switchResult = substitution;
      }
    }
    if (!_matched) {
      if (substitution instanceof CustomEMFSubstitution) {
        _matched=true;
        _switchResult = ((CustomEMFSubstitution)substitution).getValue();
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns the EMF root that was used by the matchers recorded into the given snapshot,
   *  based on the input specification and the model roots.
   */
  public Notifier getEMFRootForSnapshot(final QuerySnapshot snapshot) {
    Notifier _xifexpression = null;
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.EOBJECT);
    if (_equals) {
      EObject _xifexpression_1 = null;
      int _size = snapshot.getModelRoots().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _xifexpression_1 = snapshot.getModelRoots().get(0);
      }
      _xifexpression = _xifexpression_1;
    } else {
      Notifier _xifexpression_2 = null;
      InputSpecification _inputSpecification_1 = snapshot.getInputSpecification();
      boolean _equals_1 = Objects.equal(_inputSpecification_1, InputSpecification.RESOURCE);
      if (_equals_1) {
        Resource _xifexpression_3 = null;
        int _size_1 = snapshot.getModelRoots().size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          _xifexpression_3 = snapshot.getModelRoots().get(0).eResource();
        }
        _xifexpression_2 = _xifexpression_3;
      } else {
        ResourceSet _xifexpression_4 = null;
        InputSpecification _inputSpecification_2 = snapshot.getInputSpecification();
        boolean _equals_2 = Objects.equal(_inputSpecification_2, InputSpecification.RESOURCE_SET);
        if (_equals_2) {
          _xifexpression_4 = snapshot.eResource().getResourceSet();
        }
        _xifexpression_2 = _xifexpression_4;
      }
      _xifexpression = _xifexpression_2;
    }
    return _xifexpression;
  }
  
  /**
   * Returns the model roots that were used by the given ViatraQueryEngine.
   */
  public List<EObject> getModelRootsForEngine(final ViatraQueryEngine engine) {
    List<EObject> _switchResult = null;
    QueryScope _scope = engine.getScope();
    final QueryScope scope = _scope;
    boolean _matched = false;
    if (scope instanceof EMFScope) {
      _matched=true;
      final Function1<Notifier, List<EObject>> _function = (Notifier it) -> {
        List<EObject> _switchResult_1 = null;
        boolean _matched_1 = false;
        if (it instanceof EObject) {
          _matched_1=true;
          _switchResult_1 = Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList(((EObject)it)));
        }
        if (!_matched_1) {
          if (it instanceof Resource) {
            _matched_1=true;
            _switchResult_1 = ((Resource)it).getContents();
          }
        }
        if (!_matched_1) {
          if (it instanceof ResourceSet) {
            _matched_1=true;
            final Function1<Resource, EList<EObject>> _function_1 = (Resource it_1) -> {
              return it_1.getContents();
            };
            _switchResult_1 = IterableExtensions.<EObject>toList(Iterables.<EObject>concat(ListExtensions.<Resource, EList<EObject>>map(((ResourceSet)it).getResources(), _function_1)));
          }
        }
        return _switchResult_1;
      };
      _switchResult = IterableExtensions.<EObject>toList(Iterables.<EObject>concat(IterableExtensions.map(((EMFScope)scope).getScopeRoots(), _function)));
    }
    if (!_matched) {
      _switchResult = Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList());
    }
    return _switchResult;
  }
  
  /**
   * Returns the input specification for the given matcher.
   */
  public InputSpecification getInputSpecificationForMatcher(final ViatraQueryMatcher<?> matcher) {
    InputSpecification _switchResult = null;
    QueryScope _scope = matcher.getEngine().getScope();
    final QueryScope scope = _scope;
    boolean _matched = false;
    if (scope instanceof EMFScope) {
      _matched=true;
      InputSpecification _switchResult_1 = null;
      Notifier _head = IterableExtensions.head(((EMFScope)scope).getScopeRoots());
      boolean _matched_1 = false;
      if (_head instanceof EObject) {
        _matched_1=true;
        _switchResult_1 = InputSpecification.EOBJECT;
      }
      if (!_matched_1) {
        if (_head instanceof Resource) {
          _matched_1=true;
          _switchResult_1 = InputSpecification.RESOURCE;
        }
      }
      if (!_matched_1) {
        if (_head instanceof ResourceSet) {
          _matched_1=true;
          _switchResult_1 = InputSpecification.RESOURCE_SET;
        }
      }
      _switchResult = _switchResult_1;
    }
    if (!_matched) {
      _switchResult = InputSpecification.UNSET;
    }
    return _switchResult;
  }
  
  /**
   * Saves the matches of the given matcher (using the partial match) into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public <MATCH extends IPatternMatch> MatchSetRecord saveMatchesToSnapshot(final ViatraQueryMatcher<MATCH> matcher, final MATCH partialMatch, final QuerySnapshot snapshot) {
    final String patternFQN = matcher.getPatternName();
    final MatchSetRecord actualRecord = SnapshotFactory.eINSTANCE.createMatchSetRecord();
    actualRecord.setPatternQualifiedName(patternFQN);
    snapshot.getMatchSetRecords().add(actualRecord);
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.UNSET);
    if (_equals) {
      snapshot.getModelRoots().addAll(this.getModelRootsForEngine(matcher.getEngine()));
      snapshot.getModelRoots().remove(snapshot);
      snapshot.setInputSpecification(this.getInputSpecificationForMatcher(matcher));
    }
    actualRecord.setFilter(this.createMatchRecordForMatch(partialMatch));
    final Consumer<MATCH> _function = (MATCH match) -> {
      actualRecord.getMatches().add(this.createMatchRecordForMatch(match));
    };
    matcher.forEachMatch(partialMatch, _function);
    return actualRecord;
  }
  
  /**
   * Creates a match record that corresponds to the given match.
   *  Each parameter with a value is saved as a substitution.
   */
  public MatchRecord createMatchRecordForMatch(final IPatternMatch match) {
    final MatchRecord matchRecord = SnapshotFactory.eINSTANCE.createMatchRecord();
    final Consumer<String> _function = (String param) -> {
      Object _get = match.get(param);
      boolean _tripleNotEquals = (_get != null);
      if (_tripleNotEquals) {
        matchRecord.getSubstitutions().add(this.createSubstitution(param, match.get(param)));
      }
    };
    match.parameterNames().forEach(_function);
    return matchRecord;
  }
  
  /**
   * Creates a match set record which holds the snapshot of a single matcher instance. It is also possible to enter
   * a filter for the matcher.
   */
  public <Match extends IPatternMatch> MatchSetRecord createMatchSetRecordForMatcher(final ViatraQueryMatcher<Match> matcher, final Match filter) {
    final MatchSetRecord matchSetRecord = SnapshotFactory.eINSTANCE.createMatchSetRecord();
    final Consumer<Match> _function = (Match match) -> {
      matchSetRecord.getMatches().add(this.createMatchRecordForMatch(match));
    };
    matcher.forEachMatch(filter, _function);
    return matchSetRecord;
  }
  
  /**
   * Creates a partial match that corresponds to the given match record.
   *  Each substitution is used as a value for the parameter with the same name.
   */
  public <MATCH extends IPatternMatch> MATCH createMatchForMatchRecord(final IQuerySpecification<? extends ViatraQueryMatcher<MATCH>> querySpecification, final MatchRecord matchRecord) {
    IPatternMatch _newEmptyMatch = querySpecification.newEmptyMatch();
    final MATCH match = ((MATCH) _newEmptyMatch);
    final Consumer<MatchSubstitutionRecord> _function = (MatchSubstitutionRecord it) -> {
      Object target = this.derivedValue(it);
      match.set(it.getParameterName(), target);
    };
    matchRecord.getSubstitutions().forEach(_function);
    return match;
  }
  
  /**
   * Saves all matches of the given matcher into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public <MATCH extends IPatternMatch> MatchSetRecord saveMatchesToSnapshot(final ViatraQueryMatcher<MATCH> matcher, final QuerySnapshot snapshot) {
    return this.<MATCH>saveMatchesToSnapshot(matcher, matcher.newEmptyMatch(), snapshot);
  }
  
  /**
   * Returns the match set record for the given pattern FQN from the snapshot,
   *  if there is only one such record.
   */
  public MatchSetRecord getMatchSetRecordForPattern(final QuerySnapshot snapshot, final String patternFQN) {
    final Function1<MatchSetRecord, Boolean> _function = (MatchSetRecord it) -> {
      return Boolean.valueOf(it.getPatternQualifiedName().equals(patternFQN));
    };
    final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(snapshot.getMatchSetRecords(), _function);
    int _size = IterableExtensions.size(matchsetrecord);
    boolean _equals = (_size == 1);
    if (_equals) {
      return matchsetrecord.iterator().next();
    }
    return null;
  }
  
  /**
   * Returns the match set records for the given pattern FQN from the snapshot.
   */
  public ArrayList<MatchSetRecord> getMatchSetRecordsForPattern(final QuerySnapshot snapshot, final String patternFQN) {
    final ArrayList<MatchSetRecord> matchSetRecords = new ArrayList<MatchSetRecord>();
    final Function1<MatchSetRecord, Boolean> _function = (MatchSetRecord it) -> {
      return Boolean.valueOf(it.getPatternQualifiedName().equals(patternFQN));
    };
    Iterables.<MatchSetRecord>addAll(matchSetRecords, IterableExtensions.<MatchSetRecord>filter(snapshot.getMatchSetRecords(), _function));
    return matchSetRecords;
  }
  
  /**
   * Creates a substitution for the given parameter name using the given value.
   *  The type of the substitution is decided based on the type of the value.
   */
  public MatchSubstitutionRecord createSubstitution(final String parameterName, final Object value) {
    MatchSubstitutionRecord _switchResult = null;
    boolean _matched = false;
    if (value instanceof Enumerator) {
      _matched=true;
      EnumSubstitution _xblockexpression = null;
      {
        final EnumSubstitution sub = SnapshotFactory.eINSTANCE.createEnumSubstitution();
        sub.setValueLiteral(((Enumerator)value).getLiteral());
        sub.setParameterName(parameterName);
        _xblockexpression = sub;
      }
      _switchResult = _xblockexpression;
    }
    if (!_matched) {
      if (value instanceof EObject) {
        _matched=true;
        MatchSubstitutionRecord _xblockexpression = null;
        {
          Function<EObject, String> obj = this.getMostSpecificSerializationRule(((EObject)value).eClass());
          MatchSubstitutionRecord _xifexpression = null;
          if ((obj != null)) {
            CustomEMFSubstitution _xblockexpression_1 = null;
            {
              final CustomEMFSubstitution sub = SnapshotFactory.eINSTANCE.createCustomEMFSubstitution();
              sub.setValue(obj.apply(((EObject)value)));
              sub.setType(((EObject)value).eClass());
              sub.setParameterName(parameterName);
              _xblockexpression_1 = sub;
            }
            _xifexpression = _xblockexpression_1;
          } else {
            EMFSubstitution _xblockexpression_2 = null;
            {
              final EMFSubstitution sub = SnapshotFactory.eINSTANCE.createEMFSubstitution();
              sub.setValue(((EObject)value));
              sub.setParameterName(parameterName);
              _xblockexpression_2 = sub;
            }
            _xifexpression = _xblockexpression_2;
          }
          _xblockexpression = _xifexpression;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Integer) {
        _matched=true;
        IntSubstitution _xblockexpression = null;
        {
          final IntSubstitution sub = SnapshotFactory.eINSTANCE.createIntSubstitution();
          sub.setValue(((Integer) value).intValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Long) {
        _matched=true;
        LongSubstitution _xblockexpression = null;
        {
          final LongSubstitution sub = SnapshotFactory.eINSTANCE.createLongSubstitution();
          sub.setValue(((Long) value).longValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Double) {
        _matched=true;
        DoubleSubstitution _xblockexpression = null;
        {
          final DoubleSubstitution sub = SnapshotFactory.eINSTANCE.createDoubleSubstitution();
          sub.setValue(((Double) value).doubleValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Float) {
        _matched=true;
        FloatSubstitution _xblockexpression = null;
        {
          final FloatSubstitution sub = SnapshotFactory.eINSTANCE.createFloatSubstitution();
          sub.setValue(((Float) value).floatValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Boolean) {
        _matched=true;
        BooleanSubstitution _xblockexpression = null;
        {
          final BooleanSubstitution sub = SnapshotFactory.eINSTANCE.createBooleanSubstitution();
          sub.setValue(((Boolean) value).booleanValue());
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof String) {
        _matched=true;
        StringSubstitution _xblockexpression = null;
        {
          final StringSubstitution sub = SnapshotFactory.eINSTANCE.createStringSubstitution();
          sub.setValue(((String)value));
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      if (value instanceof Date) {
        _matched=true;
        DateSubstitution _xblockexpression = null;
        {
          final DateSubstitution sub = SnapshotFactory.eINSTANCE.createDateSubstitution();
          sub.setValue(((Date)value));
          sub.setParameterName(parameterName);
          _xblockexpression = sub;
        }
        _switchResult = _xblockexpression;
      }
    }
    if (!_matched) {
      MatchSubstitutionRecord _xblockexpression = null;
      {
        final JavaObjectAccess obj = this.accessMap.get(value.getClass().getName());
        MatchSubstitutionRecord _xifexpression = null;
        if ((obj != null)) {
          SerializedJavaObjectSubstitution _xblockexpression_1 = null;
          {
            final SerializedJavaObjectSubstitution sub = obj.toSubstitution(value);
            sub.setParameterName(parameterName);
            _xblockexpression_1 = sub;
          }
          _xifexpression = _xblockexpression_1;
        } else {
          MiscellaneousSubstitution _xblockexpression_2 = null;
          {
            final MiscellaneousSubstitution sub = SnapshotFactory.eINSTANCE.createMiscellaneousSubstitution();
            sub.setValue(value);
            sub.setParameterName(parameterName);
            _xblockexpression_2 = sub;
          }
          _xifexpression = _xblockexpression_2;
        }
        _xblockexpression = _xifexpression;
      }
      _switchResult = _xblockexpression;
    }
    return _switchResult;
  }
  
  private Function<EObject, String> getMostSpecificSerializationRule(final EClass cls) {
    Function<EObject, String> _xblockexpression = null;
    {
      Function<EObject, String> obj = this.customEObjectSerializerMap.get(cls);
      Function<EObject, String> _xifexpression = null;
      if ((obj != null)) {
        return obj;
      } else {
        EList<EClass> _eSuperTypes = cls.getESuperTypes();
        for (final EClass type : _eSuperTypes) {
          Function<EObject, String> _get = this.customEObjectSerializerMap.get(type);
          boolean _tripleNotEquals = (_get != null);
          if (_tripleNotEquals) {
            return this.customEObjectSerializerMap.get(type);
          }
        }
        EList<EClass> _eSuperTypes_1 = cls.getESuperTypes();
        for (final EClass type_1 : _eSuperTypes_1) {
          return this.getMostSpecificSerializationRule(type_1);
        }
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Retrieve a human-readable string denoting the given record
   */
  protected String _prettyPrint(final MatchRecord record) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<MatchSubstitutionRecord> _substitutions = record.getSubstitutions();
      boolean _hasElements = false;
      for(final MatchSubstitutionRecord substitution : _substitutions) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _parameterName = substitution.getParameterName();
        _builder.append(_parameterName);
        _builder.append(" = ");
        Object _derivedValue = this.derivedValue(substitution);
        String _prettyPrint = null;
        if (_derivedValue!=null) {
          _prettyPrint=this.prettyPrint(_derivedValue);
        }
        _builder.append(_prettyPrint);
      }
    }
    return _builder.toString();
  }
  
  protected String _prettyPrint(final EObject obj) {
    String _xblockexpression = null;
    {
      final EClass eClass = obj.eClass();
      StringConcatenation _builder = new StringConcatenation();
      String _name = eClass.getName();
      _builder.append(_name);
      _builder.append(" (");
      {
        EList<EAttribute> _eAllAttributes = eClass.getEAllAttributes();
        for(final EAttribute attr : _eAllAttributes) {
          _builder.append(" ");
          String _name_1 = attr.getName();
          _builder.append(_name_1);
          _builder.append(" = ");
          Object _eGet = obj.eGet(attr);
          String _prettyPrint = null;
          if (_eGet!=null) {
            _prettyPrint=this.prettyPrint(_eGet);
          }
          _builder.append(_prettyPrint);
          _builder.append(" ");
        }
      }
      _builder.append(")");
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  protected String _prettyPrint(final Object obj) {
    return obj.toString();
  }
  
  public String prettyPrint(final Object record) {
    if (record instanceof MatchRecord) {
      return _prettyPrint((MatchRecord)record);
    } else if (record instanceof EObject) {
      return _prettyPrint((EObject)record);
    } else if (record != null) {
      return _prettyPrint(record);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(record).toString());
    }
  }
}
