/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalUniqueConstraint;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form of an entity table.
 *
 * @see RelationshipMapping
 *
 * @version 2.5
 * @author Les Davis
 * @author Pascal Filion
 */
@SuppressWarnings("nls")
abstract class Table extends AbstractExternalForm implements ExternalEntityTable {

	/**
	 * The attribute name used to store and retrieve the catalog property.
	 */
	static final String CATALOG = "catalog";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the schema property.
	 */
	static final String SCHEMA = "schema";

	/**
	 * Creates a new <code>Table</code>.
	 *
	 * @param parent The parent of this external form
	 */
	Table(AbstractExternalForm parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void addUniqueConstraint(int index, ListIterator<String> columnNames) {
		UniqueConstraint uniqueConstraint = buildUniqueConstraint(index);
		uniqueConstraint.addSelf();
		uniqueConstraint.setColumnNames(columnNames);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(CATALOG);
		names.add(SCHEMA);
		return names;
	}

	private UniqueConstraint buildUniqueConstraint(int index) {
		return new UniqueConstraint(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCatalogName() {
		return getAttribute(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getSchemaName() {
		return getAttribute(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalUniqueConstraint getUniqueConstraint(int index) {

		Element element = getChild(UniqueConstraint.UNIQUE_CONSTRAINT, index);

		if (element == null) {
			return null;
		}

		return buildUniqueConstraint(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeAllUniqueConstraints()
	{
		removeChildren(UniqueConstraint.UNIQUE_CONSTRAINT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeUniqueConstraint(int index) {
		UniqueConstraint uniqueConstraint = buildUniqueConstraint(index);
		uniqueConstraint.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCatalogName(String catalog) {
		setAttribute(CATALOG, catalog);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setSchemaName(String name) {
		setAttribute(SCHEMA, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ListIterable<ExternalUniqueConstraint> uniqueConstraints() {

		int count = uniqueConstraintsSize();
		List<ExternalUniqueConstraint> uniqueConstraints = new ArrayList<ExternalUniqueConstraint>(count);

		for (int index = count; --index >= 0;) {
			ExternalUniqueConstraint uniqueConstraint = buildUniqueConstraint(index);
			uniqueConstraints.add(0, uniqueConstraint);
		}

		return new ListListIterable<ExternalUniqueConstraint>(uniqueConstraints);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int uniqueConstraintsSize() {
		return getChildrenSize(UniqueConstraint.UNIQUE_CONSTRAINT);
	}
}