/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.util.Arrays;
import org.eclipse.persistence.tools.utility.model.Model;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;
import org.eclipse.persistence.tools.utility.model.listener.ListChangeListener;

/**
 * Extend {@link ValueAspectAdapter} to listen to one or more list
 * aspects of the value in the wrapped value model.
 */
@SuppressWarnings("nls")
public class ValueListAdapter<V extends Model>
	extends ValueAspectAdapter<V>
{
	/** The names of the value's lists that we listen to. */
	protected final String[] listNames;

	/** Listener that listens to the value. */
	protected final ListChangeListener valueListListener;


	// ********** constructors **********

	/**
	 * Construct an adapter for the specified value lists.
	 */
	public ValueListAdapter(ModifiablePropertyValueModel<V> valueHolder, String... listNames) {
		super(valueHolder);
		if (listNames == null) {
			throw new NullPointerException();
		}
		this.listNames = listNames;
		this.valueListListener = this.buildValueListListener();
	}


	// ********** initialization **********

	protected ListChangeListener buildValueListListener() {
		return new ListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent event) {
				ValueListAdapter.this.itemsAdded(event);
			}
			@Override
			public void itemsRemoved(ListRemoveEvent event) {
				ValueListAdapter.this.itemsRemoved(event);
			}
			@Override
			public void itemsReplaced(ListReplaceEvent event) {
				ValueListAdapter.this.itemsReplaced(event);
			}
			@Override
			public void itemsMoved(ListMoveEvent event) {
				ValueListAdapter.this.itemsMoved(event);
			}
			@Override
			public void listCleared(ListClearEvent event) {
				ValueListAdapter.this.listCleared(event);
			}
			@Override
			public void listChanged(ListChangeEvent event) {
				ValueListAdapter.this.listChanged(event);
			}
			@Override
			public String toString() {
				return "value list listener: " + Arrays.asList(ValueListAdapter.this.listNames);
			}
		};
	}


	// ********** ValueAspectAdapter implementation **********

	@Override
	protected void engageValue_() {
		for (String listName : this.listNames) {
			this.value.addListChangeListener(listName, this.valueListListener);
		}
	}

	@Override
	protected void disengageValue_() {
		for (String listName : this.listNames) {
			this.value.removeListChangeListener(listName, this.valueListListener);
		}
	}


	// ********** change events **********

	protected void itemsAdded(@SuppressWarnings("unused") ListAddEvent event) {
		this.valueAspectChanged();
	}

	protected void itemsRemoved(@SuppressWarnings("unused") ListRemoveEvent event) {
		this.valueAspectChanged();
	}

	protected void itemsReplaced(@SuppressWarnings("unused") ListReplaceEvent event) {
		this.valueAspectChanged();
	}

	protected void itemsMoved(@SuppressWarnings("unused") ListMoveEvent event) {
		this.valueAspectChanged();
	}

	protected void listCleared(@SuppressWarnings("unused") ListClearEvent event) {
		this.valueAspectChanged();
	}

	protected void listChanged(@SuppressWarnings("unused") ListChangeEvent event) {
		this.valueAspectChanged();
	}
}