/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.w3c.dom.Element;

/**
 * The external form of a type converter, which is a child of an entity.
 *
 * @see ConvertibleMapping
 * @see EmbeddableEntity
 * @see ORMConfiguration
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class TypeConverter extends AbstractExternalForm
                          implements ExternalTypeConverter {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * The attribute name used to store and retrieve the data-type property.
	 */
	static final String DATA_TYPE = "data-type";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the object-type property.
	 */
	static final String OBJECT_TYPE = "object-type";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String TYPE_CONVERTER = "type-converter";

	/**
	 * Creates a new <code>TypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	TypeConverter(ConvertibleMapping parent) {
		super(parent);
		this.index = -1;
	}

	/**
	 * Creates a new <code>TypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	TypeConverter(EmbeddableEntity parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * Creates a new <code>TypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	TypeConverter(ORMConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {

		if (index == -1) {
			return super.addSelf(elementName, elementNamesOrder);
		}

		return addChild(getParent(), elementName, index, elementNamesOrder);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(DATA_TYPE);
		names.add(OBJECT_TYPE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatatype() {
		return getAttribute(DATA_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {

		if (index == -1) {
			return super.getElement();
		}

		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return TYPE_CONVERTER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjecttype() {
		return getAttribute(OBJECT_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		if (index == -1) {
			super.removeSelf();
		}
		else {
			removeChild(getParent(), getElementName(), index);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDatatype(String datatypeName) {
		setAttribute(DATA_TYPE, datatypeName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setObjecttype(String objecttypeName) {
		setAttribute(OBJECT_TYPE, objecttypeName);
	}
}