/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.tests.model.value;

import java.util.Collection;
import java.util.List;
import javax.swing.JList;
import junit.framework.TestCase;
import org.eclipse.persistence.tools.utility.collection.Bag;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.collection.HashBag;
import org.eclipse.persistence.tools.utility.model.AbstractModel;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;
import org.eclipse.persistence.tools.utility.model.listener.ChangeAdapter;
import org.eclipse.persistence.tools.utility.model.listener.ChangeListener;
import org.eclipse.persistence.tools.utility.model.listener.ListChangeListener;
import org.eclipse.persistence.tools.utility.model.value.CollectionListValueModelAdapter;
import org.eclipse.persistence.tools.utility.model.value.ListValueModel;
import org.eclipse.persistence.tools.utility.model.value.SimpleCollectionValueModel;
import org.eclipse.persistence.tools.utility.model.value.swing.ListModelAdapter;
import org.eclipse.persistence.tools.utility.tests.TestTools;

@SuppressWarnings("nls")
public class CollectionListValueModelAdapterTests extends TestCase {
	private ListValueModel<String> adapter;
	private SimpleCollectionValueModel<String> wrappedCollectionHolder;
	private Collection<String> wrappedCollection;

	public CollectionListValueModelAdapterTests(String name) {
		super(name);
	}

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		this.wrappedCollection = new HashBag<String>();
		this.wrappedCollectionHolder = new SimpleCollectionValueModel<String>(this.wrappedCollection);
		this.adapter = new CollectionListValueModelAdapter<String>(this.wrappedCollectionHolder);
	}

	@Override
	protected void tearDown() throws Exception {
		TestTools.clear(this);
		super.tearDown();
	}

	public void testIterator() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		Collection<String> adapterCollection = this.adapterCollection();
		assertEquals(3, adapterCollection.size());
		assertEquals(this.wrappedCollection, adapterCollection);
	}

	private Collection<String> adapterCollection() {
		return CollectionTools.collection(this.adapter.iterator());
	}

	public void testStaleValue() {
		ListChangeListener listener = new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
		};
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, listener);
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		Collection<String> adapterCollection = this.adapterCollection();
		assertEquals(3, adapterCollection.size());
		assertEquals(this.wrappedCollection, adapterCollection);

		this.adapter.removeListChangeListener(ListValueModel.LIST_VALUES, listener);
		adapterCollection = this.adapterCollection();
		assertEquals(0, adapterCollection.size());
		assertEquals(new HashBag<String>(), adapterCollection);

		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, listener);
		adapterCollection = this.adapterCollection();
		assertEquals(3, adapterCollection.size());
		assertEquals(this.wrappedCollection, adapterCollection);
	}

	public void testAdd() {
		List<String> synchList = new CoordinatedList<String>(this.adapter);
		Bag<String> synchCollection = new CoordinatedBag<String>(this.wrappedCollectionHolder);
		this.wrappedCollectionHolder.add("foo");
		assertTrue(this.wrappedCollection.contains("foo"));
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		assertEquals(6, this.wrappedCollection.size());

		Collection<String> adapterCollection = this.adapterCollection();
		assertEquals(this.wrappedCollection, adapterCollection);
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);
	}

	public void testRemove() {
		List<String> synchList = new CoordinatedList<String>(this.adapter);
		Bag<String> synchCollection = new CoordinatedBag<String>(this.wrappedCollectionHolder);
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		assertTrue(this.wrappedCollection.contains("jaz"));
		this.wrappedCollectionHolder.remove("jaz");
		assertFalse(this.wrappedCollection.contains("jaz"));
		this.wrappedCollectionHolder.remove("foo");
		assertFalse(this.wrappedCollection.contains("foo"));
		assertEquals(4, this.wrappedCollection.size());

		Collection<String> adapterCollection = this.adapterCollection();
		assertEquals(this.wrappedCollection, adapterCollection);
		assertEquals(this.wrappedCollection, CollectionTools.collection(synchList.iterator()));
		assertEquals(this.wrappedCollection, synchCollection);
	}

	public void testListSynch() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsRemoved(ListRemoveEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		this.wrappedCollectionHolder.add("joo");
		this.wrappedCollectionHolder.add("jar");
		this.wrappedCollectionHolder.add("jaz");
		this.wrappedCollectionHolder.remove("jaz");
		assertFalse(this.wrappedCollection.contains("jaz"));
		this.wrappedCollectionHolder.remove("foo");
		assertFalse(this.wrappedCollection.contains("foo"));
		assertEquals(4, this.wrappedCollection.size());

		Collection<String> adapterCollection = this.adapterCollection();
		assertEquals(this.wrappedCollection, adapterCollection);
	}

	public void testHasListeners() {
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		CoordinatedList<String> synchList = new CoordinatedList<String>(this.adapter);
		assertTrue(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		this.adapter.removeListChangeListener(ListValueModel.LIST_VALUES, synchList);
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));

		ChangeListener cl = new ChangeAdapter();
		this.adapter.addChangeListener(cl);
		assertTrue(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
		this.adapter.removeChangeListener(cl);
		assertFalse(((AbstractModel) this.adapter).hasAnyListChangeListeners(ListValueModel.LIST_VALUES));
	}

	public void testCollectionChangedToEmpty() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void listCleared(ListClearEvent e) {/* OK */}
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
		});
		this.wrappedCollectionHolder.add("foo");
		this.wrappedCollectionHolder.add("bar");
		this.wrappedCollectionHolder.add("baz");
		JList jList = new JList(new ListModelAdapter(this.adapter));
		this.wrappedCollectionHolder.setValues(new HashBag<String>());
		assertEquals(0, jList.getModel().getSize());
	}

	public void testCollectionChangedFromEmpty() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsRemoved(ListRemoveEvent e) {/* OK */}
		});
		JList jList = new JList(new ListModelAdapter(this.adapter));

		HashBag<String> bag = new HashBag<String>();
		bag.add("foo");
		bag.add("bar");
		this.wrappedCollectionHolder.setValues(bag);
		assertEquals(2, jList.getModel().getSize());
	}

	public void testCollectionChangedFromEmptyToEmpty() {
		this.adapter.addListChangeListener(ListValueModel.LIST_VALUES, new TestListChangeListener() {
			@Override
			public void itemsAdded(ListAddEvent e) {/* OK */}
			@Override
			public void itemsRemoved(ListRemoveEvent e) {/* OK */}
		});
		JList jList = new JList(new ListModelAdapter(this.adapter));

		HashBag<String> bag = new HashBag<String>();
		this.wrappedCollectionHolder.setValues(bag);
		assertEquals(0, jList.getModel().getSize());
	}


	class TestListChangeListener implements ListChangeListener {
		@Override
		public void itemsAdded(ListAddEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsRemoved(ListRemoveEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsReplaced(ListReplaceEvent e) {
			fail("unexpected event");
		}
		@Override
		public void itemsMoved(ListMoveEvent e) {
			fail("unexpected event");
		}
		@Override
		public void listCleared(ListClearEvent e) {
			fail("unexpected event");
		}
		@Override
		public void listChanged(ListChangeEvent e) {
			fail("unexpected event");
		}
	}
}