/*******************************************************************************
 * Copyright (c) 2012, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.db;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import org.eclipse.persistence.tools.db.driver.WorkbenchDriverAdapter;
import org.eclipse.persistence.tools.db.driver.WorkbenchDriverAdapterManager;
import org.eclipse.persistence.tools.db.model.ELDatabase;
import org.eclipse.persistence.tools.db.model.ELLoginSpec;
import org.eclipse.persistence.tools.db.model.platformsmodel.DatabasePlatformRepository;
import org.eclipse.persistence.tools.gen.db.Catalog;
import org.eclipse.persistence.tools.gen.db.ConnectionProfile;
import org.eclipse.persistence.tools.gen.db.Database;
import org.eclipse.persistence.tools.gen.db.Schema;
import org.eclipse.persistence.tools.gen.db.Table;
import org.eclipse.persistence.tools.utility.collection.CollectionTools;
import org.eclipse.persistence.tools.utility.iterable.ArrayIterable;
import org.eclipse.persistence.tools.utility.iterable.TransformationIterable;

/**
 * The concrete implementation of {@link Database}.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 * @author John Bracken
 */
@SuppressWarnings("nls")
public class EclipseLinkDatabase implements Database {

	private ELDatabase mwDatabase;

	/** database-specific behavior */
	private WorkbenchDriverAdapter workbenchDriverAdapter;

	public EclipseLinkDatabase(String dbPlatformName, String driverClassName, String connectionURL, String userName, String password) {
		super();
		initialize(dbPlatformName, driverClassName, connectionURL, userName, password);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String  convertNameToIdentifier(String name) {
		return this.workbenchDriverAdapter.convertNameToIdentifier(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Catalog getCatalogForIdentifier(String identifier) {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Catalog getCatalogNamed(String name) {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Catalog> getCatalogs() {
		return new ArrayList<Catalog>();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getCatalogsSize() {
		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionProfile getConnectionProfile() {
		return new EclipseLinkConnectionProfile(this, getMWDatabase());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Database getDatabase() {
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Catalog getDefaultCatalog() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultCatalogIdentifier() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Schema getDefaultSchema() {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultSchemaIdentifier() {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier() {
		return convertNameToIdentifier(getName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getIdentifier(String defaultName) {
		return null;
	}

	/**
	 * Examples:<ul>
	 * <li>Oracle etc.<ul><code>
	 *     <li>Table(FOO) vs. "Foo" => null
	 *     <li>Table(BAR) vs. "Foo" => "BAR"
	 *     <li>Table(Foo) vs. "Foo" => "\"Foo\""
	 *     <li>Table(Bar) vs. "Foo" => "\"Bar\""
	 * </code></ul>
	 * <li>PostgreSQL etc.<ul><code>
	 *     <li>Table(foo) vs. "Foo" => null
	 *     <li>Table(bar) vs. "Foo" => "bar"
	 *     <li>Table(Foo) vs. "Foo" => "\"Foo\""
	 *     <li>Table(Bar) vs. "Foo" => "\"Bar\""
	 * </code></ul>
	 * <li>SQL Server etc.<ul><code>
	 *     <li>Table(Foo) vs. "Foo" => null
	 *     <li>Table(foo) vs. "Foo" => "foo"
	 *     <li>Table(bar) vs. "Foo" => "bar"
	 *     <li>Table(Bar) vs. "Foo" => "Bar"
	 * </code></ul>
	 * </ul>
	 */
	public String getIdentifier(String name, String defaultName) {
		return this.workbenchDriverAdapter.convertNameToIdentifier(name, defaultName);
	}

	private ELDatabase getMWDatabase() {
		return this.mwDatabase;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Schema getSchemaForIdentifier(String identifier) {
		return getSchemaNamed(identifier);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Schema getSchemaNamed(String name) {
		return new EclipseLinkSchema(EclipseLinkDatabase.this, getMWDatabase(), name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<Schema> getSchemata() {
		return new TransformationIterable<String, Schema>(new Iterable<String>() {
			@Override
			public Iterator<String> iterator() {
				return getMWDatabase().schemaNames();
			}
		}) {
			@Override
			protected Schema transform(String schemaName) {
				return new EclipseLinkSchema(EclipseLinkDatabase.this, getMWDatabase(), schemaName);
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSchemataSize() {
		return this.mwDatabase.getExternalDatabase().getSchemaNames().length;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedCatalogIdentifiers() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedCatalogNames() {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedSchemaIdentifiers() {
		return getSortedSchemaNames();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Iterable<String> getSortedSchemaNames() {
		return new ArrayIterable<String>(this.mwDatabase.getExternalDatabase().getSchemaNames());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getVendorName() {
		return this.getMWDatabase().getDatabasePlatform().getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getVersion() {
		return this.getMWDatabase().getDatabasePlatform().getName();
	}

	private void initialize(String dbPlatformName, String driverClassName, String connectionURL, String userName, String password) {
		this.mwDatabase = new ELDatabase(DatabasePlatformRepository.getDefault().platformNamed(dbPlatformName));
		ELLoginSpec loginSpec = getMWDatabase().addLoginSpec("dbProject_spec");
		loginSpec.setURL(connectionURL);
		loginSpec.setUserName(userName);
		loginSpec.setPassword(password);
		loginSpec.setDriverClassName(driverClassName);
		try {
			this.mwDatabase.login();
		} catch (SQLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		this.mwDatabase.importQualifiedTablesFor(CollectionTools.collection(this.mwDatabase.externalTableDescriptions()));
		this.workbenchDriverAdapter = WorkbenchDriverAdapterManager.instance().buildAdapter(this.getVendorName(), this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Table selectTableForIdentifier(Iterable<Table> tables, String identifier) {
		throw new UnsupportedOperationException("Not supported!");
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean supportsCatalogs() {
		return false;
	}
}