/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import java.util.List;
import org.eclipse.persistence.annotations.ChangeTrackingType;
import org.eclipse.persistence.tools.mapping.ExternalPropertyHolder;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * This interface describes the configurable properties for an embeddable entity. One or more
 * embeddable entities can be declared as part of an ORM configuration.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.Embeddable
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalEmbeddable extends ExternalAccessType,
                                            ExternalPropertyHolder,
                                            ExternalConverterProvider {

	/**
	 * The element name used to store and retrieve the change-tracking child node.
	 */
	String CHANGE_TRACKING = "change-tracking";

	/**
	 * The attribute name used to store and retrieve the class property or the element name used to
	 * store and retrieve the class child text node of the customizer child node.
	 */
	String CLASS = "class";

	/**
	 * The element name used to store and retrieve the customizer child node.
	 */
	String CUSTOMIZER = "customizer";

	/**
	 * The element name used to store and retrieve the description child text node.
	 */
	String DESCRIPTION = "description";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String EMBEDDABLE = "embeddable";

	/**
	 * The attribute name used to store and retrieve the exclude-default-mappings property.
	 */
	String EXCLUDE_DEFAULT_MAPPINGS = "exclude-default-mappings";

	/**
	 * The attribute name used to store and retrieve the metadata-complete property.
	 */
	String METADATA_COMPLETE = "metadata-complete";

	/**
	 * The attribute name used to store and retrieve the parent class name.
	 */
	String PARENT_CLASS = "parent-class";

	/**
	 * The attribute name used to store and retrieve the type child text node of the change-tracking child node.
	 */
	String TYPE = "type";

	/**
	 * Adds the get and Sets method names for this mapping.
	 */
	ExternalAccessMethods addAccessMethods(String getMethodName, String setMethodName);

	/**
	 * Adds a override with the given name.
	 */
	ExternalAssociationOverride addAssociationOverride(String name);

	/**
	 * Adds a override with the given name.
	 */
	ExternalAttributeOverride addAttributeOverride(String name);

	/**
	 * Adds a basic collection mapping with the given name.
	 */
	ExternalBasicCollectionMapping addBasicCollectionMapping(String name);

	/**
	 * Adds a basic map mapping with the given name.
	 */
	ExternalBasicMapMapping addBasicMapMapping(String name);

	/**
	 * Adds a basic mapping with the given name.
	 */
	ExternalBasicMapping addBasicMapping(String name);

	/**
	 * Adds clone copy policy to this entity
	 */
	ExternalCloneCopyPolicy addCloneCopyPolicy();

	/**
	 * Adds copy policy to this entity
	 */
	ExternalCopyPolicy addCopyPolicy();

	/**
	 * Adds a element collection mapping with the given name.
	 */
	ExternalElementCollectionMapping addElementCollectionMapping(String name);

	/**
	 * Adds an embeddable mapping with the given name.
	 */
	ExternalEmbeddedMapping addEmbeddedMapping(String name);

	/**
	 * Adds an id mapping with the given name.
	 */
	ExternalIDMapping addIdMapping(String name);

	/**
	 * Adds instantiation copy policy to this entity
	 */
	ExternalInstantiationCopyPolicy addInstantiationCopyPolicy();

	/**
	 * Adds a many-to-many mapping with the given name.
	 */
	ExternalManyToManyMapping addManyToManyMapping(String name);

	/**
	 * Adds a many-to-one mapping with the given name.
	 */
	ExternalManyToOneMapping addManyToOneMapping(String name);

	/**
	 * Adds the NoSQL annotation.
	 */
	ExternalNoSql addNoSql();

	/**
	 * Adds a one-to-many mapping with the given name.
	 */
	ExternalOneToManyMapping addOneToManyMapping(String name);

	/**
	 * Adds a one-to-one mapping with the given name.
	 */
	ExternalOneToOneMapping addOneToOneMapping(String name);

	/**
	 * Adds a transient mapping with the given name.
	 */
	ExternalTransientMapping addTransientMapping(String name);

	/**
	 * Returns a list of association overrides defined on this entity.
	 */
	List<ExternalAssociationOverride> associationOverrides();

	/**
	 * Returns the count of association overrides defined on this entity.
	 */
	int associationOverridesSize();

	/**
	 * Returns a list of attribute overrides defined for this entity.
	 */
	List<ExternalAttributeOverride> attributeOverrides();

	/**
	 * Returns the count attribute overrides defined for this entity.
	 */
	int attributeOverridesSize();

	/**
	 * Returns the access methods for this mapping.
	 */
	ExternalAccessMethods getAccessMethods();

	/**
	 * Returns the association override at the specified index.
	 */
	ExternalAssociationOverride getAssociationOverride(int index);

	/**
	 * Returns the association override at the specified index.
	 */
	ExternalAttributeOverride getAttributeOverride(int index);

	/**
	 * Returns the change tracking type for this entity
	 */
	ChangeTrackingType getChangeTrackingType();

	/**
	 * Returns the {@link TextRange} for the change tracking type for this entity
	 */
	TextRange getChangeTrackingTypeTextRange();

	/**
	 * Returns the class name of the entity.
	 */
	String getClassName();

	/**
	 * Returns the {@link TextRange} of the class name attribute's value.
	 */
	TextRange getClassNameTextRange();

	/**
	 * Returns the short class name of the entity.
	 */
	String getClassShortName();

	/**
	 * Returns the clone copy policy for this entity.
	 */
	ExternalCloneCopyPolicy getCloneCopyPolicy();

	/**
	 * Returns the copy policy for this entity.
	 */
	ExternalCopyPolicy getCopyPolicy();

	/**
	 * Returns the customizer class name.
	 */
	String getCustomizerClassName();

	/**
	 * Returns the {@link TextRange} for the customizer class name value.
	 */
	TextRange getCustomizerClassNameTextRange();

	/**
	 * Returns the description text for this entity.
	 */
	String getDescription();

	/**
	 * Returns the {@link TextRange} for the description text value.
	 */
	TextRange getDescriptionTextRange();

	/**
	 * Returns the {@link TextRange} for the default mappings should be excluded for this entity value.
	 */
	TextRange getExcludeDefaultMappingsTextRange();

	@Deprecated
	int getIndex();

	/**
	 * Returns the instantiation policy for this entity.
	 */
	ExternalInstantiationCopyPolicy getInstantiationCopyPolicy();

	/**
	 * Returns the mapping at the specified position.
	 */
	ExternalMapping getMapping(int index);

	/**
	 * Returns the mapping with the given name.
	 */
	ExternalMapping getMapping(String name);

	/**
	 * Returns the {@link TextRange} for the meta data for this entity is completely described by the
	 * ORM definition value.
	 */
	TextRange getMetadataCompleteTextRange();

	/**
	 * Returns the external form of NoSql.
	 */
	ExternalNoSql getNoSql();

	/**
	 * Returns the name of the parent class of this entity.
	 */
	String getParentClassName();

	/**
	 * Returns the {@link TextRange} for the name of the parent class of this entity.
	 */
	TextRange getParentClassNameTextRange();

	/**
	 * Determines whether the access methods for this mapping is defined or not.
	 */
	boolean hasAccessMethods();

	/**
	 * Determines whether the clone copy policy for this entity is present or not.
	 */
	boolean hasCloneCopyPolicy();

	/**
	 * Determines whether the copy policy for this entity is present or not.
	 */
	boolean hasCopyPolicy();

	/**
	 * Determines whether the instantiation copy policy for this entity is present or not.
	 */
	boolean hasInstantiationCopyPolicy();

	/**
	 * Determines whether the external form of NoSql is present or not.
	 */
	boolean hasNoSql();

	/**
	 * Returns a list of the id mappings defined for this entity.
	 */
	List<ExternalIDMapping> idMappings();

	/**
	 * Indicates whether default mappings should be excluded for this entity.
	 */
	Boolean isExcludeDefaultMappings();

	/**
	 * Indicates whether the meta data for this entity is completely described by the ORM definition.
	 */
	Boolean isMetadataComplete();

	/**
	 * Returns a list of the attribute mappings defined for this entity.
	 */
	List<ExternalMapping> mappings();

	/**
	 * Returns the count of attribute mappings defined for this entity.
	 */
	int mappingsSize();

	/**
	 * Removes the access methods element from the mapping.
	 */
	void removeAccessMethods();

	/**
	 * Removes the override named.
	 */
	void removeAssociationOverride(int index);

	/**
	 * Removes the override named.
	 */
	void removeAttributeOverride(int index);

	/**
	 * Removes the clone copy policy from this entity.
	 */
	void removeCloneCopyPolicy();

	/**
	 * Removes the copy policy from this entity.
	 */
	void removeCopyPolicy();

	/**
	 * Removes the instantiation copy policy from this entity.
	 */
	void removeInstantiationCopyPolicy();

	/**
	 * Removes the mapping at the given position.
	 */
	void removeMapping(int index);

	/**
	 * Removes the mapping with the given name.
	 */
	void removeMapping(String name);

	/**
	 * Removes the NoSQL annotation.
	 */
	void removeNoSql();

	/**
	 * Sets the change tracking type for this entity.
	 */
	void setChangeTrackingType(ChangeTrackingType type);

	/**
	 * Sets the class name.
	 */
	void setClassName(String className);

	/**
	 * Sets the customizer class name.
	 */
	void setCustomizerClassName(String name);

	/**
	 * Sets the description.
	 */
	void setDescription(String description);

	/**
	 * Sets whether default mappings should be excluded for this entity.
	 */
	void setExcludeDefaultMappings(Boolean excludeDefaultMappings);

	/**
	 * Sets whether the meta data for this entity is completely described by the ORM definition
	 */
	void setMetadataComplete(Boolean metaDataComplete);

	/**
	 * Sets the name of the parent class of this entity.
	 */
	void setParentClassName(String className);
}