/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalNamedQuery;
import org.eclipse.persistence.tools.mapping.orm.ExternalQueryHint;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The abstract definition of a query, which can either be a named query (JPQL query), a named
 * native query (SQL query), or a named stored procedure query.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.6
 */
abstract class AbstractQuery extends AbstractExternalForm
                             implements ExternalNamedQuery {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>AbstractQuery</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	AbstractQuery(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalQueryHint addQueryHint(String name, String value) {
		QueryHint queryHint = buildQueryHint(-1);
		queryHint.addSelf();
		queryHint.setName(name);
		queryHint.setValue(value);
		return queryHint;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(QUERY);
		names.add(QueryHint.HINT);
		return names;
	}

	private QueryHint buildQueryHint(int index) {
		return new QueryHint(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	@Override
	@Deprecated
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getQuery() {
		return getChildTextNode(QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final QueryHint getQueryHint(int index) {

		if (hasChild(QueryHint.HINT, index)) {
			return buildQueryHint(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getQueryTextRange() {
		return getChildTextNodeTextRange(QUERY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalQueryHint> queryHints() {

		int count = queryHintsSize();
		List<ExternalQueryHint> queryHints = new ArrayList<ExternalQueryHint>(count);

		for (int index = 0; index < count; index++) {
			QueryHint queryHint = buildQueryHint(index);
			queryHints.add(queryHint);
		}

		return queryHints;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int queryHintsSize() {
		return getChildrenSize(QueryHint.HINT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeQueryHint(int index) {
		QueryHint queryHint = buildQueryHint(index);
		queryHint.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setQuery(String query) {
		updateChildTextNode(QUERY, query);
	}
}