/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.InheritanceType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessMethods;
import org.eclipse.persistence.tools.mapping.orm.ExternalDiscriminatorColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalMultitenancyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalSecondaryTable;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an entity, which is a child of an ORM configuration.
 *
 * @see ORMConfiguration
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
final class Entity extends MappedSuperclass
                   implements ExternalEntity {

	/**
	 * Creates a new <code>Entity</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	Entity(ORMConfiguration parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalDiscriminatorColumn addDiscriminatorColumn(String columnName) {

		DiscriminatorColumn column = getDiscriminatorColumn();

		if (column == null) {
			column = buildDiscriminatorColumn();
			column.addSelf();
		}

		column.setName(columnName);
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn addPrimaryKeyJoinColumn(String name) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(-1);
		pkJoinColumn.addSelf();
		pkJoinColumn.setName(name);
		return pkJoinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityTable addPrimaryTable(String tableName) {
		PrimaryTable primaryTable = buildPrimaryTable();
		primaryTable.addSelf();
		primaryTable.setName(tableName);
		return primaryTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSecondaryTable addSecondaryTable(String name, String catalog, String schema) {
		SecondaryTable secondaryTable = buildSecondaryTable(-1);
		secondaryTable.addSelf();
		secondaryTable.setName(name);
		secondaryTable.setCatalogName(catalog);
		secondaryTable.setSchemaName(schema);
		return secondaryTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSQLResultSetMapping addSqlResultSetMapping(String name) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(-1);
		sqlResultSetMapping.addSelf();
		sqlResultSetMapping.setName(name);
		return sqlResultSetMapping;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(ExternalEmbeddable.CLASS);
		names.add(PARENT_CLASS);             // EclipseLink
		names.add(ACCESS);
		names.add(CACHEABLE);
		names.add(METADATA_COMPLETE);
		names.add(READ_ONLY);                // EclipseLink
		names.add(EXISTENCE_CHECKING);       // EclipseLink
		names.add(EXCLUDE_DEFAULT_MAPPINGS); // EclipseLink
		return names;
	}

	private DiscriminatorColumn buildDiscriminatorColumn() {
		return new DiscriminatorColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ExternalEmbeddable.DESCRIPTION);
		names.add(ExternalAccessMethods.ACCESS_METHODS);
		names.add(ExternalMultitenancyPolicy.MULTITENANT);
		names.add("additional-criteria");
		names.add(CUSTOMIZER);
		names.add(CHANGE_TRACKING);
		names.add(PrimaryTable.TABLE);
		names.add(SecondaryTable.SECONDARY_TABLE);
		names.add("struct");
		names.add(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
		names.add("primary-key-foreign-key");
		names.add(NoSql.NO_SQL);
		names.add("cascade-on-delete");
		names.add("index");
		names.add(ID_CLASS);
		names.add(PrimaryKey.PRIMARY_KEY);
		names.add(INHERITANCE);
		names.add(DISCRIMINATOR_VALUE);
		names.add(DiscriminatorColumn.DISCRIMINATOR_COLUMN);
		names.add(CLASS_EXTRACTOR);
		names.add(OptimisticLocking.OPTIMISTIC_LOCKING);
		names.add(Cache.CACHE);
		names.add(CACHE_INTERCEPTOR);
		names.add("cache-index");
		names.add(FetchGroup.FETCH_GROUP);
		names.add(ClassConverter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add("serialized-converter");
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(CopyPolicy.COPY_POLICY);
		names.add(InstantiationCopyPolicy.INSTANTIATION_COPY_POLICY);
		names.add(CloneCopyPolicy.CLONE_COPY_POLICY);
		names.add("serialized-object");
		names.add(SequenceGenerator.SEQUENCE_GENERATOR);
		names.add(TableGenerator.TABLE_GENERATOR);
		names.add("uuid-generator");
		names.add("partitioning");
		names.add("replication-partitioning");
		names.add("round-robin-partitioning");
		names.add("pinned-partitioning");
		names.add("range-partitioning");
		names.add("value-partitioning");
		names.add("hash-partitioning");
		names.add("union-partitioning");
		names.add("partitioned");
		names.add(NamedQuery.NAMED_QUERY);
		names.add(NamedNativeQuery.NAMED_NATIVE_QUERY);
		names.add(NamedStoredProcedureQuery.NAMED_STORED_PROCEDURE_QUERY);
		names.add("named-stored-function-query");
		names.add("named-plsql-stored-procedure-query");
		names.add("named-plsql-stored-function-query");
		names.add("oracle-object");
		names.add("oracle-array");
		names.add("plsql-record");
		names.add("plsql-table");
		names.add(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
		names.add("query-redirectors");
		names.add(EXCLUDE_DEFAULT_LISTENERS);
		names.add(EXCLUDE_SUPERCLASS_LISTENERS);
		names.add(AbstractEntityListener.ENTITY_LISTENERS);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		names.add(Property.PROPERTY);
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add("convert");
		names.add("named-entity-graph");
		names.add(Mapping.ATTRIBUTES);
		return names;
	}

	private PrimaryKeyJoinColumn buildPrimaryKeyJoinColumn(int index) {
		return new PrimaryKeyJoinColumn(this, index);
	}

	private PrimaryTable buildPrimaryTable() {
		return new PrimaryTable(this);
	}

	private SecondaryTable buildSecondaryTable(int index) {
		return new SecondaryTable(this, index);
	}

	private SQLResultSetMapping buildSqlResultSetMapping(int index) {
		return new SQLResultSetMapping(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getClassExtractorClassName() {
		return getChildAttribute(CLASS_EXTRACTOR, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getClassExtractorClassNameTextRange() {
		return getChildAttributeTextRange(CLASS_EXTRACTOR, ExternalEmbeddable.CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DiscriminatorColumn getDiscriminatorColumn() {

		if (hasChild(DiscriminatorColumn.DISCRIMINATOR_COLUMN)) {
			return buildDiscriminatorColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDiscriminatorValue() {
		return getChildTextNode(DISCRIMINATOR_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDiscriminatorValueTextRange() {
		return getChildTextNodeTextRange(DISCRIMINATOR_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InheritanceType getInheritenceStrategy() {

		Element element = getChild(INHERITANCE);

		if (element != null) {
			return getEnumAttribute(element, STRATEGY, InheritanceType.class);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getInheritenceStrategyTextRange() {
		return getChildAttributeTextRange(INHERITANCE, STRATEGY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index) {

		if (hasChild(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN, index)) {
			return buildPrimaryKeyJoinColumn(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityTable getPrimaryTable() {

		if (hasPrimaryTable()) {
			return buildPrimaryTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSecondaryTable getSecondaryTable(int index) {

		if (hasChild(SecondaryTable.SECONDARY_TABLE)) {
			return buildSecondaryTable(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalSQLResultSetMapping getSqlResultSetMapping(int index) {

		if (hasChild(SQLResultSetMapping.SQL_RESULT_SET_MAPPING, index)) {
			return buildSqlResultSetMapping(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasPrimaryTable() {
		return hasChild(PrimaryTable.TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns() {

		int count = primaryKeyJoinColumnsSize();
		List<ExternalPrimaryKeyJoinColumn> pkJoinColumns = new ArrayList<ExternalPrimaryKeyJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			pkJoinColumns.add(buildPrimaryKeyJoinColumn(index));
		}

		return pkJoinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int primaryKeyJoinColumnsSize() {
		return getChildrenSize(PrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryKeyJoinColumn(int index) {
		PrimaryKeyJoinColumn pkJoinColumn = buildPrimaryKeyJoinColumn(index);
		pkJoinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePrimaryTable() {
		PrimaryTable primaryTable = buildPrimaryTable();
		primaryTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSecondaryTable(int index) {
		SecondaryTable secondaryTable = buildSecondaryTable(index);
		secondaryTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSqlResultSetMapping(int index) {
		SQLResultSetMapping sqlResultSetMapping = buildSqlResultSetMapping(index);
		sqlResultSetMapping.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalSecondaryTable> secondaryTables() {

		int count = secondaryTablesSize();
		List<ExternalSecondaryTable> secondaryTables = new ArrayList<ExternalSecondaryTable>(count);

		for (int index = 0; index < count; index++) {
			secondaryTables.add(buildSecondaryTable(index));
		}

		return secondaryTables;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int secondaryTablesSize() {
		return getChildrenSize(SecondaryTable.SECONDARY_TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setClassExtractorClassName(String className) {
		updateChildAttribute(CLASS_EXTRACTOR, ExternalEmbeddable.CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDiscriminatorValue(String discriminatorValue) {
		updateChildTextNode(DISCRIMINATOR_VALUE, discriminatorValue);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setInheritanceStrategy(InheritanceType value) {
		updateChildAttribute(INHERITANCE, STRATEGY, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalSQLResultSetMapping> sqlResultSetMappings() {

		int count = sqlResultSetMappingsSize();
		List<ExternalSQLResultSetMapping> sqlResultSetMappings = new ArrayList<ExternalSQLResultSetMapping>(count);

		for (int index = 0; index < count; index++) {
			sqlResultSetMappings.add(buildSqlResultSetMapping(index));
		}

		return sqlResultSetMappings;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int sqlResultSetMappingsSize() {
		return getChildrenSize(SQLResultSetMapping.SQL_RESULT_SET_MAPPING);
	}
}