/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalConversionValue;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a object type converter.
 *
 * @version 2.6
 */
final class ObjectTypeConverter extends AbstractConverter
                                implements ExternalObjectTypeConverter {

	/**
	 * Creates a new <code>ObjectTypeConverter</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	ObjectTypeConverter(AbstractExternalForm parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConversionValue addConversionValue() {
		ConversionValue conversionValue = buildConversionValue(-1);
		conversionValue.addSelf();
		return conversionValue;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(DATA_TYPE);
		names.add(OBJECT_TYPE);
		return names;
	}

	private ConversionValue buildConversionValue(int index) {
		return new ConversionValue(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ConversionValue.CONVERSION_VALUE);
		names.add(DEFAULT_OBJECT_VALUE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalConversionValue> conversionValues() {

		int count = conversionValuesSize();
		List<ExternalConversionValue> conversionValues = new ArrayList<ExternalConversionValue>(count);

		for (int index = 0; index < count; index++) {
			conversionValues.add(buildConversionValue(index));
		}

		return conversionValues;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int conversionValuesSize() {
		return getChildrenSize(ConversionValue.CONVERSION_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConversionValue getConversionValue(int index) {

		if (hasChild(ConversionValue.CONVERSION_VALUE, index)) {
			return buildConversionValue(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatatype() {
		return getAttribute(DATA_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDatatypeTextRange() {
		return getAttributeTextRange(DATA_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultValue() {
		return getChildTextNode(DEFAULT_OBJECT_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDefaultValueTextRange() {
		return getChildTextNodeTextRange(DEFAULT_OBJECT_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return OBJECT_TYPE_CONVERTER;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjecttype() {
		return getAttribute(OBJECT_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getObjecttypeTextRange() {
		return getAttributeTextRange(OBJECT_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAllConversionValue() {
		removeChildren(ConversionValue.CONVERSION_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeConversionValue(int index) {
		ConversionValue conversionValue = buildConversionValue(index);
		conversionValue.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDatatype(String datatypeName) {
		setAttribute(DATA_TYPE, datatypeName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultValue(String value) {
		updateChildTextNode(DEFAULT_OBJECT_VALUE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setObjecttype(String objecttypeName) {
		setAttribute(OBJECT_TYPE, objecttypeName);
	}
}