/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalColumnResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityResult;
import org.eclipse.persistence.tools.mapping.orm.ExternalSQLResultSetMapping;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an SQL Result Set.
 *
 * @see Entity
 * @see ORMConfiguration
 *
 * @version 2.6
 */
final class SQLResultSetMapping extends AbstractExternalForm
                                implements ExternalSQLResultSetMapping {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>SQLResultSetMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	SQLResultSetMapping(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalColumnResult addColumnResult(String name) {
		ColumnResult entityResult = buildColumnResult(-1);
		entityResult.addSelf();
		entityResult.setName(name);
		return entityResult;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityResult addEntityResult(String className) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.addSelf();
		entityResult.setEntityClassName(className);
		return entityResult;
	}

	private ColumnResult buildColumnResult(int index) {
		return new ColumnResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(EntityResult.ENTITY_RESULT);
		names.add(ColumnResult.COLUMN_RESULT);
		return names;
	}

	private EntityResult buildEntityResult(int index) {
		return new EntityResult(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalColumnResult> columnResults() {

		int count = columnResultsSize();
		List<ExternalColumnResult> columnResults = new ArrayList<ExternalColumnResult>(count);

		for (int index = 0; index < count; index++) {
			columnResults.add(buildColumnResult(index));
		}

		return columnResults;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int columnResultsSize() {
		return getChildrenSize(ColumnResult.COLUMN_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalEntityResult> entityResults() {

		int count = entityResultsSize();
		List<ExternalEntityResult> entityResults = new ArrayList<ExternalEntityResult>(count);

		for (int index = 0; index < count; index++) {
			entityResults.add(buildEntityResult(index));
		}

		return entityResults;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int entityResultsSize() {
		return getChildrenSize(EntityResult.ENTITY_RESULT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalColumnResult getColumnResult(int index) {

		if (hasChild(ColumnResult.COLUMN_RESULT, index)) {
			return buildColumnResult(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return SQL_RESULT_SET_MAPPING;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityResult getEntityResult(int index) {

		if (hasChild(EntityResult.ENTITY_RESULT, index)) {
			return buildEntityResult(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumnResult(int index) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntityResult(int index) {
		EntityResult entityResult = buildEntityResult(index);
		entityResult.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}
}