/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalUniqueConstraint;
import org.eclipse.persistence.tools.utility.TextRange;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * The external form of an entity table.
 *
 * @see RelationshipMapping
 *
 * @version 2.6
 */
abstract class Table extends AbstractExternalForm
                     implements ExternalEntityTable {

	/**
	 * Creates a new <code>Table</code>.
	 *
	 * @param parent The parent of this external form
	 */
	Table(AbstractExternalForm parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalUniqueConstraint addUniqueConstraint(ListIterable<String> columnNames) {
		UniqueConstraint uniqueConstraint = buildUniqueConstraint(-1);
		uniqueConstraint.addSelf();
		uniqueConstraint.setColumnNames(columnNames);
		return uniqueConstraint;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(CATALOG);
		names.add(SCHEMA);
		return names;
	}

	private UniqueConstraint buildUniqueConstraint(int index) {
		return new UniqueConstraint(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getCatalogName() {
		return getAttribute(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getCatalogTextRange() {
		return getAttributeTextRange(CATALOG);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getSchemaName() {
		return getAttribute(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getSchemaTextRange() {
		return getAttributeTextRange(SCHEMA);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final ExternalUniqueConstraint getUniqueConstraint(int index) {

		if (hasChild(UniqueConstraint.UNIQUE_CONSTRAINT, index)) {
			return buildUniqueConstraint(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeAllUniqueConstraints() {
		removeChildren(UniqueConstraint.UNIQUE_CONSTRAINT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeUniqueConstraint(int index) {
		UniqueConstraint uniqueConstraint = buildUniqueConstraint(index);
		uniqueConstraint.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setCatalogName(String catalog) {
		setAttribute(CATALOG, catalog);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setSchemaName(String name) {
		setAttribute(SCHEMA, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final List<ExternalUniqueConstraint> uniqueConstraints() {

		int count = uniqueConstraintsSize();
		List<ExternalUniqueConstraint> uniqueConstraints = new ArrayList<ExternalUniqueConstraint>(count);

		for (int index = 0; index < count; index++) {
			uniqueConstraints.add(buildUniqueConstraint(index));
		}

		return uniqueConstraints;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final int uniqueConstraintsSize() {
		return getChildrenSize(UniqueConstraint.UNIQUE_CONSTRAINT);
	}
}