/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import javax.persistence.ValidationMode;
import org.eclipse.persistence.config.PersistenceUnitProperties;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.ExternalProperty;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit.
 *
 * @see PersistenceConfiguration
 *
 * @version 2.6
 */
final class PersistenceUnit extends AbstractExternalForm
                            implements ExternalPersistenceUnit {

	/**
	 * The position of this external form within the list of persistence units contained in the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>PersistenceUnit</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of this external form within the parent's children
	 */
	PersistenceUnit(PersistenceConfiguration parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty addProperty(String name, String value) {
		Property property = buildProperty(-1);
		property.addSelf();
		property.setName(name);
		property.setValue(value);
		return property;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(PROVIDER);
		names.add(JTA_DATA_SOURCE);
		names.add(NON_JTA_DATA_SOURCE);
		names.add(MAPPING_FILE);
		names.add(JAR_FILE);
		names.add(CLASS);
		names.add(EXCLUDE_UNLISTED_CLASSES);
		names.add(CACHE_TYPE);
		names.add(VALIDATION_MODE);
		names.add(PROPERTIES);
		return names;
	}

	private Property buildProperty(int index) {
		return new Property(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheType getCachType() {
		return getChildEnumNode(CACHE_TYPE, CacheType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCachTypeTextRange() {
		return getChildTextNodeTextRange(CACHE_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PERSISTENCE_UNIT;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getIncludedEntityClassTextRange(String className) {
		return getChildTextNodeTextRange(CLASS, className);
	}

	@Override
	@Deprecated
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJarFilePathTextRange(int index, String path) {
		return getChildTextNodeTextRange(JAR_FILE, index, path);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcDriver() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_DRIVER);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJdbcDriverTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_DRIVER);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcPassword() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_PASSWORD);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJdbcPasswordTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_PASSWORD);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcUrl() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_URL);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJdbcUrlTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_URL);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJdbcUser() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_USER);
		return (property != null) ? property.getValue() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJdbcUserTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.JDBC_USER);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getJtaDatasource() {
		return getChildTextNode(JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getJtaDatasourceTextRange() {
		return getChildTextNodeTextRange(JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMappedClassTextRange(String name) {
		return getChildTextNodeTextRange(getElement(), CLASS, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMappingFilePathTextRange(int index, String path) {
		return getChildTextNodeTextRange(MAPPING_FILE, index, path);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNonJtaDatasource() {
		return getChildTextNode(NON_JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getNonJtaDatasourceTextRange() {
		return getChildTextNodeTextRange(NON_JTA_DATA_SOURCE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getPessimisticLockTimeout() {

		ExternalProperty property = getProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT);

		if (property != null) {
			try {
				return Integer.valueOf(property.getValue());
			}
			catch (NumberFormatException ex) {
				return null;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPessimisticLockTimeoutTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalProperty> getProperties(String name) {

		List<ExternalProperty> properties = new ArrayList<ExternalProperty>();

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				properties.add(property);
			}
		}

		return properties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(int index) {

		Element element = getChild(PROPERTIES);

		if (element == null) {
			return null;
		}

		element = getChild(element, Property.PROPERTY, index);

		if (element == null) {
			return null;
		}

		return buildProperty(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name, int index) {

		ExternalProperty property = getProperty(index);

		if ((property != null) && ObjectTools.equals(name, property.getName())) {
			return property;
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalProperty getProperty(String name, String value) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(),  name) &&
			    ObjectTools.equals(property.getValue(), value)) {

				return property;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPropertyNameTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getNameTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPropertyTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPropertyTextRange(String name, String value) {

		for (ExternalProperty property : properties()) {

			if (ObjectTools.equals(property.getName(),  name) &&
			    ObjectTools.equals(property.getValue(), value)) {

				return property.getValueTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPropertyValueTextRange(String name) {

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				return property.getTextRange();
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getProvider() {
		return getChildTextNode(PROVIDER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getProviderTextRange() {
		return getChildTextNodeTextRange(PROVIDER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getQueryTimeout() {

		ExternalProperty property = getProperty(PersistenceUnitProperties.QUERY_TIMEOUT);

		if (property != null) {
			try {
				return Integer.valueOf(property.getValue());
			}
			catch (NumberFormatException ex) {
				return null;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getQueryTimeoutTextRange() {
		ExternalProperty property = getProperty(PersistenceUnitProperties.QUERY_TIMEOUT);
		return (property != null) ? property.getValueTextRange() : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TransactionType getTransactionType() {
		return getEnumAttribute(TRANSACTION_TYPE, TransactionType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTransactionTypeTextRange() {
		return getAttributeTextRange(TRANSACTION_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ValidationMode getValidationMode() {
		return getChildEnumNode(VALIDATION_MODE, ValidationMode.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getValidationModeTextRange() {
		return getChildTextNodeTextRange(VALIDATION_MODE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> includedClasses() {
		return getChildrenTextNode(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int includedClassesSize() {
		return getChildrenSize(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> jarFiles() {
		return getChildrenTextNode(JAR_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int jarFilesSize() {
		return getChildrenSize(JAR_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> mappingFiles() {
		return getChildrenTextNode(MAPPING_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mappingFilesSize() {
		return getChildrenSize(MAPPING_FILE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalProperty> properties() {

		int count = propertiesSize();
		List<ExternalProperty> properties = new ArrayList<ExternalProperty>(count);

		for (int index = 0; index < count; index++) {
			properties.add(buildProperty(index));
		}

		return properties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int propertiesSize() {

		Element element = getChild(PROPERTIES);

		if (element != null) {
			return getChildrenSize(element, Property.PROPERTY);
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int propertiesSize(String name) {

		int count = 0;

		for (ExternalProperty property : properties()) {
			if (ObjectTools.equals(property.getName(), name)) {
				count++;
			}
		}

		return count;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeProperty(int index) {
		Property property = buildProperty(index);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeProperty(String name) {
		Property property = (Property) getProperty(name);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeProperty(String name, String value) {
		Property property = (Property) getProperty(name, value);
		property.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCacheType(CacheType type) {
		updateChildTextNode(CACHE_TYPE, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExcludeUnlistedClasses(boolean excludeUnlistedClasses) {
		updateChildTextNode(EXCLUDE_UNLISTED_CLASSES, excludeUnlistedClasses);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIncludedClasses(ListIterator<String> classes) {

		removeChildren(CLASS);

		while (classes.hasNext()) {
			addChildTextNode(CLASS, classes.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJarFiles(ListIterator<String> jarFiles) {

		removeChildren(JAR_FILE);

		while (jarFiles.hasNext()) {
			addChildTextNode(JAR_FILE, jarFiles.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJtaDatasource(String jtaDatasource) {
		updateChildTextNode(JTA_DATA_SOURCE, jtaDatasource);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappingFiles(ListIterator<String> mappingFiles) {

		removeChildren(MAPPING_FILE);

		while (mappingFiles.hasNext()) {
			addChildTextNode(MAPPING_FILE, mappingFiles.next());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNonJtaDatasource(String nonJtaDatasource) {
		updateChildTextNode(NON_JTA_DATA_SOURCE, nonJtaDatasource);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPessimisticLockTimeout(Integer length) {

		ExternalProperty property = getProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT);

		if ((length == null) && (property != null)) {
			removeProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT, property.getValue());
		}
		else if ((length != null) && (property == null)) {
			addProperty(PersistenceUnitProperties.PESSIMISTIC_LOCK_TIMEOUT, length.toString());
		}
		else if ((length != null) && (property != null)) {
			property.setValue(length.toString());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setProvider(String provider) {
		updateChildTextNode(PROVIDER, provider);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setQueryTimeout(Integer length) {

		ExternalProperty property = getProperty(PersistenceUnitProperties.QUERY_TIMEOUT);

		if ((length == null) && (property != null)) {
			removeProperty(PersistenceUnitProperties.QUERY_TIMEOUT, property.getValue());
		}
		else if ((length != null) && (property == null)) {
			addProperty(PersistenceUnitProperties.QUERY_TIMEOUT, length.toString());
		}
		else if ((length != null) && (property != null)) {
			property.setValue(length.toString());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTransactionType(TransactionType transactionType) {
		setAttribute(TRANSACTION_TYPE, transactionType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValidationMode(ValidationMode type) {
		updateChildTextNode(VALIDATION_MODE, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean shouldExcludeUnlistedClasses() {
		Boolean value = getChildBooleanNode(EXCLUDE_UNLISTED_CLASSES);
		return (value != null) && value;
	}
}