/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.tests.dom.orm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.persistence.annotations.ChangeTrackingType;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.mapping.ExternalProperty;
import org.eclipse.persistence.tools.mapping.orm.AccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessMethods;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalBasicMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalClassConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalCloneCopyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalCopyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalElementCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddable;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEmbeddedMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalIDMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalInstantiationCopyPolicy;
import org.eclipse.persistence.tools.mapping.orm.ExternalManyToManyMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalManyToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSql;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalOneToManyMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalOneToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransformationMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalTransientMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalVariableOneToOneMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalVersionMapping;
import org.eclipse.persistence.tools.mapping.tests.AbstractExternalFormTests;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.w3c.dom.Node;

/**
 * This unit-tests tests the behavior of {@link ExternalEmbeddable}.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public class EmbeddableTests<T extends ExternalEmbeddable> extends AbstractExternalFormTests<T> {

	final ChildNodeTester<T, ExternalAccessMethods> buildAccessMethodsTester() {
		return AccessMethodsTests.buildAccessMethodsTester();
	}

	final AttributeNodeTester<T, AccessType> buildAccessTester() {
		return AccessTypeTests.buildAccessAttributeNodeTester();
	}

	final ChildListNodeTester<T, ExternalAssociationOverride, String> buildAssociationOverrideTester() {
		return new ChildListNodeTester<T, ExternalAssociationOverride, String>() {
			@Override
			public ExternalAssociationOverride addChild(T form, String value) {
				return form.addAssociationOverride(value);
			}
			@Override
			public ExternalAssociationOverride getChild(T form, int index) {
				return form.getAssociationOverride(index);
			}
			@Override
			public ExternalAssociationOverride getChildForm(List<ExternalAssociationOverride> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalAssociationOverride> getChildren(T form) {
				return form.associationOverrides();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.associationOverridesSize();
			}
			@Override
			public String getChildValue(ExternalAssociationOverride childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "ASSOCIATION_OVERRIDE_" + 1;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalAssociationOverride.ASSOCIATION_OVERRIDE;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeAssociationOverride(index);
			}
		};
	}

	final ChildListNodeTester<T, ExternalAttributeOverride, String> buildAttributeOverrideTester() {
		return new ChildListNodeTester<T, ExternalAttributeOverride, String>() {
			@Override
			public ExternalAttributeOverride addChild(T form, String value) {
				return form.addAttributeOverride(value);
			}
			@Override
			public ExternalAttributeOverride getChild(T form, int index) {
				return form.getAttributeOverride(index);
			}
			@Override
			public ExternalAttributeOverride getChildForm(List<ExternalAttributeOverride> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalAttributeOverride> getChildren(T form) {
				return form.attributeOverrides();
			}
			@Override
			public int getChildrenSize(T form) {
				return form.attributeOverridesSize();
			}
			@Override
			public String getChildValue(ExternalAttributeOverride childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "ATTRIBUTE_OVERRIDE_" + 1;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalAttributeOverride.ATTRIBUTE_OVERRIDE;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(T form, int index) {
				form.removeAttributeOverride(index);
			}
		};
	}

	final ContainerChildListNodeTester<T, ExternalMapping, String> buildAttributesTester() {

		return new ContainerChildListNodeTester<T, ExternalMapping, String>() {

			private List<String> expectedValues;
			private List<String> mappingNames;
			private List<Integer> orderedIndices;
			private List<String> orderedValues;

			{
				orderedIndices        = new ArrayList<Integer>();
				orderedValues         = new ArrayList<String>();
				mappingNames          = buildMappingNames();
				expectedValues        = buildExpectedValues();
			}

			@Override
			public ExternalMapping addChild(T form, String value) {

				// Extract the mapping name
				String mappingName = value.substring(0, value.indexOf("_"));

				// Calculate the position of insertion based on the order
				// of the mapping type defined in the XML schema
				int index = insertionIndex(orderedValues, mappingName);

				// Store the value based on the order of insertion in the document
				orderedValues.add(index, value);

				// Now adjust the positions of insertion
				adjustOrderedIndices(index, +1);

				// Store the index of insertion based on the order defined in the schema
				orderedIndices.add(index);

				// Now add the mapping
				return addMapping(form, value, mappingName);
			}

			private ExternalMapping addMapping(T form,
			                                   String value,
			                                   String mappingName) {

				if (ExternalBasicMapping.BASIC.equals(mappingName)) {
					return form.addBasicMapping(value);
				}

				if (ExternalBasicCollectionMapping.BASIC_COLLECTION.equals(mappingName)) {
					return form.addBasicCollectionMapping(value);
				}

				if (ExternalBasicMapMapping.BASIC_MAP.equals(mappingName)) {
					return form.addBasicMapMapping(value);
				}

				if (ExternalManyToOneMapping.MANY_TO_ONE.equals(mappingName)) {
					return form.addManyToOneMapping(value);
				}

				if (ExternalOneToManyMapping.ONE_TO_MANY.equals(mappingName)) {
					return form.addOneToManyMapping(value);
				}

				if (ExternalOneToOneMapping.ONE_TO_ONE.equals(mappingName)) {
					return form.addOneToOneMapping(value);
				}

				if (ExternalManyToManyMapping.MANY_TO_MANY.equals(mappingName)) {
					return form.addManyToManyMapping(value);
				}

				if (ExternalElementCollectionMapping.ELEMENT_COLLECTION.equals(mappingName)) {
					return form.addElementCollectionMapping(value);
				}

				if (ExternalEmbeddedMapping.EMBEDDED.equals(mappingName)) {
					return form.addEmbeddedMapping(value);
				}

				if (ExternalTransientMapping.TRANSIENT.equals(mappingName)) {
					return form.addTransientMapping(value);
				}

				throw new IllegalAccessError("The value is unknown: " + value);
			}

			private void adjustOrderedIndices(int index, int direction) {

				for (int position = 0, count = orderedIndices.size(); position < count; position++) {
					int value = orderedIndices.get(position);
					if (value >= index) {
						orderedIndices.set(position, value + direction);
					}
				}
			}

			/**
			 * Generates a list of 30 values that is based on the mapping name.
			 * <p>
			 * Format: "&lt;mapping name&gt; _ [ 0, 2 ] _ &lt;unique id&gt;"
			 *
			 * @return A list of unique values
			 */
			private List<String> buildExpectedValues() {

				List<String> values = new ArrayList<String>();

				for (int index = 0; index < 3; index++) {
					for (String mappingName : mappingNames) {
						values.add(mappingName + "_" + index);
					}
				}

				// Now shuffle that list
				Collections.shuffle(values);

				return values;
			}

			private List<String> buildMappingNames() {
				List<String> names = new ArrayList<String>();
				names.add(ExternalBasicMapping.BASIC);
				names.add(ExternalBasicCollectionMapping.BASIC_COLLECTION);
				names.add(ExternalBasicMapMapping.BASIC_MAP);
				names.add(ExternalManyToOneMapping.MANY_TO_ONE);
				names.add(ExternalOneToManyMapping.ONE_TO_MANY);
				names.add(ExternalOneToOneMapping.ONE_TO_ONE);
				names.add(ExternalManyToManyMapping.MANY_TO_MANY);
				names.add(ExternalElementCollectionMapping.ELEMENT_COLLECTION);
				names.add(ExternalEmbeddedMapping.EMBEDDED);
				names.add(ExternalTransientMapping.TRANSIENT);
				return names;
			}

			@Override
			public ExternalMapping getChild(T form, int index) {

				// Retrieve the position of insertion within the document
				index = orderedIndices.get(index);

				// Return the mapping that was added in the order specified by the schema
				return form.getMapping(index);
			}

			@Override
			public ExternalMapping getChildForm(List<ExternalMapping> childForms, int index) {

				// Retrieve the position of insertion within the document
				index = orderedIndices.get(index);

				// Get the child form at the position it was inserted into the document
				return childForms.get(index);
			}

			@Override
			public List<ExternalMapping> getChildren(T form) {
				return form.mappings();
			}

			@Override
			public int getChildrenSize(T form) {
				return form.mappingsSize();
			}

			@Override
			public String getChildValue(ExternalMapping childForm) {
				return childForm.getName();
			}

			@Override
			public String getContainerNodeName() {
				return ExternalMapping.ATTRIBUTES;
			}

			@Override
			public String getExpectedChildValue(int index) {
				// Return the mapping name that is ordered based on the mapping type
				return expectedValues.get(index);
			}

			@Override
			public List<String> getExpectedChildValues() {
				// Returned the generate list of mapping names
				return expectedValues;
			}

			@Override
			public String getNodeName() {
				return ExternalMapping.ATTRIBUTES;
			}

			@Override
			public List<String> getNodeNames() {
				List<String> names = new ArrayList<String>();
				names.add(ExternalEmbeddable.DESCRIPTION);
				names.add(ExternalIDMapping.ID);
				names.add(ExternalEmbeddedIDMapping.EMBEDDED_ID);
				names.add(ExternalBasicMapping.BASIC);
				names.add(ExternalBasicCollectionMapping.BASIC_COLLECTION);
				names.add(ExternalBasicMapMapping.BASIC_MAP);
				names.add(ExternalVersionMapping.VERSION);
				names.add(ExternalManyToOneMapping.MANY_TO_ONE);
				names.add(ExternalOneToManyMapping.ONE_TO_MANY);
				names.add(ExternalOneToOneMapping.ONE_TO_ONE);
				names.add(ExternalVariableOneToOneMapping.VARIABLE_ONE_TO_ONE);
				names.add(ExternalManyToManyMapping.MANY_TO_MANY);
				names.add(ExternalElementCollectionMapping.ELEMENT_COLLECTION);
				names.add(ExternalEmbeddedMapping.EMBEDDED);
				names.add(ExternalTransformationMapping.TRANSFORMATION);
				names.add(ExternalTransientMapping.TRANSIENT);
				names.add("structure");
				names.add("array");
				return names;
			}

			private int insertionIndex(List<String> values, String mappingType) {

				// First get the index of the mapping type within the list of registered mapping types
				int mappingTypeIndex = mappingNames.indexOf(mappingType);

				// Iterate through the list of cached values
				for (int index = 0; index < values.size(); index++) {

					String value = values.get(index);
					String type = value.substring(0, value.indexOf("_"));

					int otherMappingTypeIndex = mappingNames.indexOf(type);

					if (mappingTypeIndex < otherMappingTypeIndex) {
						return index;
					}
				}

				return values.size();
			}

			@Override
			public void removeChild(T form, int index) {

				// Retrieve the position of insertion within the document
				index = orderedIndices.remove(index);
				orderedValues.remove(index);

				// Remove the mapping that was added in the order specified by the schema
				form.removeMapping(index);

				// Now adjust the positions of insertion
				adjustOrderedIndices(index, -1);
			}
		};
	}

	final ChildAttributeTester<T, ChangeTrackingType> buildChangeTrackingTester() {
		return new ChildAttributeTester<T, ChangeTrackingType>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalEmbeddable.CHANGE_TRACKING;
			}
			@Override
			public ChangeTrackingType getDefaultValue() {
				return ChangeTrackingType.AUTO;
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.TYPE;
			}
			@Override
			public ChangeTrackingType getValue(T form) {
				return form.getChangeTrackingType();
			}
			@Override
			public ChangeTrackingType getValue1() {
				return ChangeTrackingType.ATTRIBUTE;
			}
			@Override
			public ChangeTrackingType getValue2() {
				return ChangeTrackingType.DEFERRED;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, ChangeTrackingType value) {
				form.setChangeTrackingType(value);
			}
			@Override
			public String toString(ChangeTrackingType value) {
				return value.name();
			}
		};
	}

	final AttributeNodeTester<T, String> buildClassTester() {
		return new AttributeNodeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return true;
			}
			@Override
			public String getDefaultValue() {
				return defaultClassName();
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.CLASS;
			}
			@Override
			public String getValue(T form) {
				return form.getClassName();
			}
			@Override
			public String getValue1() {
				return "org.test.Embeddable_1";
			}
			@Override
			public String getValue2() {
				return "org.test.Embeddable_2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return false;
			}
			@Override
			public void setValue(T form, String value) {
				form.setClassName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ChildNodeTester<T, ExternalCloneCopyPolicy> buildCloneCopyPolicyTester() {
		return new ChildNodeTester<T, ExternalCloneCopyPolicy>() {
			@Override
			public ExternalCloneCopyPolicy addChild(T form) {
				return form.addCloneCopyPolicy();
			}
			@Override
			public ExternalCloneCopyPolicy getChild(T form) {
				return form.getCloneCopyPolicy();
			}
			@Override
			public String getNodeName() {
				return ExternalCloneCopyPolicy.CLONE_COPY_POLICY;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasCloneCopyPolicy();
			}
			@Override
			public void removeChild(T form) {
				form.removeCloneCopyPolicy();
			}
		};
	}

	final ChildListNodeTester<T, ExternalClassConverter, String> buildConverterTester() {
		return ConverterTests.buildConverterTester();
	}

	final ChildNodeTester<T, ExternalCopyPolicy> buildCopyPolicyTester() {
		return new ChildNodeTester<T, ExternalCopyPolicy>() {
			@Override
			public ExternalCopyPolicy addChild(T form) {
				return form.addCopyPolicy();
			}
			@Override
			public ExternalCopyPolicy getChild(T form) {
				return form.getCopyPolicy();
			}
			@Override
			public String getNodeName() {
				return ExternalCopyPolicy.COPY_POLICY;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasCopyPolicy();
			}
			@Override
			public void removeChild(T form) {
				form.removeCopyPolicy();
			}
		};
	}

	final ChildAttributeTester<T, String> buildCustomizerTester() {
		return new ChildAttributeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalEmbeddable.CUSTOMIZER;
			}
			@Override
			public String getDefaultValue() {
				return "org.default.MyCustomizer";
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.CLASS;
			}
			@Override
			public String getValue(T form) {
				return form.getCustomizerClassName();
			}
			@Override
			public String getValue1() {
				return "org.test.Customizer1";
			}
			@Override
			public String getValue2() {
				return "org.test.Customizer2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setCustomizerClassName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final TextNodeTester<T, String> buildDescriptionTester() {
		return new TextNodeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getDefaultValue() {
				return "test default";
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.DESCRIPTION;
			}
			@Override
			public String getValue(T form) {
				return form.getDescription();
			}
			@Override
			public String getValue1() {
				return "testing embeddable";
			}
			@Override
			public String getValue2() {
				return "something else";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setDescription(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final ExternalFormBuilder<T> buildEmbeddableBuilder() {
		return new ExternalFormBuilder<T>() {
			@Override
			@SuppressWarnings("unchecked")
			public T buildExternalForm() throws IOException {
				ExternalORMConfiguration orm = ORMConfigurationTests.buildExternalForm();
				return (T) orm.addEmbeddable(defaultClassName());
			}
			@Override
			@SuppressWarnings("unchecked")
			public T buildExternalForm(ExternalForm parentForm) {
				return (T) ((ExternalORMConfiguration) parentForm).addEmbeddable(defaultClassName());
			}
			@Override
			public int getDefaultAttributeCount() {
				return 1;
			}
			@Override
			public Node getNode(T form) {
				return ObjectTools.execute(form, "getElement");
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.EMBEDDABLE;
			}
			@Override
			public List<String> getTreeNodeNames() {
				return ListTools.list(
					ExternalORMConfiguration.ENTITY_MAPPINGS,
					ExternalEmbeddable.EMBEDDABLE
				);
			}
		};
	}

	final ChildNodeTester<T, ExternalInstantiationCopyPolicy> buildInstantiationCopyPolicyTester() {
		return new ChildNodeTester<T, ExternalInstantiationCopyPolicy>() {
			@Override
			public ExternalInstantiationCopyPolicy addChild(T form) {
				return form.addInstantiationCopyPolicy();
			}
			@Override
			public ExternalInstantiationCopyPolicy getChild(T form) {
				return form.getInstantiationCopyPolicy();
			}
			@Override
			public String getNodeName() {
				return ExternalInstantiationCopyPolicy.INSTANTIATION_COPY_POLICY;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasInstantiationCopyPolicy();
			}
			@Override
			public void removeChild(T form) {
				form.removeInstantiationCopyPolicy();
			}
		};
	}

	final AttributeNodeTester<T, Boolean> buildMetadataCompleteTester() {
		return new AttributeNodeTester<T, Boolean>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public Boolean getDefaultValue() {
				return Boolean.TRUE;
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.METADATA_COMPLETE;
			}
			@Override
			public Boolean getValue(T form) {
				return form.isMetadataComplete();
			}
			@Override
			public Boolean getValue1() {
				return Boolean.TRUE;
			}
			@Override
			public Boolean getValue2() {
				return Boolean.FALSE;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, Boolean value) {
				form.setMetadataComplete(value);
			}
			@Override
			public String toString(Boolean value) {
				return value.toString();
			}
		};
	}

	final ChildNodeTester<T, ExternalNoSql> buildNoSqlTester() {
		return new ChildNodeTester<T, ExternalNoSql>() {
			@Override
			public ExternalNoSql addChild(T form) {
				return form.addNoSql();
			}
			@Override
			public ExternalNoSql getChild(T form) {
				return form.getNoSql();
			}
			@Override
			public String getNodeName() {
				return ExternalNoSql.NO_SQL;
			}
			@Override
			public boolean hasChild(T form) {
				return form.hasNoSql();
			}
			@Override
			public void removeChild(T form) {
				form.removeNoSql();
			}
		};
	}

	final ChildListNodeTester<T, ExternalObjectTypeConverter, String> buildObjectTypeConverterTester() {
		return ConverterTests.buildObjectTypeConverterTester();
	}

	final String buildOracleArrayTester() {
		return "oracle-array";
	}

	final String buildOracleObjectTester() {
		return "oracle-object";
	}

	final AttributeNodeTester<T, String> buildParentClassTester() {
		return new AttributeNodeTester<T, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getDefaultValue() {
				return "org.parent.ParentClass";
			}
			@Override
			public String getNodeName() {
				return ExternalEmbeddable.PARENT_CLASS;
			}
			@Override
			public String getValue(T form) {
				return form.getParentClassName();
			}
			@Override
			public String getValue1() {
				return "org.test.ParentClass1";
			}
			@Override
			public String getValue2() {
				return "org.test.ParentClass2";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(T form, String value) {
				form.setParentClassName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	final String buildPlsqlRecordTester() {
		return "plsql-record";
	}

	final String buildPlsqlTableTester() {
		return "plsql-table";
	}

	final ChildListNodeTester<T, ExternalProperty, String> buildPropertyTester() {
		return PropertyTests.buildPropertyTester();
	}

	final String buildSerializedConverterTester() {
		return ConverterTests.buildSerializedConverterTester();
	}

	final ChildListNodeTester<T, ExternalStructConverter, String> buildStructConverterTester() {
		return ConverterTests.buildStructConverterTester();
	}

	final String buildStructTester() {
		return "struct";
	}

	final ChildListNodeTester<T, ExternalTypeConverter, String> buildTypeConverterTester() {
		return ConverterTests.buildTypeConverterTester();
	}

	String defaultClassName() {
		return "org.eclipse.persistence.tool.mappings.tests.Embeddable";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void populate(RootNodeTester tester) {

		tester.setBuilder(buildEmbeddableBuilder());

		// Attributes
		tester.addAttribute(buildClassTester());
		tester.addAttribute(buildParentClassTester());                    // EclipseLink
		tester.addAttribute(buildAccessTester());
		tester.addAttribute(buildMetadataCompleteTester());

		// Child nodes
		tester.addTextNode(buildDescriptionTester());
		tester.addNode(buildAccessMethodsTester());                       // EclipseLink
		tester.addChildAttribute(buildCustomizerTester());                // EclipseLink
		tester.addChildAttribute(buildChangeTrackingTester());            // EclipseLink
		tester.addUnsupportedNode(buildStructTester());                   // EclipseLink
		tester.addNode(buildNoSqlTester());                               // EclipseLink
		tester.addListNodes(buildConverterTester());                      // EclipseLink
		tester.addListNodes(buildTypeConverterTester());                  // EclipseLink
		tester.addListNodes(buildObjectTypeConverterTester());            // EclipseLink
		tester.addUnsupportedAttribute(buildSerializedConverterTester()); // EclipseLink
		tester.addListNodes(buildStructConverterTester());                // EclipseLink
		tester.addNode(buildCopyPolicyTester());                          // EclipseLink
		tester.addNode(buildInstantiationCopyPolicyTester());             // EclipseLink
		tester.addNode(buildCloneCopyPolicyTester());                     // EclipseLink
		tester.addUnsupportedNode(buildOracleObjectTester());             // EclipseLink
		tester.addUnsupportedNode(buildOracleArrayTester());              // EclipseLink
		tester.addUnsupportedNode(buildPlsqlRecordTester());              // EclipseLink
		tester.addUnsupportedNode(buildPlsqlTableTester());               // EclipseLink
		tester.addListNodes(buildPropertyTester());                       // EclipseLink
		tester.addListNodes(buildAttributeOverrideTester());              // EclipseLink
		tester.addListNodes(buildAssociationOverrideTester());            // EclipseLink
		tester.addContainerListNodes(buildAttributesTester());
	}
}