/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.tests.dom.orm;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.persistence.InheritanceType;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntity;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalORMConfiguration;
import org.eclipse.persistence.tools.mapping.orm.ExternalPrimaryKeyJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalSecondaryTable;
import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.collection.ListTools;
import org.w3c.dom.Node;

/**
 * This unit-tests tests the behavior of {@link ExternalEntity}.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class EntityTests extends MappedSuperclassTests<ExternalEntity> {

	private String buildCascadeOnDeleteTester() {
		return "cascade-on-delete";
	}

	private ExternalFormBuilder<ExternalEntity> buildEntityBuilder() {
		return new ExternalFormBuilder<ExternalEntity>() {
			@Override
			public ExternalEntity buildExternalForm() throws IOException {
				ExternalORMConfiguration orm = ORMConfigurationTests.buildExternalForm();
				return orm.addEntity(defaultClassName());
			}
			@Override
			public ExternalEntity buildExternalForm(ExternalForm parentForm) {
				return ((ExternalORMConfiguration) parentForm).addEntity(defaultClassName());
			}
			@Override
			public int getDefaultAttributeCount() {
				return 1;
			}
			@Override
			public Node getNode(ExternalEntity form) {
				return ObjectTools.execute(form, "getElement");
			}
			@Override
			public String getNodeName() {
				return ExternalEntity.ENTITY;
			}
			@Override
			public List<String> getTreeNodeNames() {
				return ListTools.list(
					ExternalORMConfiguration.ENTITY_MAPPINGS,
					ExternalEntity.ENTITY
				);
			}
		};
	}

	private String buildIndexTester() {
		return "index";
	}

	private ChildAttributeTester<ExternalEntity, InheritanceType> buildInheritanceTester() {
		return new ChildAttributeTester<ExternalEntity, InheritanceType>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getChildNodeName() {
				return ExternalEntity.INHERITANCE;
			}
			@Override
			public InheritanceType getDefaultValue() {
				return InheritanceType.JOINED;
			}
			@Override
			public String getNodeName() {
				return ExternalEntity.STRATEGY;
			}
			@Override
			public InheritanceType getValue(ExternalEntity form) {
				return form.getInheritenceStrategy();
			}
			@Override
			public InheritanceType getValue1() {
				return InheritanceType.SINGLE_TABLE;
			}
			@Override
			public InheritanceType getValue2() {
				return InheritanceType.TABLE_PER_CLASS;
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalEntity form, InheritanceType value) {
				form.setInheritanceStrategy(value);
			}
			@Override
			public String toString(InheritanceType value) {
				return value.name();
			}
		};
	}

	private AttributeNodeTester<ExternalEntity, String> buildNameTester() {
		return new AttributeNodeTester<ExternalEntity, String>() {
			@Override
			public boolean doesNodeAlreadyExist() {
				return false;
			}
			@Override
			public String getDefaultValue() {
				return "default_name";
			}
			@Override
			public String getNodeName() {
				return ExternalEntity.NAME;
			}
			@Override
			public String getValue(ExternalEntity form) {
				return form.getName();
			}
			@Override
			public String getValue1() {
				return "employee";
			}
			@Override
			public String getValue2() {
				return "address";
			}
			@Override
			public boolean isNodeDeletedWithNullValue() {
				return false;
			}
			@Override
			public boolean isNullAllowed() {
				return true;
			}
			@Override
			public void setValue(ExternalEntity form, String value) {
				form.setName(value);
			}
			@Override
			public String toString(String value) {
				return value;
			}
		};
	}

	private String buildPrimaryKeyForeignKeyTester() {
		return "primary-key-foreign-key";
	}

	private ChildListNodeTester<ExternalEntity, ExternalPrimaryKeyJoinColumn, String> buildPrimaryKeyJoinColumnTester() {
		return new ChildListNodeTester<ExternalEntity, ExternalPrimaryKeyJoinColumn, String>() {
			@Override
			public ExternalPrimaryKeyJoinColumn addChild(ExternalEntity form, String value) {
				return form.addPrimaryKeyJoinColumn(value);
			}
			@Override
			public ExternalPrimaryKeyJoinColumn getChild(ExternalEntity form, int index) {
				return form.getPrimaryKeyJoinColumn(index);
			}
			@Override
			public ExternalPrimaryKeyJoinColumn getChildForm(List<ExternalPrimaryKeyJoinColumn> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalPrimaryKeyJoinColumn> getChildren(ExternalEntity form) {
				return form.primaryKeyJoinColumns();
			}
			@Override
			public int getChildrenSize(ExternalEntity form) {
				return form.primaryKeyJoinColumnsSize();
			}
			@Override
			public String getChildValue(ExternalPrimaryKeyJoinColumn childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "PRIMARY_KEY_JOIN_COLUMN_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalPrimaryKeyJoinColumn.PRIMARY_KEY_JOIN_COLUMN;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(ExternalEntity form, int index) {
				form.removePrimaryKeyJoinColumn(index);
			}
		};
	}

	private ChildListNodeTester<ExternalEntity, ExternalSecondaryTable, String> buildSecondaryTableTester() {
		return new ChildListNodeTester<ExternalEntity, ExternalSecondaryTable, String>() {
			@Override
			public ExternalSecondaryTable addChild(ExternalEntity form, String value) {
				return form.addSecondaryTable(value, null, null);
			}
			@Override
			public ExternalSecondaryTable getChild(ExternalEntity form, int index) {
				return form.getSecondaryTable(index);
			}
			@Override
			public ExternalSecondaryTable getChildForm(List<ExternalSecondaryTable> childForms, int index) {
				return childForms.get(index);
			}
			@Override
			public List<ExternalSecondaryTable> getChildren(ExternalEntity form) {
				return form.secondaryTables();
			}
			@Override
			public int getChildrenSize(ExternalEntity form) {
				return form.secondaryTablesSize();
			}
			@Override
			public String getChildValue(ExternalSecondaryTable childForm) {
				return childForm.getName();
			}
			@Override
			public String getExpectedChildValue(int index) {
				return "SECONDARY_TABLE_" + index;
			}
			@Override
			public List<String> getExpectedChildValues() {
				List<String> values = new ArrayList<String>();
				for (int index = 0; index < 10; index++) {
					values.add(getExpectedChildValue(index));
				}
				return values;
			}
			@Override
			public String getNodeName() {
				return ExternalSecondaryTable.SECONDARY_TABLE;
			}
			@Override
			public List<String> getNodeNames() {
				return Collections.singletonList(getNodeName());
			}
			@Override
			public void removeChild(ExternalEntity form, int index) {
				form.removeSecondaryTable(index);
			}
		};
	}

	private ChildNodeTester<ExternalEntity, ExternalEntityTable> buildTableTester() {
		return new ChildNodeTester<ExternalEntity, ExternalEntityTable>() {
			@Override
			public ExternalEntityTable addChild(ExternalEntity form) {
				return form.addPrimaryTable(null);
			}
			@Override
			public ExternalEntityTable getChild(ExternalEntity form) {
				return form.getPrimaryTable();
			}
			@Override
			public String getNodeName() {
				return ExternalEntityTable.TABLE;
			}
			@Override
			public boolean hasChild(ExternalEntity form) {
				return form.hasPrimaryTable();
			}
			@Override
			public void removeChild(ExternalEntity form) {
				form.removePrimaryTable();
			}
		};
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	String defaultClassName() {
		return "org.eclipse.persistence.tool.mappings.tests.Entity";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void populate(RootNodeTester tester) {

		tester.setBuilder(buildEntityBuilder());

		// Attributes
		tester.addAttribute(buildNameTester());
		tester.addAttribute(buildClassTester());
		tester.addAttribute(buildParentClassTester());            // EclipseLink
		tester.addAttribute(buildAccessTester());
		tester.addAttribute(buildCacheableTester());              // EclipseLink
		tester.addAttribute(buildMetadataCompleteTester());
		tester.addAttribute(buildReadOnlyTester());               // EclipseLink
		tester.addAttribute(buildExistenceCheckingTester());      // EclipseLink
		tester.addAttribute(buildExcludeDefaultMappingsTester()); // EclipseLink

		// Child nodes
		tester.addTextNode(buildDescriptionTester());
		tester.addNode(buildAccessMethodsTester());                             // EclipseLink
		tester.addNode(buildMultitenantTester());                               // EclipseLink
		tester.addUnsupportedNode(buildAdditionalCriteriaTester());             // EclipseLink
		tester.addChildAttribute(buildCustomizerTester());                      // EclipseLink
		tester.addChildAttribute(buildChangeTrackingTester());                  // EclipseLink

		tester.addNode(buildTableTester());
		tester.addListNodes(buildSecondaryTableTester());
		tester.addUnsupportedNode(buildStructTester());
		tester.addListNodes(buildPrimaryKeyJoinColumnTester());
		tester.addUnsupportedNode(buildPrimaryKeyForeignKeyTester());
		tester.addNode(buildNoSqlTester());
		tester.addUnsupportedNode(buildCascadeOnDeleteTester());
		tester.addUnsupportedNode(buildIndexTester());

		tester.addChildAttribute(buildIdClassTester());
		tester.addNode(buildPrimaryKeyTester());                                // EclipseLink

		tester.addChildAttribute(buildInheritanceTester());

		tester.addNode(buildOptimisticLockingTester());                         // EclipseLink
		tester.addNode(buildCacheTester());                                     // EclipseLink
		tester.addChildAttribute(buildCacheInterceptorTester());                // EclipseLink
		tester.addUnsupportedNode(buildCacheIndexTester());                     // EclipseLink
		tester.addListNodes(buildFetchGroupTester());                           // EclipseLink
		tester.addListNodes(buildConverterTester());                            // EclipseLink
		tester.addListNodes(buildTypeConverterTester());                        // EclipseLink
		tester.addListNodes(buildObjectTypeConverterTester());                  // EclipseLink
		tester.addUnsupportedNode(buildSerializedConverterTester());            // EclipseLink
		tester.addListNodes(buildStructConverterTester());                      // EclipseLink
		tester.addNode(buildCopyPolicyTester());                                // EclipseLink
		tester.addNode(buildInstantiationCopyPolicyTester());                   // EclipseLink
		tester.addNode(buildCloneCopyPolicyTester());                           // EclipseLink
		tester.addUnsupportedNode(buildSerializedObjectTester());               // EclipseLink
		tester.addNode(buildSequenceGeneratorTester());                         // EclipseLink
		tester.addNode(buildTableGeneratorTester());                            // EclipseLink
		tester.addUnsupportedNode(buildUuidGeneratorTester());                  // EclipseLink
		tester.addUnsupportedNode(buildPartitioningTester());                   // EclipseLink
		tester.addUnsupportedNode(buildReplicationPartitioningTester());        // EclipseLink
		tester.addUnsupportedNode(buildRoundRobinPartitioningTester());         // EclipseLink
		tester.addUnsupportedNode(buildPinnedPartitioningTester());             // EclipseLink
		tester.addUnsupportedNode(buildRangePartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildValuePartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildHashPartitioningTester());               // EclipseLink
		tester.addUnsupportedNode(buildUnionPartitioningTester());              // EclipseLink
		tester.addUnsupportedNode(buildPartitionedTester());                    // EclipseLink
		tester.addListNodes(buildNamedQueryTester());                           // EclipseLink
		tester.addListNodes(buildNamedNativeQueryTester());                     // EclipseLink
		tester.addListNodes(buildNamedStoredProcedureQueryTester());            // EclipseLink
		tester.addUnsupportedNode(buildNamedStoredFunctionQueryTester());       // EclipseLink
		tester.addUnsupportedNode(buildNamedPlsqlStoredProcedureQueryTester()); // EclipseLink
		tester.addUnsupportedNode(buildNamedPlsqlStoredFunctionQueryTester());  // EclipseLink
		tester.addUnsupportedNode(buildOracleObjectTester());                   // EclipseLink
		tester.addUnsupportedNode(buildOracleArrayTester());                    // EclipseLink
		tester.addUnsupportedNode(buildPlsqlRecordTester());                    // EclipseLink
		tester.addUnsupportedNode(buildPlsqlTableTester());                     // EclipseLink
		tester.addUnsupportedNode(buildSqlResultSetMappingTester());            // EclipseLink
		tester.addUnsupportedNode(buildQueryRedirectorsTester());               // EclipseLink
		tester.addBooleanNode(buildExcludeDefaultListenersTester());
		tester.addBooleanNode(buildExcludeSuperclassListenersTester());
		tester.addContainerListNodes(buildEntityListenersTester());
		tester.addChildAttribute(buildPrePersistClassTester());
		tester.addChildAttribute(buildPostPersistClassTester());
		tester.addChildAttribute(buildPreRemoveClassTester());
		tester.addChildAttribute(buildPostRemoveClassTester());
		tester.addChildAttribute(buildPreUpdateClassTester());
		tester.addChildAttribute(buildPostUpdateClassTester());
		tester.addChildAttribute(buildPostLoadClassTester());
		tester.addListNodes(buildPropertyTester());                             // EclipseLink
		tester.addListNodes(buildAttributeOverrideTester());                    // EclipseLink
		tester.addListNodes(buildAssociationOverrideTester());                  // EclipseLink
		tester.addContainerListNodes(buildAttributesTester());
	}
}