/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.node;

import java.util.Arrays;
import org.eclipse.persistence.tools.utility.ObjectTools;

/**
 * This class is a straightforward implementation of the {@link Problem} interface.
 */
public class DefaultProblem
	implements Problem
{
	private final Node source;
	private final String messageKey;
	private final int messageType;
	private final Object[] messageArguments;


	DefaultProblem(Node source, String messageKey, int messageType, Object[] messageArguments) {
		super();
		if ((source == null) || (messageKey == null) || (messageArguments == null)) {
			throw new NullPointerException();
		}
		this.source = source;
		this.messageKey = messageKey;
		this.messageType = messageType;
		this.messageArguments = messageArguments;
	}


	// ********** Problem implementation **********

	@Override
	public Node source() {
		return this.source;
	}

	@Override
	public String messageKey() {
		return this.messageKey;
	}

	@Override
	public int messageType() {
		return this.messageType;
	}

	@Override
	public Object[] messageArguments() {
		return this.messageArguments;
	}


	// ********** Object overrides **********

	/**
	 * We implement #equals(Object) because problems are repeatedly
	 * re-calculated and the resulting problems merged with the existing
	 * set of problems; and we want to keep the original problems and
	 * ignore any freshly-generated duplicates.
	 * Also, problems are not saved to disk....
	 */
	@Override
	public boolean equals(Object o) {
		if ( ! (o instanceof Problem)) {
			return false;
		}
		Problem other = (Problem) o;
		return this.source == other.source()
				&& this.messageKey.equals(other.messageKey())
				&& Arrays.equals(this.messageArguments, other.messageArguments());
	}

	@Override
	public int hashCode() {
		return this.source.hashCode() ^ this.messageKey.hashCode();
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this, this.messageKey);
	}
}