/*
 * Copyright (c) 1998, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Oracle - initial API and implementation from Oracle TopLink
package org.eclipse.persistence.internal.jpa.parsing;

import org.eclipse.persistence.queries.ObjectLevelReadQuery;
import org.eclipse.persistence.queries.ReportQuery;

import org.eclipse.persistence.expressions.*;
import org.eclipse.persistence.internal.expressions.*;

/**
 * INTERNAL
 * <p><b>Purpose</b>: Superclass for literals (String, Integer, Float, Character, ...)
 * <p><b>Responsibilities</b>:<ul>
 * <li> Maintain the literal being represented
 * <li> Print to a string
 * <li> Answer if the node is completely built
 * </ul>
 *    @author Jon Driscoll and Joel Lucuik
 *    @since TopLink 4.0
 */
public class LiteralNode extends Node implements AliasableNode {
    public java.lang.Object literal;

    /**
     * Return a new LiteralNode.
     */
    public LiteralNode() {
        super();
    }

    /**
     * INTERNAL
     * Apply this node to the passed query
     */
    @Override
    public void applyToQuery(ObjectLevelReadQuery theQuery, GenerationContext context) {
        if (theQuery.isReportQuery()) {
            ReportQuery reportQuery = (ReportQuery)theQuery;
            reportQuery.addAttribute("CONSTANT", generateExpression(context));
        }

    }

    /**
     * INTERNAL
     * Generate the a new EclipseLink ConstantExpression for this node.
     */
    @Override
    public Expression generateExpression(GenerationContext context) {
        Expression whereClause = new ConstantExpression(getLiteral(), context.getBaseExpression());
        return whereClause;
    }

    /**
     * INTERNAL
     * Return the literal
     */
    @Override
    public String getAsString() {
        return getLiteral().toString();
    }

    /**
     * Insert the method's description here.
     * Creation date: (12/21/00 10:51:48 AM)
     * @return java.lang.Object
     */
    public java.lang.Object getLiteral() {
        return literal;
    }

    /**
     * INTERNAL
     * Is this a literal node
     */
    @Override
    public boolean isLiteralNode() {
        return true;
    }

    /**
     * Insert the method's description here.
     * Creation date: (12/21/00 10:51:48 AM)
     * @param newLiteral java.lang.Object
     */
    public void setLiteral(java.lang.Object newLiteral) {
        literal = newLiteral;
    }

    @Override
    public String toString(int indent) {
        StringBuilder buffer = new StringBuilder();
        toStringIndent(indent, buffer);
        buffer.append(toStringDisplayName() + "[" + getLiteral() + "]");
        return buffer.toString();
    }

    @Override
    public boolean isAliasableNode(){
        return true;
    }
}
