# Commands covered:  resource
#
# This file contains a collection of tests for one or more of the Tcl
# built-in commands.  Sourcing this file into Tcl runs the tests and
# generates output for errors.  No output means no errors were found.
#
# Copyright (c) 1996-1997 Sun Microsystems, Inc.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
# RCS: @(#) $Id: resource.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest
    namespace import -force ::tcltest::*
}

test resource-1.1 {resource tests} {macOnly} {
    list [catch {resource} msg] $msg
} {1 {wrong # args: should be "resource option ?arg ...?"}}
test resource-1.2 {resource tests} {macOnly} {
    list [catch {resource _bad_} msg] $msg
} {1 {bad option "_bad_": must be close, delete, files, list, open, read, types, or write}}

# resource open & close tests
test resource-2.1 {resource open & close tests} {macOnly} {
    list [catch {resource open} msg] $msg
} {1 {wrong # args: should be "resource open fileName ?permissions?"}}
test resource-2.2 {resource open & close tests} {macOnly} {
    list [catch {resource open resource.test r extraArg} msg] $msg
} {1 {wrong # args: should be "resource open fileName ?permissions?"}}
test resource-2.3 {resource open & close tests} {macOnly} {
    list [catch {resource open resource.test bad_perms} msg] $msg
} {1 {illegal access mode "bad_perms"}}
test resource-2.4 {resource open & close tests} {macOnly} {
    list [catch {resource open _bad_file_} msg] $msg
} {1 {file does not exist}}
test resource-2.5 {resource open & close tests} {macOnly} {
    testWriteTextResource -rsrc fileRsrcName -file rsrc.file  {error "don't source me"}
    set id [resource open rsrc.file]
    resource close $id
    file delete rsrc.file
} {}
test resource-2.6 {resource open & close tests} {macOnly} {
    catch {file delete rsrc.file}
    testWriteTextResource -rsrc fileRsrcName -file rsrc.file  {A test string}
    set id [resource open rsrc.file]
    set result [string compare [resource open rsrc.file] $id]
    lappend result [resource read TEXT fileRsrcName $id]
    resource close $id
    file delete rsrc.file
    set result
} {0 {A test string}}
test resource-2.7 {resource open & close tests} {macOnly} {
    catch {file delete rsrc.file}
    testWriteTextResource -rsrc fileRsrcName -file rsrc.file  {error "don't source me"}
    set id [resource open rsrc.file r]
    set result [catch {resource open rsrc.file w} mssg]
    resource close $id
    file delete rsrc.file
    lappend result $mssg
    set result
} {1 {Resource already open with different permissions.}}
test resource-2.8 {resource open & close tests} {macOnly} {
    list [catch {resource close} msg] $msg
} {1 {wrong # args: should be "resource close resourceRef"}}
test resource-2.9 {resource open & close tests} {macOnly} {
    list [catch {resource close foo bar} msg] $msg
} {1 {wrong # args: should be "resource close resourceRef"}}
test resource-2.10 {resource open & close tests} {macOnly} {
    list [catch {resource close _bad_resource_} msg] $msg
} {1 {invalid resource file reference "_bad_resource_"}}
test resource-2.11 {resource open & close tests} {macOnly} {
    set result [catch {resource close System} mssg]
    lappend result $mssg
} {1 {can't close "System" resource file}}
test resource-2.12 {resource open & close tests} {macOnly} {
    set result [catch {resource close application} mssg]
    lappend result $mssg
} {1 {can't close "application" resource file}}

# Tests for listing resources
test resource-3.1 {resource list tests} {macOnly} {
    list [catch {resource list} msg] $msg
} {1 {wrong # args: should be "resource list resourceType ?resourceRef?"}}
test resource-3.2 {resource list tests} {macOnly} {
    list [catch {resource list _bad_type_} msg] $msg
} {1 {expected Macintosh OS type but got "_bad_type_"}}
test resource-3.3 {resource list tests} {macOnly} {
    list [catch {resource list TEXT _bad_ref_} msg] $msg
} {1 {invalid resource file reference "_bad_ref_"}}
test resource-3.4 {resource list tests} {macOnly} {
    list [catch {resource list TEXT _bad_ref_ extraArg} msg] $msg
} {1 {wrong # args: should be "resource list resourceType ?resourceRef?"}}
test resource-3.5 {resource list tests} {macOnly} {
    catch {file delete rsrc.file}
    testWriteTextResource -rsrc fileRsrcName -file rsrc.file  {error "don't source me"}
    set id [resource open rsrc.file]
    catch "resource list TEXT $id" result
    resource close $id
    set result
} {fileRsrcName}
test resource-3.6 {resource list tests} {macOnly} {
    # There should not be any resource of this type
    resource list XXXX
} {}
test resource-3.7 {resource list tests} {macOnly} {
    set resourceList [resource list STR#]
    if {[lsearch $resourceList {Tcl Environment Variables}] == -1} {
        set result {couldn't find resource that should exist}
    } else {
        set result ok
    }
} {ok}

# Tests for reading resources
test resource-4.1 {resource read tests} {macOnly} {
    list [catch {resource read} msg] $msg
} {1 {wrong # args: should be "resource read resourceType resourceId ?resourceRef?"}}
test resource-4.2 {resource read tests} {macOnly} {
    list [catch {resource read TEXT} msg] $msg
} {1 {wrong # args: should be "resource read resourceType resourceId ?resourceRef?"}}
test resource-4.3 {resource read tests} {macOnly} {
    list [catch {resource read STR# {_non_existant_resource_}} msg] $msg
} {1 {could not load resource}}
test resource-4.4 {resource read tests} {macOnly} {
    # The following resource should exist and load OK without error
    catch {resource read STR# {Tcl Environment Variables}}
} {0}

# Tests for getting resource types
test resource-5.1 {resource types tests} {macOnly} {
    list [catch {resource types _bad_ref_} msg] $msg
} {1 {invalid resource file reference "_bad_ref_"}}
test resource-5.2 {resource types tests} {macOnly} {
    list [catch {resource types _bad_ref_ extraArg} msg] $msg
} {1 {wrong # args: should be "resource types ?resourceRef?"}}
test resource-5.3 {resource types tests} {macOnly} {
    # This should never cause an error
    catch {resource types}
} {0}
test resource-5.4 {resource types tests} {macOnly} {
    testWriteTextResource -rsrc fileRsrcName -file rsrc.file  {error "don't source me"}
    set id [resource open rsrc.file]
    set result [resource types $id]
    resource close $id
    set result
} {TEXT}

# resource write tests
test resource-6.1 {resource write tests} {macOnly} {
    list [catch {resource write} msg] $msg
} {1 {wrong # args: should be "resource write ?-id resourceId? ?-name resourceName? ?-file resourceRef? ?-force? resourceType data"}}
test resource-6.2 {resource write tests} {macOnly} {
    list [catch {resource write _bad_type_ data} msg] $msg
} {1 {expected Macintosh OS type but got "_bad_type_"}}
test resource-6.3 {resource write tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    resource close $id
    set id [resource open rsrc2.file r]
    set result [catch {resource write -file $id -name Hello TEXT foo} errMsg]
    lappend result [string compare $errMsg "cannot write to resource file \"$id\", it was opened read only"]
    lappend result [lsearch [resource list TEXT $id] Hello]
    resource close $id
    file delete rsrc2.file
    set result   
} {1 0 -1}
test resource-6.4 {resource write tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    resource write -file $id -name Hello TEXT {set x "our test data"}
    source -rsrc Hello rsrc2.file
    resource close $id
    file delete rsrc2.file
    set x
} {our test data}
test resource-6.5 {resource write tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    resource write -file $id -id 256 TEXT {HAHAHAHAHAHAHA}
    set result [catch {resource write -file $id -id 256 TEXT {HOHOHOHOHOHO}} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {the resource 256 already exists, use "-force" to overwrite it.}}
test resource-6.6 {resource write tests} {macOnly} {
    catch {file delete rsrc2.file}
    testWriteTextResource -rsrc fileRsrcName -rsrcid 256  -file rsrc2.file  -protected {error "don't tread on me"}
    set id [resource open rsrc2.file w]
    set result [catch {resource write -id 256 -force -file $id TEXT {NAHNAHNANAHNAH}} mssg] 
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {could not write resource id 256 of type TEXT, it was protected.}}
test resource-6.7 {resource write tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    resource write -file $id -id 256 -name FOO TEXT {set x [list "our first test data"]}
    resource write -file $id -id 256 -name BAR -force TEXT {set x [list "our second test data"]}
    source -rsrcid 256 rsrc2.file
    lappend x [resource list TEXT $id]
    resource close $id
    file delete rsrc2.file
    set x
} {{our second test data} BAR}

#Tests for listing open resource files
test resource-7.1 {resource file tests} {macOnly} {
    catch {resource files foo bar} mssg
    set mssg
} {wrong # args: should be "resource files ?resourceId?"}
test resource-7.2 {resource file tests} {macOnly} {
    catch {file delete rsrc2.file}
    set rsrcFiles [resource files]
    set id [resource open rsrc2.file w]
    set result [string compare $rsrcFiles [lrange [resource files] 1 end]]
    lappend result [string compare $id [lrange [resource files] 0 0]]
    resource close $id
    file delete rsrc2.file
    set result
} {0 0}
test resource-7.3 {resource file tests} {macOnly} {
    set result 0
    foreach file [resource files] {
        if {[catch {resource types $file}] != 0} {
            set result 1
        }
    }
    set result
} {0}
test resource-7.4 {resource file tests} {macOnly} {
    catch {resource files __NO_SUCH_RESOURCE__} mssg
    set mssg
} {invalid resource file reference "__NO_SUCH_RESOURCE__"}
test resource-7.5 {resource file tests} {macOnly} {
    set sys [resource files System]
    string compare $sys [file join $env(SYS_FOLDER) System]
} {0}
test resource-7.6 {resource file tests} {macOnly} {
    set app [resource files application]
    string compare $app [info nameofexecutable]
} {0}

#Tests for the resource delete command
test resource-8.1 {resource delete tests} {macOnly} {
    list [catch {resource delete} msg] $msg
} {1 {wrong # args: should be "resource delete ?-id resourceId? ?-name resourceName? ?-file resourceRef? resourceType"}}
test resource-8.2 {resource delete tests} {macOnly} {
    list [catch {resource delete TEXT} msg] $msg
} {1 {you must specify either "-id" or "-name" or both to "resource delete"}}
test resource-8.3 {resource delete tests} {macOnly} {
    set result [catch {resource delete -file ffffff -id 128 TEXT} mssg]
    lappend result $mssg    
} {1 {invalid resource file reference "ffffff"}}    
test resource-8.4 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    testWriteTextResource -rsrc fileRsrcName -rsrcid 128 -file rsrc2.file {Some stuff}
    set id [resource open rsrc2.file r]
    set result [catch {resource delete -id 128 -file $id TEXT} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result [string compare $mssg "cannot delete from resource file \"$id\", it was opened read only"]   
} {1 0}
test resource-8.5 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    testWriteTextResource -rsrc fileRsrcName -rsrcid 128 -file rsrc2.file {Some stuff}
    set id [resource open rsrc2.file w]
    set result [catch {resource delete -id 128 -file $id _bad_type_} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {expected Macintosh OS type but got "_bad_type_"}}
test resource-8.5.1 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    set result [catch {resource delete -id 128 -file $id TEXT} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {resource not found}}
test resource-8.6 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    set result [catch {resource delete -name foo -file $id TEXT} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {resource not found}}
test resource-8.7 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    set id [resource open rsrc2.file w]
    resource write -file $id -name foo -id 128 TEXT {some stuff}
    resource write -file $id -name bar -id 129 TEXT {some stuff}
    set result [catch {resource delete -name foo -id 129 -file $id TEXT} mssg]
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {"-id" and "-name" values do not point to the same resource}}
test resource-8.8 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    testWriteTextResource -rsrc fileRsrcName -rsrcid 256  -file rsrc2.file  -protected {error "don't tread on me"}
    set id [resource open rsrc2.file w]
    set result [catch {resource delete -id 256 -file $id TEXT } mssg] 
    resource close $id
    file delete rsrc2.file
    lappend result $mssg
} {1 {resource cannot be deleted: it is protected.}}
test resource-8.9 {resource delete tests} {macOnly} {
    catch {file delete rsrc2.file}
    testWriteTextResource -rsrc fileRsrcName -rsrcid 128 -file rsrc2.file {Some stuff}
    set id [resource open rsrc2.file w]
    set result [resource list TEXT $id]
    resource delete -id 128 -file $id TEXT
    lappend result [resource list TEXT $id]
    resource close $id
    file delete rsrc2.file
    set result
} {fileRsrcName {}}
    
# Tests for the Mac version of the source command
catch {file delete rsrc.file}
test resource-9.1 {source command} {macOnly} {
    testWriteTextResource -rsrc fileRsrcName -rsrcid 128 \
	    -file rsrc.file  {set rsrc_foo 1}
    catch {unset rsrc_foo}
    source -rsrc fileRsrcName rsrc.file
    list [catch {set rsrc_foo} msg] $msg
} {0 1}
test resource-9.2 {source command} {macOnly} {
    catch {unset rsrc_foo}
    list [catch {source -rsrc no_resource rsrc.file} msg] $msg
} {1 {The resource "no_resource" could not be loaded from rsrc.file.}}
test resource-9.3 {source command} {macOnly} {
    catch {unset rsrc_foo}
    source -rsrcid 128 rsrc.file
    list [catch {set rsrc_foo} msg] $msg
} {0 1}
test resource-9.4 {source command} {macOnly} {
    catch {unset rsrc_foo}
    list [catch {source -rsrcid bad_int rsrc.file} msg] $msg
} {1 {expected integer but got "bad_int"}}
test resource-9.5 {source command} {macOnly} {
    catch {unset rsrc_foo}
    list [catch {source -rsrcid 100 rsrc.file} msg] $msg
} {1 {The resource "ID=100" could not be loaded from rsrc.file.}}

# cleanup
catch {file delete rsrc.file}
::tcltest::cleanupTests
return












