/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.servlets;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class CrossOriginFilter
implements Filter {
    private static final Logger LOG = Log.getLogger(CrossOriginFilter.class);
    private static final String ORIGIN_HEADER = "Origin";
    public static final String ACCESS_CONTROL_REQUEST_METHOD_HEADER = "Access-Control-Request-Method";
    public static final String ACCESS_CONTROL_REQUEST_HEADERS_HEADER = "Access-Control-Request-Headers";
    public static final String ACCESS_CONTROL_ALLOW_ORIGIN_HEADER = "Access-Control-Allow-Origin";
    public static final String ACCESS_CONTROL_ALLOW_METHODS_HEADER = "Access-Control-Allow-Methods";
    public static final String ACCESS_CONTROL_ALLOW_HEADERS_HEADER = "Access-Control-Allow-Headers";
    public static final String ACCESS_CONTROL_MAX_AGE_HEADER = "Access-Control-Max-Age";
    public static final String ACCESS_CONTROL_ALLOW_CREDENTIALS_HEADER = "Access-Control-Allow-Credentials";
    public static final String ALLOWED_ORIGINS_PARAM = "allowedOrigins";
    public static final String ALLOWED_METHODS_PARAM = "allowedMethods";
    public static final String ALLOWED_HEADERS_PARAM = "allowedHeaders";
    public static final String PREFLIGHT_MAX_AGE_PARAM = "preflightMaxAge";
    public static final String ALLOW_CREDENTIALS_PARAM = "allowCredentials";
    private static final String ANY_ORIGIN = "*";
    private static final List<String> SIMPLE_HTTP_METHODS = Arrays.asList("GET", "POST", "HEAD");
    private boolean anyOriginAllowed;
    private List<String> allowedOrigins = new ArrayList<String>();
    private List<String> allowedMethods = new ArrayList<String>();
    private List<String> allowedHeaders = new ArrayList<String>();
    private int preflightMaxAge = 0;
    private boolean allowCredentials;

    public void init(FilterConfig config) throws ServletException {
        String allowedMethodsConfig;
        String[] allowedOrigins;
        String allowedOriginsConfig = config.getInitParameter(ALLOWED_ORIGINS_PARAM);
        if (allowedOriginsConfig == null) {
            allowedOriginsConfig = ANY_ORIGIN;
        }
        for (String allowedOrigin : allowedOrigins = allowedOriginsConfig.split(",")) {
            if ((allowedOrigin = allowedOrigin.trim()).length() <= 0) continue;
            if (ANY_ORIGIN.equals(allowedOrigin)) {
                this.anyOriginAllowed = true;
                this.allowedOrigins.clear();
                break;
            }
            this.allowedOrigins.add(allowedOrigin);
        }
        if ((allowedMethodsConfig = config.getInitParameter(ALLOWED_METHODS_PARAM)) == null) {
            allowedMethodsConfig = "GET,POST,HEAD";
        }
        this.allowedMethods.addAll(Arrays.asList(allowedMethodsConfig.split(",")));
        String allowedHeadersConfig = config.getInitParameter(ALLOWED_HEADERS_PARAM);
        if (allowedHeadersConfig == null) {
            allowedHeadersConfig = "X-Requested-With,Content-Type,Accept,Origin";
        }
        this.allowedHeaders.addAll(Arrays.asList(allowedHeadersConfig.split(",")));
        String preflightMaxAgeConfig = config.getInitParameter(PREFLIGHT_MAX_AGE_PARAM);
        if (preflightMaxAgeConfig == null) {
            preflightMaxAgeConfig = "1800";
        }
        try {
            this.preflightMaxAge = Integer.parseInt(preflightMaxAgeConfig);
        }
        catch (NumberFormatException x) {
            LOG.info("Cross-origin filter, could not parse '{}' parameter as integer: {}", new Object[]{PREFLIGHT_MAX_AGE_PARAM, preflightMaxAgeConfig});
        }
        String allowedCredentialsConfig = config.getInitParameter(ALLOW_CREDENTIALS_PARAM);
        if (allowedCredentialsConfig == null) {
            allowedCredentialsConfig = "true";
        }
        this.allowCredentials = Boolean.parseBoolean(allowedCredentialsConfig);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Cross-origin filter configuration: allowedOrigins = " + allowedOriginsConfig + ", " + ALLOWED_METHODS_PARAM + " = " + allowedMethodsConfig + ", " + ALLOWED_HEADERS_PARAM + " = " + allowedHeadersConfig + ", " + PREFLIGHT_MAX_AGE_PARAM + " = " + preflightMaxAgeConfig + ", " + ALLOW_CREDENTIALS_PARAM + " = " + allowedCredentialsConfig, new Object[0]);
        }
    }

    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        this.handle((HttpServletRequest)request, (HttpServletResponse)response, chain);
    }

    private void handle(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws IOException, ServletException {
        String origin = request.getHeader(ORIGIN_HEADER);
        if (origin != null && this.isEnabled(request)) {
            if (this.originMatches(origin)) {
                if (this.isSimpleRequest(request)) {
                    LOG.debug("Cross-origin request to {} is a simple cross-origin request", new Object[]{request.getRequestURI()});
                    this.handleSimpleResponse(request, response, origin);
                } else if (this.isPreflightRequest(request)) {
                    LOG.debug("Cross-origin request to {} is a preflight cross-origin request", new Object[]{request.getRequestURI()});
                    this.handlePreflightResponse(request, response, origin);
                } else {
                    LOG.debug("Cross-origin request to {} is a non-simple cross-origin request", new Object[]{request.getRequestURI()});
                    this.handleSimpleResponse(request, response, origin);
                }
            } else {
                LOG.debug("Cross-origin request to " + request.getRequestURI() + " with origin " + origin + " does not match allowed origins " + this.allowedOrigins, new Object[0]);
            }
        }
        chain.doFilter((ServletRequest)request, (ServletResponse)response);
    }

    protected boolean isEnabled(HttpServletRequest request) {
        Enumeration connections = request.getHeaders("Connection");
        while (connections.hasMoreElements()) {
            String connection = (String)connections.nextElement();
            if (!"Upgrade".equalsIgnoreCase(connection)) continue;
            Enumeration upgrades = request.getHeaders("Upgrade");
            while (upgrades.hasMoreElements()) {
                String upgrade = (String)upgrades.nextElement();
                if (!"WebSocket".equalsIgnoreCase(upgrade)) continue;
                return false;
            }
        }
        return true;
    }

    private boolean originMatches(String originList) {
        String[] origins;
        if (this.anyOriginAllowed) {
            return true;
        }
        if (originList.trim().length() == 0) {
            return false;
        }
        for (String origin : origins = originList.split(" ")) {
            if (origin.trim().length() == 0) continue;
            for (String allowedOrigin : this.allowedOrigins) {
                Matcher matcher;
                if (!(allowedOrigin.contains(ANY_ORIGIN) ? (matcher = this.createMatcher(origin, allowedOrigin)).matches() : allowedOrigin.equals(origin))) continue;
                return true;
            }
        }
        return false;
    }

    private Matcher createMatcher(String origin, String allowedOrigin) {
        String regex = this.parseAllowedWildcardOriginToRegex(allowedOrigin);
        Pattern pattern = Pattern.compile(regex);
        return pattern.matcher(origin);
    }

    private String parseAllowedWildcardOriginToRegex(String allowedOrigin) {
        String regex = allowedOrigin.replace(".", "\\.");
        return regex.replace(ANY_ORIGIN, ".*");
    }

    private boolean isSimpleRequest(HttpServletRequest request) {
        String method = request.getMethod();
        if (SIMPLE_HTTP_METHODS.contains(method)) {
            return request.getHeader(ACCESS_CONTROL_REQUEST_METHOD_HEADER) == null;
        }
        return false;
    }

    private boolean isPreflightRequest(HttpServletRequest request) {
        String method = request.getMethod();
        if (!"OPTIONS".equalsIgnoreCase(method)) {
            return false;
        }
        return request.getHeader(ACCESS_CONTROL_REQUEST_METHOD_HEADER) != null;
    }

    private void handleSimpleResponse(HttpServletRequest request, HttpServletResponse response, String origin) {
        response.setHeader(ACCESS_CONTROL_ALLOW_ORIGIN_HEADER, origin);
        if (this.allowCredentials) {
            response.setHeader(ACCESS_CONTROL_ALLOW_CREDENTIALS_HEADER, "true");
        }
    }

    private void handlePreflightResponse(HttpServletRequest request, HttpServletResponse response, String origin) {
        boolean methodAllowed = this.isMethodAllowed(request);
        if (!methodAllowed) {
            return;
        }
        boolean headersAllowed = this.areHeadersAllowed(request);
        if (!headersAllowed) {
            return;
        }
        response.setHeader(ACCESS_CONTROL_ALLOW_ORIGIN_HEADER, origin);
        if (this.allowCredentials) {
            response.setHeader(ACCESS_CONTROL_ALLOW_CREDENTIALS_HEADER, "true");
        }
        if (this.preflightMaxAge > 0) {
            response.setHeader(ACCESS_CONTROL_MAX_AGE_HEADER, String.valueOf(this.preflightMaxAge));
        }
        response.setHeader(ACCESS_CONTROL_ALLOW_METHODS_HEADER, this.commify(this.allowedMethods));
        response.setHeader(ACCESS_CONTROL_ALLOW_HEADERS_HEADER, this.commify(this.allowedHeaders));
    }

    private boolean isMethodAllowed(HttpServletRequest request) {
        String accessControlRequestMethod = request.getHeader(ACCESS_CONTROL_REQUEST_METHOD_HEADER);
        LOG.debug("{} is {}", new Object[]{ACCESS_CONTROL_REQUEST_METHOD_HEADER, accessControlRequestMethod});
        boolean result = false;
        if (accessControlRequestMethod != null) {
            result = this.allowedMethods.contains(accessControlRequestMethod);
        }
        LOG.debug("Method {} is" + (result ? "" : " not") + " among allowed methods {}", new Object[]{accessControlRequestMethod, this.allowedMethods});
        return result;
    }

    private boolean areHeadersAllowed(HttpServletRequest request) {
        String accessControlRequestHeaders = request.getHeader(ACCESS_CONTROL_REQUEST_HEADERS_HEADER);
        LOG.debug("{} is {}", new Object[]{ACCESS_CONTROL_REQUEST_HEADERS_HEADER, accessControlRequestHeaders});
        boolean result = true;
        if (accessControlRequestHeaders != null) {
            String[] headers;
            for (String header : headers = accessControlRequestHeaders.split(",")) {
                boolean headerAllowed = false;
                for (String allowedHeader : this.allowedHeaders) {
                    if (!header.trim().equalsIgnoreCase(allowedHeader.trim())) continue;
                    headerAllowed = true;
                    break;
                }
                if (headerAllowed) continue;
                result = false;
                break;
            }
        }
        LOG.debug("Headers [{}] are" + (result ? "" : " not") + " among allowed headers {}", new Object[]{accessControlRequestHeaders, this.allowedHeaders});
        return result;
    }

    private String commify(List<String> strings) {
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < strings.size(); ++i) {
            if (i > 0) {
                builder.append(",");
            }
            String string = strings.get(i);
            builder.append(string);
        }
        return builder.toString();
    }

    public void destroy() {
        this.anyOriginAllowed = false;
        this.allowedOrigins.clear();
        this.allowedMethods.clear();
        this.allowedHeaders.clear();
        this.preflightMaxAge = 0;
        this.allowCredentials = false;
    }
}

