/* AppDir 1.1
 *
 * Namespace: ADIR_ / adir_
 */

/* Copyright (c) 2009, The Static Void Project
 *
 * Permission to use, copy, modify, and/or distribute this software for 
 * any purpose with or without fee is hereby granted, provided that the 
 * above copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* AppDir is a simple file system abstraction layer which can be used to
 * avoid portability issues.
 *
 * Warning:
 *
 * The current code does not do much error checking and does not use
 * dynamic memory allocation to keep things fast and simple. All functions
 * share the same path buffer!
 *
 * Portability:
 *
 * AppDir supports Windows, Mac OS X, and Linux.
 */



#include "appdir.h"


/* Detect platform and set appropriate values
 */
#if defined(_WIN32)
#       define ADIR_WINDOWS
#	define ADIR_DELIMITER	'\\'
#elif defined(__APPLE__)
#       define ADIR_MAC_OS_X
#	define ADIR_DELIMITER	'/'
#elif defined(__linux__)
#       define ADIR_LINUX
#	define ADIR_DELIMITER	'/'
#else
#       error Unsupported platform
#endif



/*
 *
 * Cross-platform code
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>


/* Boolean type */
typedef unsigned char	ADIR_BOOL;
enum {ADIR_FALSE, ADIR_TRUE};


/* Maximal path size */
#define ADIR_PATH_SIZE	256


/* Path storage */
static char             Path[ADIR_PATH_SIZE];

/* Pointer to the end of the stored path */
static char *		PathEnd;


/* User data directory path */
static char		UserData[ADIR_PATH_SIZE];

/* App data directory path */
static char		AppData[ADIR_PATH_SIZE];



/* Returns a file path
 */
char * adir_file_path(ADIR_DIRECTORY directory, const char * a_directory,
        const char * a_file
)
{
        char abstract_path[ADIR_PATH_SIZE];
        char *p = abstract_path;

        while (*a_directory != '\0') *p++ = *a_directory++;
        *p++ = ADIR_DELIMITER;
        while (*a_file != '\0') *p++ = *a_file++;
        *p = '\0';

        return adir_path(directory, abstract_path);
}




/*
 *
 * Windows code
 *
 */
#ifdef ADIR_WINDOWS

#define WIN32_LEAN_AND_MEAN
#define STRICT
#include <windows.h>
#include <shlobj.h>

/* Data for adir_files() / adir_file()
 */
static HANDLE           Handle;
static WIN32_FIND_DATA  FindData;
static ADIR_BOOL	FindDataValid;




/* Initialization
 */
ADIR_RESULT adir_init(const char * app_name)
{
        char * p = UserData;

        if (app_name == NULL) return ADIR_ERR;

        /* Determine user data directory */
        if (SHGetFolderPath(NULL, CSIDL_PERSONAL | CSIDL_FLAG_CREATE, NULL,
                0, UserData) != S_OK) return ADIR_ERR;

        while (*p != '\0') ++p; *p++ = ADIR_DELIMITER;
        while (*app_name != '\0') *p++ = *app_name++;
        *p++ = ADIR_DELIMITER;  *p = '\0';

        /* Create user data directory if necessary */
        CreateDirectory(UserData, NULL);

        /* Determine app data directory */
        AppData[0] = '.'; AppData[1] = ADIR_DELIMITER; AppData[2] = '\0';

        /* Initialize internal data */
        Handle = INVALID_HANDLE_VALUE;

        return ADIR_OK;
}



/* Shut down
 */
void adir_quit(void)
{
        /* Close handle if necessary */
        if (Handle != INVALID_HANDLE_VALUE) FindClose(Handle);

}



/* Returns a path (file or directory)
 */
char * adir_path(ADIR_DIRECTORY directory, const char * abstract_path)
{
        char * concrete_path = Path;
        char * p;

        if (directory == ADIR_APP_DATA) {
                p = AppData;
        } else if (directory == ADIR_USER_DATA) {
                p = UserData;
        } else {
                return NULL;
        }

        while (*p != '\0') *concrete_path++ = *p++;

        if (abstract_path == ADIR_ROOT_DIRECTORY) goto done;

        while (*abstract_path != '\0') {

                if (*abstract_path == '/') {
                        *concrete_path++ = ADIR_DELIMITER;
                } else {
                        *concrete_path++ = *abstract_path;
                }

                ++abstract_path;
        }

done:
        *concrete_path = '\0'; PathEnd = concrete_path; return Path;
}



/* Used together with adir_file() to iterate over all files in a directory
 */
void adir_files(ADIR_DIRECTORY directory, const char *abstract_path)
{
        char * concrete_path = adir_path(directory, abstract_path);

        /* Attach '*' to concrete path */
        *PathEnd++ = '*'; *PathEnd = '\0';

        /* Close old handle if necessary */
        if (Handle != INVALID_HANDLE_VALUE) FindClose(Handle);

        /* Open new handle */
        Handle = FindFirstFile(concrete_path, &FindData);
        FindDataValid = ADIR_TRUE;

        /* Remove '*' from concrete path */
        *(--PathEnd) = '\0';
}



/* Used together with adir_files() to iterate over all files in a directory
 */
char * adir_file(void)
{
        char *file = NULL;

        if (!FindDataValid) return NULL;

        for (;;) {

                /* Ignore '.' and '..' entries and directories */
                while (FindData.cFileName[0] == '.' ||
                        FindData.dwFileAttributes &
                        FILE_ATTRIBUTE_DIRECTORY) {

                        if (!FindNextFile(Handle, &FindData))
                                goto done;
                }

                {
                        const char *s = FindData.cFileName; char *d = Path;
                        while (*s != '\0') *d++ = *s++; *d = '\0';
                }

                file = Path;
                if (!FindNextFile(Handle, &FindData)) goto done;
                return file;
        }

done:
        FindDataValid = ADIR_FALSE;
        FindClose(Handle);
        Handle = INVALID_HANDLE_VALUE;
        return file;
}



/* Used together with adir_files() to iterate over all files in a directory
 */
char * adir_file_with_path(void)
{
        char *file = NULL;

        if (!FindDataValid) return NULL;

        for (;;) {

                /* Ignore '.' and '..' entries and directories */
                while (FindData.cFileName[0] == '.' ||
                        FindData.dwFileAttributes &
                        FILE_ATTRIBUTE_DIRECTORY) {

                        if (!FindNextFile(Handle, &FindData))
                                goto done;
                }

                {
                        const char *s = FindData.cFileName; char *d = PathEnd;
                        while (*s != '\0') *d++ = *s++; *d = '\0';
                }
                file = Path;
                if (!FindNextFile(Handle, &FindData)) goto done;
                return file;
        }

done:
        FindDataValid = ADIR_FALSE;
        FindClose(Handle);
        Handle = INVALID_HANDLE_VALUE;
        return file;
}




#endif











/*
 *
 * Common UNIX code
 *
 */
#if defined(ADIR_MAC_OS_X) || defined(ADIR_LINUX)

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>


/* Data for adir_files() / adir_file()
 */
static DIR *            DirectoryStream;
static ADIR_BOOL	DirectoryStreamValid;



/* Shut down
 */
void adir_quit(void)
{
        /* Close directory stream if necessary */
        if (DirectoryStreamValid) closedir(DirectoryStream);
}



/* Returns a path
 */
char * adir_path(ADIR_DIRECTORY directory, const char * abstract_path)
{
        char * concrete_path = Path;
        char * p;

        if (directory == ADIR_APP_DATA) {
                p = AppData;
        } else if (directory == ADIR_USER_DATA) {
                p = UserData;
        } else {
                return NULL;
        }

        while (*p != '\0') *concrete_path++ = *p++;

        if (abstract_path == ADIR_ROOT_DIRECTORY) goto done;

        while (*abstract_path != '\0') *concrete_path++ = *abstract_path++;

done:
        *concrete_path = '\0'; PathEnd = concrete_path; return Path;
}



/* Used together with adir_file() to iterate over all files in a directory
 */
void adir_files(ADIR_DIRECTORY directory, const char *abstract_path)
{
        char * concrete_path = adir_path(directory, abstract_path);

        /* Close old directory stream if necessary */
        if (DirectoryStreamValid) closedir(DirectoryStream);

        /* Open new directory stream */
        DirectoryStream = opendir(concrete_path);
        DirectoryStreamValid = ADIR_TRUE;
}



/* Used together with adir_files() to iterate over all files in a directory
 */
char * adir_file(void)
{
        struct dirent * entry;

        if (!DirectoryStreamValid) return NULL;

        while ((entry = readdir(DirectoryStream)) != NULL) {

                /* Ignore '.' and '..' entries and directories */
                if (entry->d_name[0] == '.' || entry->d_type == DT_DIR) {
                        continue;
                }

                {
                        const char *s = entry->d_name; char *d = Path;
                        while (*s != '\0') *d++ = *s++; *d = '\0';
                }

                return Path;
        }

        closedir(DirectoryStream); DirectoryStreamValid = ADIR_FALSE;
        return NULL;
}



/* Used together with adir_files() to iterate over all files in a directory
 */
char * adir_file_with_path(void)
{
        struct dirent * entry;

        if (!DirectoryStreamValid) return NULL;

        while ((entry = readdir(DirectoryStream)) != NULL) {

                /* Ignore '.' and '..' entries and directories */
                if (entry->d_name[0] == '.' || entry->d_type == DT_DIR) {
                        continue;
                }

                {
                        const char *s = entry->d_name; char *d = PathEnd;
                        while (*s != '\0') *d++ = *s++; *d = '\0';
                }
                return Path;
        }

        closedir(DirectoryStream); DirectoryStreamValid = ADIR_FALSE;
        return NULL;
}


#endif









#ifdef ADIR_MAC_OS_X

/*
 *
 * Mac OS X code
 *
 */
#include <CoreServices/CoreServices.h>



/* Initialization
 */
ADIR_RESULT adir_init(const char * app_name)
{
        FSRef fsref;
        CFBundleRef bundleref;
        struct stat s;

        if (app_name == NULL) return ADIR_ERR;

        /* Determine app resources directory
         */
        if ((bundleref = CFBundleGetMainBundle()) != NULL)  {
                CFURLRef url = CFBundleCopyResourcesDirectoryURL(bundleref);

                if (url) {
                        if (CFURLGetFSRef(url, &fsref)) {
                                char * p = AppData;

                                FSRefMakePath(&fsref, (UInt8 *)AppData,
                                        ADIR_PATH_SIZE
                                );

                                while (*p != '\0') ++p;
                                *p++ = ADIR_DELIMITER; *p = '\0';
                        }

                        CFRelease(url);
                }
        }

        /* Determine app support directory, create if necessary
         */
        if (!FSFindFolder(kUserDomain, kApplicationSupportFolderType,
                kDontCreateFolder, &fsref)) {

                if (!FSRefMakePath(&fsref, (UInt8 *)UserData,
                        ADIR_PATH_SIZE)) {
                        char * p = UserData;

                        while (*p != '\0') ++p;
                        *p++ = ADIR_DELIMITER;
                        while (*app_name != '\0') *p++ = *app_name++;
                        *p++ = ADIR_DELIMITER; *p = '\0';

                        if (stat(UserData, &s)) mkdir(UserData, 0700);
                }
        }

        /* Initialize internal data */
        DirectoryStreamValid = ADIR_FALSE;

        return ADIR_OK;
}

#endif





#ifdef ADIR_LINUX

/*
 *
 * Linux code
 *
 */


/* Initialization
 */
ADIR_RESULT adir_init(const char * app_name)
{
        char * p;
        struct stat sb;

        if (app_name == NULL) return ADIR_ERR;

        /* Determine user data directory */
        if ((p = getenv("HOME")) == NULL) return ADIR_ERR;
        strcpy(UserData, p);
        p = UserData;
        while (*p != '\0') ++p; *p++ = ADIR_DELIMITER; *p++ = '.';
        while (*app_name != '\0') *p++ = *app_name++;
        *p++ = ADIR_DELIMITER;  *p = '\0';

        /* Create user data directory if necessary */
        if (stat(UserData, &sb) != 0) {
                mkdir(UserData, S_IRUSR | S_IWUSR | S_IXUSR);
        }

        /* Determine app data directory */
        AppData[0] = '.'; AppData[1] = ADIR_DELIMITER; AppData[2] = '\0';

        /* Initialize internal data */
        DirectoryStreamValid = ADIR_FALSE;

        return ADIR_OK;
}



#endif


