// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// This file contains commonly used definitions of video capture.

#ifndef THIRD_PARTY_BLINK_PUBLIC_COMMON_MEDIA_VIDEO_CAPTURE_H_
#define THIRD_PARTY_BLINK_PUBLIC_COMMON_MEDIA_VIDEO_CAPTURE_H_

#include "base/functional/callback.h"
#include "base/time/time.h"
#include "media/capture/video/video_capture_feedback.h"
#include "media/capture/video_capture_types.h"

namespace media {
class VideoFrame;
}  // namespace media

namespace blink {

// This callback is used to deliver video frames.
//
// |estimated_capture_time| - The capture time of the delivered video
// frame. This field represents the local time at which either: 1) the frame
// was generated, if it was done so locally; or 2) the targeted play-out time
// of the frame, if it was generated from a remote source. Either way, an
// implementation should not present the frame before this point-in-time. This
// value is NOT a high-resolution timestamp, and so it should not be used as a
// presentation time; but, instead, it should be used for buffering playback
// and for A/V synchronization purposes. NOTE: It is possible for this value
// to be null if the current implementation lacks this timing information.
//
// |video_frame->timestamp()| gives the presentation timestamp of the video
// frame relative to the first frame generated by the corresponding source.
// Because a source can start generating frames before a subscriber is added,
// the first video frame delivered may not have timestamp equal to 0.
//
// |scaled_video_frames| are alternative representations of |video_frame|
// where scaling has already been applied. Scaled frames may be produced by
// some capturers that listen to media::VideoCaptureFeedback. However, because
// not all capturers support in-capturer scaling or due to delays in the
// pipeline, listeners have to be prepared for scaled video frames not being
// present or being present with other sizes than most recently requested.
using VideoCaptureDeliverFrameCB = base::RepeatingCallback<void(
    scoped_refptr<media::VideoFrame> video_frame,
    std::vector<scoped_refptr<media::VideoFrame>> scaled_video_frames,
    base::TimeTicks estimated_capture_time)>;

// Callback for delivering dropped frame notifications.
using VideoCaptureNotifyFrameDroppedCB = base::RepeatingClosure;

// Callback for informing when new crop-versions are applied.
using VideoCaptureCropVersionCB =
    base::RepeatingCallback<void(uint32_t crop_version)>;

using VideoCaptureDeviceFormatsCB =
    base::OnceCallback<void(const media::VideoCaptureFormats&)>;

using VideoCaptureFeedbackCB = media::VideoCaptureFeedbackCB;

// Current status of the video capture device. It's used by multiple classes in
// browser process and renderer process. Browser process sends information about
// the current capture state and error to the renderer process using this type.
enum VideoCaptureState {
  VIDEO_CAPTURE_STATE_STARTING,
  VIDEO_CAPTURE_STATE_STARTED,
  VIDEO_CAPTURE_STATE_PAUSED,
  VIDEO_CAPTURE_STATE_RESUMED,
  VIDEO_CAPTURE_STATE_STOPPING,
  VIDEO_CAPTURE_STATE_STOPPED,
  VIDEO_CAPTURE_STATE_ERROR,
  VIDEO_CAPTURE_STATE_ERROR_SYSTEM_PERMISSIONS_DENIED,
  VIDEO_CAPTURE_STATE_ENDED,
  VIDEO_CAPTURE_STATE_LAST = VIDEO_CAPTURE_STATE_ENDED
};

using VideoCaptureStateUpdateCB =
    base::RepeatingCallback<void(VideoCaptureState)>;

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_PUBLIC_COMMON_MEDIA_VIDEO_CAPTURE_H_
